# coding: utf-8
from __future__ import unicode_literals
from collections import OrderedDict

ALL_FIELDS = 1000001


class Primitive(object):
    def asdict(self):
        raise NotImplementedError


def get_schema(obj):
    if isinstance(obj, Primitive):
        return obj.asdict()
    else:
        return obj


class SchemaObject(Primitive):
    default_fields = ('title', 'help_text', 'const', 'enum')
    fields = ()

    def __init__(self, *args, **kwargs):
        self.attrs = attrs = OrderedDict()

        allowed_fields = set(self.default_fields)
        allowed_fields = allowed_fields.union(self.fields)

        for field, value in kwargs.items():
            if field.startswith('x-') or field in allowed_fields:
                attrs[field] = value

    def __setitem__(self, key, value):
        self.attrs[key] = value

    def asdict(self):

        attrs = OrderedDict()

        for field, value in self.attrs.items():
            attrs[field] = get_schema(value)

        return attrs


class Any(SchemaObject):
    def asdict(self):
        return {}


class AnyOf(SchemaObject):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.types = args

    def asdict(self):
        schema = OrderedDict(
            [('anyof', [type_.asdict() for type_ in self.types])]
        )

        schema.update(super().asdict())
        return schema


class Type(SchemaObject):
    def __init__(self, typename, **kwargs):
        super().__init__(**kwargs)
        self.typename = typename

    def asdict(self):
        schema = OrderedDict([('type', self.typename)])

        schema.update(super().asdict())
        return schema


class Number(Type):
    fields = (
        'multipleOf',
        'maximum',
        'exclusiveMaximum',
        'minimum',
        'exclusiveMinimum',
    )

    def __init__(self, *args, **kwargs):
        super().__init__('number', **kwargs)


class Integer(Number):
    def __init__(self, *args, **kwargs):
        super(Number, self).__init__('integer', **kwargs)


class Decimal(Number):
    def __init__(self, *args, **kwargs):
        kwargs['format'] = 'decimal'
        super().__init__(**kwargs)


class String(Type):
    fields = ('title', 'format', 'minLength', 'maxLength', 'enum')

    def __init__(self, *args, **kwargs):
        super().__init__('string', **kwargs)


class Email(String):
    def __init__(self, *args, **kwargs):
        super().__init__(format='email')


class DateTime(String):
    def __init__(self, *args, **kwargs):
        super().__init__(format='date-time')


class Date(String):
    def __init__(self, *args, **kwargs):
        super().__init__(format='date')


class File(Type):
    def __init__(self, *args, **kwargs):
        super().__init__('file', **kwargs)


class Null(Type):
    def __init__(self, *args, **kwargs):
        super().__init__('null', **kwargs)


class Nullable(AnyOf):
    def __init__(self, item_type):
        super().__init__(Null(), item_type)


class Boolean(Type):
    def __init__(self, *args, **kwargs):
        super().__init__('boolean', **kwargs)


class Map(Primitive):
    def __init__(self, *items):
        self.items = items

    def asdict(self):
        return OrderedDict(
            [(key, get_schema(value)) for key, value in self.items]
        )


class Object(Type):
    fields = ('properties', 'required')

    def __init__(self, *args, **kwargs):

        required = kwargs.pop('required', [])

        if required == ALL_FIELDS:
            required = [field for field in OrderedDict(args).keys()]

        kwargs['required'] = required

        super().__init__('object', properties=Map(*args), **kwargs)


class Array(Type):
    fields = ('items', 'minItems')

    def __init__(self, item_type, **kwargs):
        super().__init__('array', items=item_type, **kwargs)
