# coding: utf-8

import datetime
import re
from decimal import Decimal
from typing import Union

from django.utils.encoding import force_text, smart_text
from django.utils.translation import gettext as _
from rest_framework import relations, serializers

from procu import jsonschema as js
from .utils.duration import duration_string, parse_duration


# ------------------------------------------------------------------------------


class ModelFieldMetaMixin(object):

    default_error_messages = {
        'required': _('FIELD_BASE::REQUIRED'),
        'null': _('FIELD_BASE::REQUIRED'),
    }

    def __init__(self, *args, **kwargs):
        label = kwargs.get('label', None)
        self.original_label = label

        super().__init__(*args, **kwargs)

    def get_model_field_meta(
        self: Union[serializers.Field, 'ModelFieldMetaMixin']
    ):

        attrs = {}
        model_field = None

        if self.parent and hasattr(self.parent, 'model_info'):
            model_info = self.parent.model_info
            field_name = self.field_name

            model_field = model_info.fields.get(field_name)

            if model_field is None:
                relation = model_info.relations.get(field_name)
                if relation:
                    model_field = relation.model_field

        # ----------------------------------------------------------------------
        # Fetch field titles and help texts
        title = None

        if getattr(self, 'original_label', False):
            title = force_text(self.original_label, strings_only=True)

        if not title and model_field is not None:
            help_text = force_text(getattr(model_field, 'help_text', ''))
            if help_text:
                attrs['help_text'] = help_text

            title = force_text(getattr(model_field, 'verbose_name', ''))

        if not title:
            title = force_text(self.label, strings_only=True)

        if title:
            attrs['title'] = title.strip()

        return attrs


# ------------------------------------------------------------------------------


class MetaMixin(ModelFieldMetaMixin):
    def get_schema(self, *args, **kwargs):
        raise NotImplementedError(self)

    def meta(self: Union[serializers.Field, 'MetaMixin'], *args, **kwargs):

        schema = self.get_schema(*args, **kwargs)

        schema.attrs.update(self.get_model_field_meta())

        for key, value in self.style.items():
            if key.startswith('x-'):
                schema[key] = value

        return schema


# ------------------------------------------------------------------------------


class Field(MetaMixin, serializers.Field):
    pass


# ------------------------------------------------------------------------------


class DurationField(MetaMixin, serializers.DurationField):
    default_error_messages = {'invalid': _('FIELD_DURATION::INVALID{format}')}
    format_example = '3w 4d 12h'

    def __init__(self, **kwargs):
        style = kwargs.pop('style', {})
        style['x-hint'] = self.format_example
        super().__init__(style=style, **kwargs)

    def to_internal_value(self, value):

        if isinstance(value, datetime.timedelta):
            return value

        parsed = parse_duration(str(value))

        if parsed is not None:
            return parsed

        self.fail('invalid', format=self.format_example)

    def to_representation(self, value):
        return duration_string(value)

    def get_schema(self, *args, **kwargs):
        return js.String(format='duration', maxLength=255)


# ------------------------------------------------------------------------------


class IntegerField(MetaMixin, serializers.IntegerField):
    def get_schema(self, *args, **kwargs):

        schema = js.Integer()

        if self.max_value is not None:
            schema['maximum'] = self.max_value

        if self.min_value is not None:
            schema['minimum'] = self.min_value

        if not self.allow_null:
            schema['x-required'] = True

        return schema


class DecimalField(MetaMixin, serializers.DecimalField):
    def get_schema(self, *args, **kwargs):

        schema = js.Decimal()

        schema['maximum'] = int('9' * self.max_digits)
        schema['multipleOf'] = 10 ** (-self.decimal_places)

        if self.max_value is not None:
            schema['maximum'] = min(
                schema.attrs.get('maximum', self.max_value), self.max_value
            )

        if self.min_value is not None:
            schema['minimum'] = max(
                schema.attrs.get('minimum', self.min_value), self.min_value
            )

        if not self.allow_null:
            schema['x-required'] = True

        return schema


class FloatField(MetaMixin, serializers.FloatField):
    def get_schema(self, *args, **kwargs):

        schema = js.Number()

        if self.max_value is not None:
            schema['maximum'] = self.max_value

        if self.min_value is not None:
            schema['minimum'] = self.min_value

        return schema


class MonetaryField(DecimalField):
    def __init__(self, decimal_places=2, normalize=False, **kwargs):
        max_digits = 20
        self.normalize = normalize

        kwargs['allow_null'] = kwargs.pop('allow_null', True)
        kwargs['min_value'] = 0
        kwargs['localize'] = False
        kwargs['coerce_to_string'] = False
        super().__init__(max_digits, decimal_places, **kwargs)

    def to_internal_value(self, data):
        data = smart_text(data).strip()
        data = re.sub(r'[^\d,.]+', '', data, flags=re.UNICODE)

        data = data.replace(',', '.')
        parts = data.split('.')

        if len(parts) > 1:
            data = '%s.%s' % (''.join(parts[:-1]), parts[-1])

        return super().to_internal_value(data)

    def run_validation(self, data=serializers.empty):
        # Interpret empty string as null-value
        if not smart_text(data).strip() and self.allow_null:
            return None
        return super().run_validation(data)

    def to_representation(self, value):
        quantized = super().to_representation(value)
        assert isinstance(quantized, Decimal)

        if self.normalize:
            # Drop all-zero fractional part (useful for tax values)
            integral = quantized.to_integral_value()
            if integral == quantized:
                quantized = integral

        return '{0:f}'.format(quantized)


# ------------------------------------------------------------------------------


class BooleanField(MetaMixin, serializers.BooleanField):
    def get_schema(self, *args, **kwargs):
        return js.Boolean()


# ------------------------------------------------------------------------------


class CharFieldMetaMixin(object):
    def get_schema(self: serializers.CharField, *args, **kwargs):
        write = kwargs.get('write', False)

        schema = js.String()

        if self.max_length is not None:
            schema['maxLength'] = self.max_length

        if self.min_length is not None:
            schema['minLength'] = self.min_length

        if not self.allow_blank:
            schema['minLength'] = max(self.min_length or 0, 1)

            if write:
                schema['x-required'] = True

        return schema


class CharField(CharFieldMetaMixin, MetaMixin, serializers.CharField):
    pass


class EmailField(CharFieldMetaMixin, MetaMixin, serializers.EmailField):
    pass


class RegexField(CharFieldMetaMixin, MetaMixin, serializers.RegexField):
    pass


class SlugField(CharFieldMetaMixin, MetaMixin, serializers.SlugField):
    pass


class URLField(CharFieldMetaMixin, MetaMixin, serializers.URLField):
    def get_schema(self, *args, **kwargs):
        schema = super().get_schema(*args, **kwargs)
        schema['format'] = 'url'
        return schema


class IPAddressField(CharFieldMetaMixin, MetaMixin, serializers.IPAddressField):
    def get_schema(self, *args, **kwargs):
        schema = super().get_schema(*args, **kwargs)
        schema['format'] = 'ip'
        return schema


# ------------------------------------------------------------------------------


class DateTimeField(MetaMixin, serializers.DateTimeField):
    def __init__(self, *args, **kwargs):

        # Errors
        msgs = kwargs.pop('error_messages', {})
        msgs['invalid'] = _('FIELD_DATETIME::WRONG_FORMAT')

        # Hint
        style = kwargs.pop('style', {})
        style['x-hint'] = _('FIELD_DATETIME::HINT')

        super().__init__(*args, error_messages=msgs, style=style, **kwargs)

    def get_schema(self, *args, **kwargs):
        schema = js.String(format='date-time')

        if not self.allow_null:
            schema['x-required'] = True

        return schema


class DateField(MetaMixin, serializers.DateField):
    def __init__(self, *args, **kwargs):

        # Errors
        msgs = kwargs.pop('error_messages', {})
        msgs['invalid'] = _('FIELD_DATE::WRONG_FORMAT')

        # Hint
        style = kwargs.pop('style', {})
        style['x-hint'] = _('FIELD_DATE::HINT')

        super().__init__(*args, error_messages=msgs, style=style, **kwargs)

    def get_schema(self, *args, **kwargs):
        schema = js.String(format='date')

        if not self.allow_null:
            schema['x-required'] = True

        return schema


class TimeField(MetaMixin, serializers.TimeField):
    def get_schema(self, *args, **kwargs):
        return js.String(format='time')


# ------------------------------------------------------------------------------


class ChoiceField(MetaMixin, serializers.ChoiceField):
    def get_schema(self, *args, **kwargs):
        schema = js.SchemaObject()

        schema['enum'] = list(self.choices.values())

        return schema


class FileField(MetaMixin, serializers.FileField):
    def get_schema(self, *args, **kwargs):
        return js.File()


class ImageField(MetaMixin, serializers.ImageField):
    pass


class ModelField(MetaMixin, serializers.ModelField):
    pass


class NullBooleanField(MetaMixin, serializers.NullBooleanField):
    pass


class FilePathField(MetaMixin, serializers.FilePathField):
    pass


class ReadOnlyField(MetaMixin, serializers.ReadOnlyField):
    pass


# ------------------------------------------------------------------------------


class RelatedFieldMixin(object):
    @classmethod
    def many_init(cls, *args, **kwargs):
        list_kwargs = {'child_relation': cls(*args, **kwargs)}
        for key in kwargs.keys():
            if key in relations.MANY_RELATION_KWARGS:
                list_kwargs[key] = kwargs[key]
        return ManyRelatedField(**list_kwargs)

    def __new__(cls, *args, **kwargs):
        # We override this method in order to automagically create
        # `ManyRelatedField` classes instead when `many=True` is set.
        if kwargs.pop('many', False):
            return cls.many_init(*args, **kwargs)

        return super().__new__(cls, *args, **kwargs)


class ManyRelatedField(MetaMixin, serializers.ManyRelatedField):
    def get_schema(self, *args, **kwargs):

        child_schema = self.child_relation.meta(*args, **kwargs)
        child_schema.attrs.pop('title', None)
        child_schema.attrs.pop('help_text', None)

        schema = js.Array(child_schema)

        if not self.allow_empty:
            schema['x-required'] = True

        return schema


class RelatedField(RelatedFieldMixin, MetaMixin, serializers.RelatedField):
    pass


class SlugRelatedField(
    RelatedFieldMixin, MetaMixin, serializers.SlugRelatedField
):
    pass


class HyperlinkedIdentityField(
    RelatedFieldMixin, MetaMixin, serializers.HyperlinkedIdentityField
):
    pass


class JSONField(MetaMixin, serializers.JSONField):
    def get_schema(self, *args, **kwargs):
        return js.Object()


class HyperlinkedRelatedField(
    RelatedFieldMixin, MetaMixin, serializers.HyperlinkedRelatedField
):
    pass


class PrimaryKeyRelatedField(
    RelatedFieldMixin, MetaMixin, serializers.PrimaryKeyRelatedField
):
    def get_schema(self, *args, **kwargs):
        schema = js.Integer()

        if not self.allow_null:
            schema['x-required'] = True

        return schema


class SerializerMethodField(MetaMixin, serializers.SerializerMethodField):
    def get_schema(self, *args, **kwargs):
        return js.SchemaObject()


# ------------------------------------------------------------------------------


class PKPrettyField(PrimaryKeyRelatedField):
    def __init__(self, queryset=None, serializer=None, filter=None, **kwargs):
        self.serializer = serializer
        if filter:
            queryset = queryset.filter(**filter)
        self.queryset = queryset
        super().__init__(**kwargs)

    def use_pk_only_optimization(self):
        return False

    def to_representation(self, value):
        if self.serializer is not None:
            return self.serializer(value, context=self.context).data
        else:
            super().to_representation(value)

    def get_schema(self, *args, **kwargs):

        write = kwargs.get('write', False)

        if write:
            schema = js.Integer()

            if not self.allow_null:
                schema['x-required'] = True

            return schema

        else:
            return self.serializer(context=self.context).meta(*args, **kwargs)


# ------------------------------------------------------------------------------


class ListField(MetaMixin, serializers.ListField):
    def get_schema(self, *args, **kwargs):
        schema = js.Array(self.child.meta(*args, **kwargs))

        if self.max_length:
            schema['maxLength'] = self.max_length

        if self.min_length:
            schema['minLength'] = self.min_length

        if not self.allow_empty or self.min_length:
            schema['x-required'] = True

        return schema
