# coding: utf-8

from typing import List, Optional

from django.core.exceptions import ImproperlyConfigured
from django.db.models import ManyToManyField, Q

# noinspection PyUnresolvedReferences
from rest_framework.filters import (
    BaseFilterBackend,
    SearchFilter,  # Just in case we'll want to override it
)

from procu.api.utils import strtobool


class BaseListFilter(BaseFilterBackend):
    query_param = None
    field = None

    def get_values(self, request) -> Optional[List]:
        query_list = request.GET.getlist(self.query_param)
        return self.filter_values(query_list)

    def filter_values(self, values):

        output = set()

        for value in values:
            if value == 'none':
                output.add(None)

            else:
                try:
                    value = int(value)
                    if value > 0:
                        output.add(value)

                except Exception:
                    pass

        return list(output) if output else None

    def filter_queryset(self, request, queryset, view):

        if None in (self.query_param, self.field):
            raise ImproperlyConfigured('Query parameter and field must be set')

        values = self.get_values(request)

        if values is None:
            return queryset

        if values:
            condition = Q(**{f'{self.field}__in': filter(None, values)})

            if None in values:
                condition |= Q(**{f'{self.field}__isnull': True})

        else:
            condition = Q()

        return self.apply_filter(queryset, condition)

    def apply_filter(self, queryset, condition):

        qs = queryset.filter(condition)

        field = queryset.model._meta.get_field(self.field)

        # Prevent from having duplicated instances when filtering by
        # many-to-many fields.
        if isinstance(field, ManyToManyField):
            qs = qs.distinct()

        return qs


class IncludeFilter(BaseListFilter):
    query_param = 'id'
    field = 'id'


class ExcludeFilter(BaseListFilter):
    query_param = 'exclude_id'
    field = 'id'

    def apply_filter(self, queryset, cond):
        return super().apply_filter(queryset, ~cond)


class ShowDeletedFilter(BaseFilterBackend):
    def filter_queryset(self, request, queryset, view):
        show_deleted = strtobool(request.GET.get('show_deleted'))

        if not show_deleted:
            queryset = queryset.filter(is_deleted=False)

        return queryset
