# coding: utf-8

from collections import OrderedDict

from django.core.exceptions import ImproperlyConfigured, PermissionDenied
from django.http import Http404
from rest_framework import exceptions
from rest_framework.metadata import BaseMetadata
from rest_framework.permissions import SAFE_METHODS
from rest_framework.request import clone_request

from procu.api.utils import is_readonly
from procu.rest import serializers

ALL_METHODS = ('GET', 'POST', 'PUT', 'PATCH', 'DELETE')
ALL_COMPONENTS = ('in', 'out', 'types', '')


class OpenAPIMetadata(BaseMetadata):
    def determine_metadata(self, request, view):

        spec = OrderedDict()

        if not hasattr(view, 'get_serializer'):
            return spec

        allowed_methods = set(view.allowed_methods)

        if is_readonly():
            allowed_methods &= set(SAFE_METHODS)

        # ----------------------------------------------------------------------

        method_components = {}

        for method in ALL_METHODS:

            parts = serializers.ListSerializer(
                child=serializers.ChoiceField(choices=ALL_COMPONENTS),
                allow_empty=False,
                data=request.GET.getlist(method.lower()),
            )

            if parts.is_valid():
                method_components[method] = parts.validated_data

        if not method_components:
            method_components = {
                method: ALL_COMPONENTS for method in ALL_METHODS
            }

        # ----------------------------------------------------------------------

        view.disable_reversion = True

        for method, components in method_components.items():

            if method not in allowed_methods:
                continue

            view.request = clone_request(request, method)

            try:
                view.initial(view.request)

                # Test global permissions
                if hasattr(view, 'check_permissions'):
                    view.check_permissions(view.request)

                # Test object permissions
                try:
                    if hasattr(view, 'get_object'):
                        obj = view.object
                        view.check_object_permissions(view.request, obj)
                except AssertionError:
                    pass

            except (exceptions.APIException, PermissionDenied, Http404):
                pass

            else:
                spec[method.lower()] = self.get_method_info(
                    method, view, components
                )

            finally:
                view.request = request

        return spec

    @staticmethod
    def get_method_info(method, view, components):

        operation = OrderedDict()

        if 'types' in components:
            operation['produces'] = [
                renderer.media_type for renderer in view.renderer_classes
            ]
            operation['consumes'] = [
                parser.media_type for parser in view.parser_classes
            ]

        meta_method = 'meta_%s' % method.lower()

        if callable(getattr(view, 'meta_%s' % method.lower(), None)):
            info = getattr(view, meta_method)(components)

            if not isinstance(info, dict):
                raise ImproperlyConfigured(
                    "{klass}.{method}() returned {rtype}, expected dictionary".format(
                        klass=view.__class__.__name__,
                        method=meta_method,
                        rtype=type(info).__name__,
                    )
                )
            operation.update(info)

        return operation
