# coding: utf-8

from django.conf import settings
from rest_framework import exceptions
from rest_framework.permissions import (
    BasePermission,
    IsAuthenticated,
    SAFE_METHODS,
)


class WithSupplierOnly(BasePermission):
    def has_permission(self, request, view):
        supplier = request.user.supplier
        return supplier and not supplier.is_deleted


class NonColdOnly(BasePermission):
    def has_permission(self, request, view):
        return not request.user.is_cold


class NonProductionOnly(BasePermission):
    def has_permission(self, request, view):
        return not settings.IS_PRODUCTION


class ExternalOnly(IsAuthenticated):
    def has_permission(self, request, view):
        is_auth = super().has_permission(request, view)
        return is_auth and (not request.user.is_staff)


class StaffOnly(IsAuthenticated):
    def has_permission(self, request, view):
        is_auth = super().has_permission(request, view)
        return is_auth and (request.user.is_staff)


class NonManagerReadOnly(StaffOnly):
    def has_permission(self, request, view):
        perm = super().has_permission(request, view)

        if request.method in SAFE_METHODS:
            return perm

        return perm and request.user.has_perm('api.progress_enquiry')


class ListPermission(BasePermission):
    model_class = None

    default_perms_map = {
        'GET': ['%(app_label)s.list_%(model_name)s'],
        'POST': ['%(app_label)s.create_%(model_name)s'],
        'PUT': [
            '%(app_label)s.create_%(model_name)s',
            '%(app_label)s.delete_%(model_name)s',
        ],
    }

    def __init__(self):
        self.perms_required = self.default_perms_map.copy()
        self.perms_required.update(getattr(self, 'perms_map', {}))

    def get_model_class(self, view):

        if self.model_class is not None:
            return self.model_class

        if hasattr(view, 'get_queryset'):
            queryset = view.get_queryset()
        else:
            queryset = getattr(view, 'queryset', None)

        assert queryset is not None, (
            'Cannot apply DjangoObjectPermissions on a view that '
            'does not set `.queryset` or have a `.get_queryset()` method.'
        )

        return queryset.model

    def get_required_permissions(self, method, model_cls):
        """
        Given a model and an HTTP method, return the list of permission
        codes that the user is required to have.
        """
        kwargs = {
            'app_label': model_cls._meta.app_label,
            'model_name': model_cls._meta.model_name,
        }

        try:
            return [perm % kwargs for perm in self.perms_required[method]]
        except KeyError:
            raise exceptions.MethodNotAllowed(method)

    def has_permission(self, request, view):

        is_authenticated = bool(request.user and request.user.is_authenticated)

        if not is_authenticated:
            return False

        if not request.user.is_staff:
            return False

        # Workaround to ensure DjangoModelPermissions are not applied
        # to the root view when using DefaultRouter.
        if getattr(view, '_ignore_model_permissions', False):
            return True

        model_cls = self.get_model_class(view)

        perms = self.get_required_permissions(request.method, model_cls)
        return request.user.has_perms(perms)


class EntryPermission(ListPermission):

    default_perms_map = {
        'GET': ['%(app_label)s.retrieve_%(model_name)s'],
        'POST': ['%(app_label)s.create_%(model_name)s'],
        'PATCH': ['%(app_label)s.update_%(model_name)s'],
        'DELETE': ['%(app_label)s.delete_%(model_name)s'],
    }

    def get_required_object_permissions(self, method, model_cls):
        kwargs = {
            'app_label': model_cls._meta.app_label,
            'model_name': model_cls._meta.model_name,
        }

        try:
            perms = frozenset(
                perm % kwargs for perm in self.perms_required[method]
            )
            return perms
        except KeyError:
            raise exceptions.MethodNotAllowed(method)

    def has_object_permission(self, request, view, obj):

        model_cls = self.get_model_class(view)
        user = request.user

        if not user.is_authenticated:
            return False

        if not user.is_staff:
            return False

        perms = self.get_required_object_permissions(request.method, model_cls)
        return perms.issubset(obj.permissions(user))
