# coding: utf-8

from django.db import models
from rest_framework import serializers
from rest_framework.compat import postgres_fields
from rest_framework.exceptions import ValidationError
from rest_framework.utils.model_meta import get_field_info

from procu import jsonschema as js

# noinspection PyUnresolvedReferences
from .fields import (
    BooleanField,
    CharField,
    CharField,
    ChoiceField,
    DateField,
    DateTimeField,
    DecimalField,
    DurationField,
    EmailField,
    Field,
    FileField,
    FilePathField,
    FloatField,
    HyperlinkedIdentityField,
    HyperlinkedRelatedField,
    IPAddressField,
    ImageField,
    IntegerField,
    JSONField,
    ListField,
    ManyRelatedField,
    ModelField,
    ModelFieldMetaMixin,
    MonetaryField,
    NullBooleanField,
    PKPrettyField,
    PrimaryKeyRelatedField,
    ReadOnlyField,
    RegexField,
    RelatedField,
    SerializerMethodField,
    SlugField,
    SlugRelatedField,
    TimeField,
    URLField,
)

check_for_nested = serializers.raise_errors_on_nested_writes


def conditional_check_for_nested(method_name, serializer, validated_data):
    if serializer.check_for_nested_writes:
        check_for_nested(method_name, serializer, validated_data)


serializers.raise_errors_on_nested_writes = conditional_check_for_nested

# ------------------------------------------------------------------------------


class ListSerializer(ModelFieldMetaMixin, serializers.ListSerializer):
    def __init__(self, *args, **kwargs):
        self.max_length = kwargs.pop('max_length', None)
        super().__init__(*args, **kwargs)

    def run_validation(self, data=serializers.empty):
        (is_empty_value, data) = self.validate_empty_values(data)
        if is_empty_value:
            return data

        value = self.to_internal_value(data)
        try:
            self.run_validators(value)
            value = self.validate(value)
            assert (
                value is not None
            ), '.validate() should return the validated data'

        except (ValidationError) as exc:
            # Do not transform error details into an object
            # with non_fields_error key.
            raise ValidationError(exc.detail)

        return value

    def validate(self, attrs):
        if self.max_length is not None and len(attrs) > self.max_length:
            raise ValidationError(
                f'The list has more than {self.max_length} items'
            )
        return super().validate(attrs)

    def meta(self, *args, **kwargs):
        schema = js.Array(self.child.meta(*args, **kwargs))

        if self.max_length is not None:
            schema['maxLength'] = self.max_length

        if not self.allow_empty:
            schema['x-required'] = True
            schema['minLength'] = 1

        # Serializer can be a field too!
        schema.attrs.update(self.get_model_field_meta())

        for key, value in self.style.items():
            if key.startswith('x-'):
                schema[key] = value

        return schema


# ------------------------------------------------------------------------------


fields_cache = {}


class SerializerMixin(ModelFieldMetaMixin):

    # @property
    # def _readable_fields(self):
    #     global fields_cache
    #
    #     class_name = '%s.%s' % (self.__module__, self.__class__.__name__)
    #
    #     try:
    #         return fields_cache[class_name]
    #
    #     except KeyError:
    #         fields_cache[class_name] = readable_fields = [
    #             field for field in self.fields.values()
    #             if not field.write_only
    #         ]
    #
    #         return readable_fields

    @classmethod
    def many_init(cls, *args, **kwargs):

        allow_empty = kwargs.pop('allow_empty', None)
        child_serializer = cls(*args, **kwargs)
        list_kwargs = {'child': child_serializer}
        if allow_empty is not None:
            list_kwargs['allow_empty'] = allow_empty
        list_kwargs.update(
            {
                key: value
                for key, value in kwargs.items()
                if key in serializers.LIST_SERIALIZER_KWARGS
            }
        )
        meta = getattr(cls, 'Meta', None)
        list_serializer_class = getattr(
            meta, 'list_serializer_class', ListSerializer
        )
        return list_serializer_class(*args, **list_kwargs)

    def meta(self, *args, **kwargs):
        write = kwargs.get('write', False)

        fields = []
        required_fields = []

        for field_name, field in self.fields.items():

            if not hasattr(field, 'meta'):
                class_name = str(field.__class__)
                raise ValidationError(
                    {'detail': '{} does not support OPTIONS'.format(class_name)}
                )

            if write and getattr(field, 'read_only', False):
                continue

            fields.append((field_name, field.meta(*args, **kwargs)))

            if not write or field.required:
                required_fields.append(field_name)

        schema = js.Object(*fields, required=required_fields)

        # Serializer can be a field too!
        schema.attrs.update(self.get_model_field_meta())

        return schema


class Serializer(SerializerMixin, serializers.Serializer):
    pass

    # Add dummy methods to make inspections happy.
    # Direct use of Serializer in this project means does not imply
    # creating and saving instances anyway.

    def create(self, validated_data):
        pass

    def update(self, instance, validated_data):
        pass


class ModelSerializer(SerializerMixin, serializers.ModelSerializer):
    """
    Our own version of ModelSerializer in case we need some global customisation
    """

    # Controls check for nested writes. It allows us to use nested serializers
    # to validate json-fields.
    check_for_nested_writes = True

    model_info = None

    serializer_field_mapping = {
        models.AutoField: IntegerField,
        models.BigIntegerField: IntegerField,
        models.BooleanField: BooleanField,
        models.CharField: CharField,
        models.CommaSeparatedIntegerField: CharField,
        models.DateField: DateField,
        models.DateTimeField: DateTimeField,
        models.DecimalField: DecimalField,
        models.EmailField: EmailField,
        models.Field: ModelField,
        models.FileField: FileField,
        models.FloatField: FloatField,
        models.ImageField: ImageField,
        models.IntegerField: IntegerField,
        models.NullBooleanField: NullBooleanField,
        models.PositiveIntegerField: IntegerField,
        models.PositiveSmallIntegerField: IntegerField,
        models.SlugField: SlugField,
        models.SmallIntegerField: IntegerField,
        models.TextField: CharField,
        models.TimeField: TimeField,
        models.URLField: URLField,
        models.GenericIPAddressField: IPAddressField,
        models.FilePathField: FilePathField,
        postgres_fields.JSONField: JSONField,
    }

    serializer_related_field = PrimaryKeyRelatedField
    serializer_choice_field = ChoiceField

    serializer_related_to_field = SlugRelatedField
    serializer_url_field = HyperlinkedIdentityField

    def meta(self, *args, **kwargs):
        self.model_info = get_field_info(self.Meta.model)
        return super().meta(*args, **kwargs)


if postgres_fields:
    ModelSerializer.serializer_field_mapping[
        postgres_fields.ArrayField
    ] = ListField
