# coding: utf-8

from django.core.exceptions import ImproperlyConfigured
from django.db import models
from django.db.models.query_utils import DeferredAttribute

from .api import wiki_format


def fmt_field(name):
    return name + '_html'


class WFMixin(object):
    wf_config = None

    def __init__(self, *args, **kwargs):
        self.wf_config = kwargs.pop('wf_config', None)
        assert self.wf_config

        super().__init__(*args, **kwargs)

    def deconstruct(self: models.Field):
        field_name, path, args, kwargs = super().deconstruct()
        kwargs['wf_config'] = self.wf_config
        return field_name, path, args, kwargs

    def pre_save(self: models.Field, model_instance, add):
        raw_text = getattr(model_instance, self.attname)
        setattr(
            model_instance,
            fmt_field(self.attname),
            wiki_format(raw_text, config=self.wf_config),
        )
        return raw_text


class WFTextField(WFMixin, models.TextField):
    pass


class WFCharField(WFMixin, models.CharField):
    pass


FIELDS_CLASSES = {'CharField': WFCharField, 'TextField': WFTextField}


class FormattedField(models.Field):
    description = 'WFText'

    def __init__(self, *args, **kwargs):
        self.wf_config = kwargs.pop('wf_config', None)

        if self.wf_config is None:
            raise ImproperlyConfigured(
                "FormattedField requires `wf_config' parameter"
            )

        field_class = kwargs.pop('field_class', None)
        self.wf_field_class = FIELDS_CLASSES[field_class().get_internal_type()]

        super().__init__(*args, **kwargs)

    def get_internal_type(self):
        return 'FormattedField'

    def contribute_to_class(self, cls, name, **kwargs):

        fmt_name = fmt_field(name)
        field_name, path, args, kwargs = self.deconstruct()

        field_raw = self.wf_field_class(
            wf_config=self.wf_config, *args, **kwargs
        )

        field_raw.name = name

        field_html = models.TextField(
            editable=False, blank=True, null=True, default=None
        )
        field_html.name = fmt_name

        field_raw.model = field_html.model = cls

        field_raw.set_attributes_from_name(name)
        field_html.set_attributes_from_name(fmt_name)

        cls._meta.add_field(field_raw)
        cls._meta.add_field(field_html)

        setattr(cls, field_raw.attname, DeferredAttribute(field_raw.attname))
        setattr(cls, field_html.attname, DeferredAttribute(field_html.attname))
