import re

from django import http
from django.conf import settings
from django.db import DEFAULT_DB_ALIAS
from django.http import HttpResponse
from django.utils.deprecation import MiddlewareMixin
from django_replicated.dbchecker import db_is_alive
from django_replicated.utils import routers


class ReadOnlyMiddleware(MiddlewareMixin):
    """
    Миддлварина для перевода сервиса в режим read-only
    """
    def process_request(self, request):
        if request.path.startswith('/admin') or request.path.startswith('/ping'):
            request.is_service_readonly = False
            return

        else:
            request.is_service_readonly = not db_is_alive(
                db_name=DEFAULT_DB_ALIAS,
                cache_seconds=getattr(settings, 'READ_ONLY_DOWNTIME', 20),
                number_of_tries=getattr(settings, 'READ_ONLY_TRIES', 1),
            )

        if request.is_service_readonly:
            routers.use_state('slave')

        if request.method == 'POST' and request.is_service_readonly:
            message = 'SERVICE IS READ-ONLY'
            return http.HttpResponse(
                status=500,
                reason=message,
                content=message,
            )

    def process_response(self, request, response):
        """
        Если роутер был перенаправлен на слейв из-за флага read-only,
        возвращаем все на место
        """
        if getattr(request, 'is_service_readonly', False):
            if routers.context.state_stack:
                routers.revert()

        return response


class AliveMiddleware(MiddlewareMixin):
    """
    Простенькая миддлварина, через которую можно проверить,
    что приложение запущено
    """
    ping_re = re.compile(r'^/ping/?$')

    def process_request(self, request):
        if self.ping_re.match(request.path):
            return HttpResponse("I'm alive!")
