# coding: utf-8

import logging
import re


logger = logging.getLogger(__name__)

DEFAULT_EMAIL_SUBJECT_FORMAT = '{environmentId} {status}'
DEFAULT_EMAIL_MESSAGE_FORMAT = 'Окружение {env_link} перешло в статус {status}.\n\n{comment}'

TICKET_STAND_SECTION_NAME = 'Стенд'
TICKET_STAND_SECTION_BODY = '{stand_link}'


class ConfigItem(object):
    def __init__(self, on_statuses=None):
        self.on_statuses = on_statuses

    def should_trigger(self, status):
        return self.on_statuses is None or status in self.on_statuses


class NotifyEmailConfigItem(ConfigItem):
    def __init__(self, data):
        super(NotifyEmailConfigItem, self).__init__(data.get('on_statuses'))
        for field, default in (
                ('recipients', list()),
                ('subject_format', DEFAULT_EMAIL_SUBJECT_FORMAT),
                ('message_format', DEFAULT_EMAIL_MESSAGE_FORMAT),
        ):
            setattr(self, field, data.get(field, default))


class TrackerConfigItem(ConfigItem):
    def __init__(self, data):
        super(TrackerConfigItem, self).__init__(data.get('on_statuses'))
        for field, default in (
                ('transition', None),
                ('set_version', {
                    'enabled': False,
                    'version_groups': None,
                    'field_name': None,
                }),
                ('update_description', {
                    'enabled': False,
                    'section_name': None,
                    'section_body': None,
                    'stand_link_constructor': lambda env_link: env_link
                }),
                ('comment', {
                    'enabled': False,
                    'text': '',
                    'dont_repeat': False,
                }),
        ):
            setattr(self, field, data.get(field, default))


class Config(object):
    def __init__(self, data):
        self.notify = {}
        for notify_type, notify_data in data.get('notify', {}).items():
            if notify_type == 'email':
                self.notify['email'] = NotifyEmailConfigItem(notify_data)
        self.tracker = None
        if 'tracker' in data:
            self.tracker = TrackerConfigItem(data['tracker'])


wiki_testing_config = Config({
    'notify': {
        'email': {
            'recipients': ['wiki-deployment@yandex-team.ru', ],
            'on_statuses': ['DEPLOYED', 'FAILED', ],
        }
    },
    'tracker': {
        'transition': 'inTesting',
        'set_version': {
            'enabled': True,
            'version_groups': 2,
            'field_name': 'fixVersions',
        },
        'on_statuses': ['DEPLOYED', ],
    },
})

wiki_production_config = Config({
    'notify': {
        'email': {
            'recipients': ['wiki-deployment@yandex-team.ru', ],
            'on_statuses': ['DEPLOYED', 'FAILED', ],
        }
    },
    'tracker': {
        'transition': 'deployed',
        'set_version': {
            'enabled': True,
            'version_groups': 2,
            'field_name': 'fixVersions',
        },
        'on_statuses': ['DEPLOYED', ],
    }
})


def _get_wiki_stand_link_by_env_link(env_link):
    link_parts = env_link.split('/')
    stand_slug = link_parts[-1]
    application = link_parts[-2]
    wiki_instance = application.split('-')[-1]
    subdomain = 'yandex' if wiki_instance in ('business', 'school') else 'yandex-team'
    return 'https://wiki-{slug}-front-{instance}.test.tools.{subdomain}.ru/'.format(
        slug=stand_slug,
        instance=wiki_instance,
        subdomain=subdomain
    )


wiki_stand_config = Config({
    'tracker': {
        'transition': 'readyForVerifying',
        'update_description': {
            'enabled': True,
            'section_name': TICKET_STAND_SECTION_NAME,
            'section_body': TICKET_STAND_SECTION_BODY,
            'stand_link_constructor': _get_wiki_stand_link_by_env_link
        },
        'on_statuses': ['DEPLOYED', ],
    }
})

forms_testing_config = Config({
    'tracker': {
        'transition': 'inTesting',
        'on_statuses': ['DEPLOYED', ],
        'set_version': {
            'enabled': True,
            'field_name': 'tags',
        },
    },
})

forms_production_config = Config({
    'tracker': {
        'transition': 'deployed',
        'on_statuses': ['DEPLOYED', ],
        'set_version': {
            'enabled': True,
            'field_name': 'tags',
        },
    },

})

cab_back_testing_config = Config({
    'tracker': {
        'transition': 'testing',
        'on_statuses': ['DEPLOYED', ],
    },
})

review_testing_config = Config({
    'tracker': {
        'transition': 'testing',
        'on_statuses': ['DEPLOYED', ],
    },
})

review_stand_config = Config({
    'tracker': {
        'transition': 'testingDev',
        'on_statuses': ['DEPLOYED', ],
    },
})

easymeeting_testing_config = Config({
    'tracker': {
        'transition': 'testing',
        'on_statuses': ['DEPLOYED', ],
    },
})

avia_testing_config = Config({
    'notify': {
        'email': {
            'recipients': ['avia-dev@yandex-team.ru'],
            'on_statuses': ['DEPLOYED', 'FAILED'],
        }
    },
    'tracker': {
        'set_version': {
            'enabled': True,
            'field_name': 'fixVersions',
        },
        'on_statuses': ['DEPLOYED'],
    },
})

abcd_testing_config = Config({
    'tracker': {
        'transition': 'inTesting',
        'comment': {
            'enabled': True,
            'text': 'Выкатилось в testing. Опишите, как тестировать этот тикет, пожалуйста',
            'dont_repeat': True,
        },
        'on_statuses': ['DEPLOYED', ],
    },
})

abc_production_config = Config({
    'tracker': {
        'transition': 'close',
        'comment': {
            'enabled': True,
            'text': 'Выкатилось в прод. Не забудь проверить жизнеспособность прода '
                    'и провести дополнительные работы, если это требуется в тикете.',
            'dont_repeat': True,
        },
        'on_statuses': ['DEPLOYED', ],
    },
})

qooker_production_config = Config({
    'tracker': {
        'transition': 'close',
        'comment': {
            'enabled': True,
            'text': 'Выкатилось в прод.',
            'dont_repeat': True,
        },
        'on_statuses': ['DEPLOYED', ],
    },
})

access_tools_config = Config({
    'notify': {
        'email': {
            'recipients': ['tools-access-deployment@yandex-team.ru'],
            'on_statuses': ['DEPLOYED', 'FAILED'],
        },
    },
    'tracker': {
        'transition': 'inTesting',
        'set_version': {
            'enabled': True,
            'field_name': 'fixVersions',
        },
        'comment': {
            'enabled': True,
            'text': 'Выкатилось в testing. Опишите, как тестировать этот тикет, пожалуйста',
            'dont_repeat': True,
        },
        'on_statuses': ['DEPLOYED'],
    }
})

vins_develop_config = Config({
    'tracker': {
        'on_statuses': ['DEPLOYED'],
        'comment': {
            'enabled': True,
            'text': 'Выкатилось в develop VINSа',
            'dont_repeat': True
        }
    }
})


def _get_directory_stand_link_by_env_link(env_link):
    link_parts = env_link.split('/')
    stand_slug = link_parts[-1]

    return """
[[https://{stand_slug}.directory.test.ws.yandex.net/ https://badger.yandex-team.ru/qloud/workspace/directory/{stand_slug}/backend/version.svg]]
[[https://{stand_slug}.portal.test.ws.yandex.ru/ https://badger.yandex-team.ru/qloud/workspace/portal/{stand_slug}/back/version.svg]]
    """.format(stand_slug=stand_slug)


directory_testing_config = Config({
    'tracker': {
        'transition': 'readyForTest',
        'on_statuses': ['DEPLOYED', ],
        'set_version': {
            'enabled': True,
            'field_name': 'tags',
        },
        'update_description': {
            'enabled': True,
            'section_name': "Стенд",
            'section_body': "{stand_link}",
            'stand_link_constructor': _get_directory_stand_link_by_env_link,
        },
    },
})

directory_production_config = Config({
    'tracker': {
        'transition': 'closed',
        'on_statuses': ['DEPLOYED', ],
        'set_version': {
            'enabled': True,
            'field_name': 'tags',
        },
    },
})


CONFIG = {
    r'tools\.wiki-(back|front)-(intranet|business|school|evaluation)\.testing': wiki_testing_config,
    r'tools\.wiki-(back|front)-(intranet|business|school|evaluation)\.production': wiki_production_config,
    r'tools\.wiki-stand-(back|front)-(intranet|business|school|evaluation)\.(.*)': wiki_stand_config,

    r'tools\.wf-ws(-business)?\.testing': wiki_testing_config,
    r'tools\.wf-ws(-business)?\.production': wiki_production_config,

    r'tools\.magiclinks-business\.testing': wiki_testing_config,
    r'tools\.magiclinks-business\.production': wiki_production_config,

    r'tools\.abc\.testing': abcd_testing_config,
    r'tools\.abc\.production': abc_production_config,
    r'tools\.abc-www\.testing': abcd_testing_config,
    r'tools\.dispenser-www\.testing': abcd_testing_config,
    r'tools\.dispenser\.testing': abcd_testing_config,

    r'tools\.qooker\.production': qooker_production_config,

    r'tools\.cab-back\.testing': cab_back_testing_config,

    r'tools\.review\.testing': review_testing_config,
    r'tools\.review-stands\.(.*)': review_stand_config,

    r'tools\.easymeeting\.testing': easymeeting_testing_config,

    r'(tools|tools\-b2b)\.forms-admin-(biz|ext|int)-www\.testing': forms_testing_config,
    r'(tools|tools\-b2b)\.forms-admin-(biz|ext|int)-www\.production': forms_production_config,

    r'tools\.forms-back-ext\.testing': forms_testing_config,
    r'tools\.forms-back-ext\.production': forms_production_config,

    r'tools\.forms-(internal|external|biz|anonymous|preview|biz\-preview)-www\.testing': forms_testing_config,
    r'tools\.forms-(internal|external|biz|anonymous|preview|biz\-preview)-www\.production': forms_production_config,

    r'avia\.avia\-backend\.testing': avia_testing_config,

    r'tools-access\.(idm|signer|crt|cauth)\.testing': access_tools_config,
    r'tools\.(idm|crt)-www\.testing': access_tools_config,

    r'tools\.delta\.testing': wiki_testing_config,
    r'tools\.delta\.production': wiki_production_config,

    r'voice-ext\.vins-int\.develop': vins_develop_config,

    r'workspace\.directory\.internal-prod': directory_production_config,
    r'workspace\.directory\.dir-.*': directory_testing_config,
    r'workspace\.portal\.prod': directory_production_config,

}

PREPARED_CONFIG = {re.compile(env_regex): config for env_regex, config in CONFIG.items()}


def get_config(env):
    # В будущем за конфигом будем ходить в БД.

    matched_env_pattern = next(
        (env_pattern for env_pattern in PREPARED_CONFIG if re.match(env_pattern, env) is not None),
        None
    )

    if matched_env_pattern:
        logger.info('"%s" matched "%s"', env, matched_env_pattern.pattern)
        return PREPARED_CONFIG[matched_env_pattern]

    return None
