# coding: utf-8

import logging
import re

from django.conf import settings
from startrek_client import Startrek, exceptions

from .utils import get_env_link_by_hook_data

logger = logging.getLogger(__name__)

TRACKER_TICKET_PATTERN = re.compile(r'[A-Z]{2,}-\d+')
DEPLOY_COMMENT_HEADER_PATTERN = re.compile('Releasing version ([0-9.-]+), changelog:')

TRACKER_API_URL = 'https://st-api.yandex-team.ru'

tracker = Startrek(
    useragent='qooker',
    base_url=TRACKER_API_URL,
    token=settings.TRACKER_OAUTH_TOKEN,
)


def transit_tickets(transition_name, hook_data):
    ticket_keys = re.findall(TRACKER_TICKET_PATTERN, hook_data['comment'])
    for ticket_key in ticket_keys:
        try:
            ticket = tracker.issues[ticket_key]
            try:
                transition = ticket.transitions[transition_name]
                if transition_name == 'close':
                    transition.execute(comment='Closed by Qooker', resolution='fixed')
                else:
                    transition.execute()
                logger.info('Ticket %s moved to "%s"', ticket_key, transition_name)
            except exceptions.NotFound:
                logger.warn('No transition "%s" in ticket %s with status "%s", skipping ticket',
                            transition_name, ticket_key, ticket.status.name)
        except exceptions.StartrekError:
            logger.exception('Error for ticket %s', ticket_key)


def comment_tickets(hook_data, comment_text, dont_repeat=False):
    ticket_keys = re.findall(TRACKER_TICKET_PATTERN, hook_data['comment'])
    for ticket_key in ticket_keys:
        try:
            ticket = tracker.issues[ticket_key]
            if dont_repeat:
                comments = ticket.comments.get_all()
                conditions = (
                    c.text == comment_text and c.createdBy.login == settings.ROBOT_LOGIN
                    for c in comments
                )
                # нет ни одного коммента с таким текстом от нашего робота
                if not any(conditions):
                    ticket.comments.create(text=comment_text)
            else:
                ticket.comments.create(text=comment_text)
        except exceptions.StartrekError:
            logger.exception('Error for ticket %s', ticket_key)


def set_tickets_version(hook_data, version_groups, field_name):
    header = re.search(DEPLOY_COMMENT_HEADER_PATTERN, hook_data['comment'])
    if header is None:
        return
    version_name = header.group(1)

    if version_groups is not None:
        version_name = _crop_version_to_groups(version_name, version_groups)

    ticket_keys = re.findall(TRACKER_TICKET_PATTERN, hook_data['comment'])

    def _set_tag(tag, ticket_object):
        if tag in ticket_object.tags:
            # Этот тег уже проставлен тикету
            return

        tags = ticket_object.tags
        tags.append(tag)

        ticket_object.update(tags=tags)

    def _set_fixVersions(version, ticket_object):
        if any(str(v.id) == str(version.id) for v in ticket_object.fixVersions):
            # Эта версия уже проставлена тикету
            return

        new_versions = ticket_object.fixVersions
        new_versions.append(version)

        ticket_object.update(fixVersions=new_versions)

    def _find_version_by_name():
        # У нас есть версия и ключи тикетов из комментария деплоя.
        # Теперь надо из ключей тикетов вынуть ключи очередей
        # и найти очередь, в которой есть такая версия.
        queue_keys = {t.split('-')[0] for t in ticket_keys}
        for queue_key in queue_keys:
            try:
                queue = tracker.queues[queue_key]
            except exceptions.StartrekError:
                continue
            for queue_version in queue.versions:
                if queue_version.name == version_name:
                    return queue_version
        return None

    FIELDS_PROCESSORS_MAP = {
        # Набор поддерживаемых полей стартрека, при добавлении
        # нового поля обязательно следует реализовать функцию
        # для ключа set_function - которая будет проставлять
        # версию в нужное поле стартрека
        #
        # process_data_function - указывается если требуется
        # дополнительная обработка версии, например, для fixVersions
        # проверяется есть ли среди очередей тикетов очередь,
        # в которой есть версия с таким же именем. И только если она есть,
        # то мы добавляем эту версию в список fixVersions тикетов.
        'tags': {
            'set_function': _set_tag
        },
        'fixVersions': {
            'set_function': _set_fixVersions,
            'process_data_function': _find_version_by_name,
        },
    }

    field_processor = FIELDS_PROCESSORS_MAP.get(field_name)
    if not field_processor or not field_processor.get('set_function'):
        raise ValueError('You should implement working with this field first - {}'.format(field_name))

    if field_processor.get('process_data_function'):
        processed_data = field_processor['process_data_function']()
        if not processed_data:
            return
    else:
        processed_data = version_name

    for ticket_key in ticket_keys:
        try:
            ticket = tracker.issues[ticket_key]
            field_processor['set_function'](processed_data, ticket)
        except exceptions.StartrekError:
            logger.exception('Error for ticket %s', ticket_key)


def update_tickets_description(section_name, section_body, stand_link_constructor, hook_data):
    ticket_keys = re.findall(TRACKER_TICKET_PATTERN, hook_data['comment'])
    for ticket_key in ticket_keys:
        try:
            ticket = tracker.issues[ticket_key]
            env_link = get_env_link_by_hook_data(hook_data)
            stand_link = stand_link_constructor(env_link)
            _update_ticket_description(ticket, section_name, section_body, stand_link)
        except exceptions.StartrekError:
            logger.exception('Error for ticket %s', ticket_key)


def _update_ticket_description(ticket, section_name, section_body, stand_link):
    section_name_wt = '=={}'.format(section_name)
    section_body_wt = section_body.format(stand_link=stand_link)
    section_wt = '{}\n{}'.format(section_name_wt, section_body_wt)

    description = ticket.description or ''
    if section_name_wt in description:
        if description.count(section_name_wt) > 1:
            logger.error(
                'Ticket %s has multiple "%s" sections, failed to update description',
                ticket.key, section_name
            )
            return

        other_description, section_part = description.split(section_name_wt)
        if '==' in section_part:
            logger.error(
                'Section "%s" part in ticket %s has other sections, but it must be last,'
                'failed to update description',
                section_name, ticket.key
            )
            return

        new_description = '{}\n{}'.format(other_description, section_wt)
    else:
        new_description = '{}\n{}'.format(description, section_wt)

    if new_description != description:
        ticket.update(description=new_description, ignore_version_change=True)


def _crop_version_to_groups(version, groups):
    # Оставляем groups чисел версии слева.
    parts = []
    current_part = ''
    in_group = True
    for c in version:
        if in_group != ('0' <= c <= '9'):
            parts.append(current_part)
            current_part = ''
            in_group = not in_group
        current_part += c
    if current_part != '':
        parts.append(current_part)
    parts = parts[:2 * groups - 1]
    return ''.join(parts)
