# coding: utf-8


import json
import logging
from review.frontend.views import person_reviews
import waffle
from collections import defaultdict
from itertools import chain
from typing import Dict

import attr
from django.utils.decorators import method_decorator
from django.views.decorators import csrf
from django.db.models import Q

from review.bi import (
    models as bi_models,
    logic,
)
from review.bi.serializers import AssignmentDataSerializer
from review.core.logic import roles
from review.core import const
from review.lib import encryption, views, errors
from review.staff import models as staff_models
from review.staff.const import STAFF_ROLE


logger = logging.getLogger(__name__)


def get_lte_18_grade_persons_ids(observables_logins):
    grades = logic.get_actual_grades(observables_logins)
    return [
        person_id
        for person_id, grade in grades.items()
        if logic.grade_lte_18(grade)
    ]


def get_person_roles_qs(roles, user):
    return (
        staff_models.HR.objects
        .filter(type__in=roles, hr_person=user)
        .values('cared_person_id')
    )


def get_allowed_qs(user, observables_logins):
    res_query = staff_models.Person.objects.filter(login__in=observables_logins)
    if roles.has_global_roles(user, [const.ROLE.GLOBAL.BI_VIEWER]):
        return res_query

    hr_cared_persons_qs = get_person_roles_qs([STAFF_ROLE.HR.FINANCE_VIEWER], user)
    hr_analyst_persons_qs = get_person_roles_qs([STAFF_ROLE.HR.HR_ANALYST], user)

    subordinates_persons_qs = (
        staff_models.Subordination.objects
        .filter(subject=user)
        .values('object_id')
    )

    lte_18_grade_persons_ids = get_lte_18_grade_persons_ids(observables_logins)

    hr_q = Q(id__in=hr_analyst_persons_qs) & Q(id__in=lte_18_grade_persons_ids) | Q(id__in=hr_cared_persons_qs)
    subordinates_q = Q(id__in=subordinates_persons_qs)
    user_q = Q(id=user.id)

    return res_query.filter(hr_q | subordinates_q | user_q)


def get_observable_person(user, observable_login):
    allowed_qs = get_allowed_qs(user, [observable_login])

    try:
        return allowed_qs[0]
    except IndexError:
        if staff_models.Person.objects.filter(login=observable_login).exists():
            raise errors.PermissionDenied(
                'User has no rights to see {login} income',
                login=observable_login,
            )
        else:
            raise errors.NotFound('User {login} not found', login=observable_login)


class PersonAssignmentView(views.View):

    def process_get(self, auth, data):
        observable_login = data.get('login') or auth.user.login
        observable = get_observable_person(auth.user, observable_login)

        serializer_context = {}
        if waffle.switch_is_active('enable_review_announcement_report_date'):
            serializer_context['review_announced'] = logic.is_person_review_announced(observable.login)

        assignments = logic.get_assignments([observable.id], serializer_context).get(observable.id, [])
        main_assignment = logic.get_main_assignment(assignments)
        rates = {}
        if main_assignment is not None:
            rates = logic.get_rates_for_currency(
                org_id=main_assignment.get('organization_id'),
                currency=main_assignment.get('currency'),
            )

        return {
            'assignments': assignments,
            'rates': {
                currency: float(rate)
                for currency, rate in rates.items()
            },
        }


@method_decorator(csrf.csrf_exempt, name='dispatch')
class PersonAssignmentListView(views.View):

    @staticmethod
    def assignment_list_data(allowed_qs):
        assignments_qs = (
            bi_models.BIPersonAssignment.objects
            .filter(person_id__in=allowed_qs.values('id'))
            .select_related('person')
        )
        result = defaultdict(list)
        for assignment in assignments_qs:
            login = assignment.person.login
            result[login].append(AssignmentDataSerializer.serialize(dict(id=assignment.id, **assignment.data)))
        return result

    def process_post(self, auth, data):
        logins = data.get('logins')
        if not logins:
            return {}
        login_list = logins.split(',')
        allowed_qs = get_allowed_qs(auth.user, login_list)
        return self.assignment_list_data(allowed_qs) or {}


def _to_response_format(_, field, value):
    """
    1. Marks has random number of parameters, stored in "meta".
       Need to move those parameters directly to mark object.
       "Mark" parameter from meta conflicts with actual mark value
       and is useless, so it needs to be deleted.
    2. Replaces field "from_" to "from" (couldn't use "from" due to being keyword)
    """
    if not attr.has(type(value)):
        return value
    serialized = attr.asdict(value)
    if field.name in ['mark_last', 'mark_current']:
        meta = serialized.pop('meta')
        meta.pop('mark', None)
        serialized.update(meta)
    if field.name == 'income':
        for sub_field in ['per_calendar_year', 'from_current_date', 'detailed']:
            for it in serialized[sub_field]:
                it['from'] = it.pop('from_')
    return serialized


def serialize_income(inc):
    # type: (logic.PersonIncome) -> Dict
    return attr.asdict(
        inc,
        recurse=False,
        value_serializer=_to_response_format,
    )


class PersonIncomeView(views.View):

    def process_get(self, auth, data):
        observable_login = data.get('login') or auth.user.login
        observable = get_observable_person(auth.user, observable_login)

        is_finance_viewer_for_observable = False
        if waffle.switch_is_active('bi_income_show_long_forecast'):
            is_finance_viewer_for_observable = (
                get_person_roles_qs([STAFF_ROLE.HR.FINANCE_VIEWER], auth.user)
                .filter(cared_person=observable)
                .exists()
            )
        res = logic.get_person_income(
            login=observable.login,
            long_forecast=is_finance_viewer_for_observable,
        )
        if res:
            return serialize_income(res)
        return {}


@method_decorator(csrf.csrf_exempt, name='dispatch')
class PersonIncomeListView(views.View):

    def process_post(self, auth, data):
        logins = data.get('logins')

        if not logins:
            return {}
        login_list = logins.split(',')

        allowed_qs = (
            get_allowed_qs(auth.user, login_list)
            .values_list('id')
        )
        person_incomes = logic.get_person_incomes(allowed_qs)

        return {
            login: serialize_income(inc)
            for login, inc in person_incomes.items()
        }
