# coding: utf-8

from typing import Dict, List

import attr
from django import forms

from review.lib import views
from review.finance import (
    logic,
    person,
)
from review.oebs.const import OEBS_DATA_TYPES
from review.shortcuts import const
from review.shortcuts import models


class FinanceForm(forms.Form):
    persons = forms.ModelMultipleChoiceField(
        queryset=models.Person.objects.all(),
        to_field_name='login',
    )
    fields = forms.MultipleChoiceField(
        choices=[(x, x) for x in person.Person.data_fields],
        required=False,
    )

    def clean_fields(self):
        fields = self.cleaned_data['fields']
        if not fields:
            return list(const.OEBS_DATA_TYPES)
        return fields


class FinanceView(views.View):
    form_cls_get = FinanceForm
    cached_result = True

    def process_get(self, auth, data):
        filter_by_persons = data.get('persons') or []
        persons = logic.get_persons(
            subject=auth.user,
            filters={'persons': filter_by_persons},
            cached_result=self.cached_result,
        )
        return {
            'filters': {
                'persons': [p.login for p in filter_by_persons],
                'fields': data['fields'],
            },
            'result': serialize_persons(persons, data['fields'])
        }


class FinanceRawView(FinanceView):
    """ Gets data from oebs-api """
    cached_result = False


def _empty_to_none(inst, field, value):
    if value != const.EMPTY:
        return value


def _get_filter_oebs_fields(fields):
    not_requested = OEBS_DATA_TYPES - set(fields)

    def filter_(attrib, _):
        return attrib.name not in not_requested

    return filter_


def serialize_persons(persons, fields):
    # type: (List[person.Person], List[str]) -> Dict[str, Dict]
    filter_fields = _get_filter_oebs_fields(fields)
    return {
        person.login: attr.asdict(
            person,
            filter=filter_fields,
            value_serializer=_empty_to_none,
        )
        for person in persons
    }
