# coding: utf-8
from django import forms

from review.core import const
from review.core.logic import assemble, review_actions
from review.shortcuts import models
from review.lib import views
from review.frontend.views.person_reviews import PersonReviewsModeReviewView


class PersonReviewsApiView(PersonReviewsModeReviewView):

    ROLES = frozenset([
        const.ROLE.PERSON.SELF,
        const.ROLE.DEPARTMENT.HR_PARTNER,
        const.ROLE.DEPARTMENT.HR_ANALYST,
        const.ROLE.DEPARTMENT.HEAD,
        const.ROLE.GLOBAL.EXPORTER,
    ])

    DEFAULT_API_FIELDS = (
        const.FIELDS.ID,
        const.FIELDS.MARK,
        const.FIELDS.REVIEW_ID,
        const.FIELDS.REVIEW_START_DATE,
        const.FIELDS.REVIEW_FINISH_DATE,
        const.FIELDS.PERSON_LOGIN,
        const.FIELDS.PERSON_FIRST_NAME,
        const.FIELDS.PERSON_LAST_NAME,
        const.FIELDS.REVIEW_TYPE,
        const.FIELDS.GOLDSTAR,
        const.FIELDS.STATUS,
        const.FIELDS.TAG_AVERAGE_MARK,
        const.FIELDS.TAKEN_IN_AVERAGE,
        const.FIELDS.UMBRELLA,
        const.FIELDS.MAIN_PRODUCT,
    )

    def get_fields(self, data):
        return data.pop('fields', None) or self.DEFAULT_API_FIELDS


class RemovePersonReviewForm(forms.Form):
    person_reviews = forms.ModelMultipleChoiceField(
        queryset=models.PersonReview.objects.all(),
        required=True,
    )


class RemovePersonReviews(views.View):

    form_cls_post = RemovePersonReviewForm

    def process_post(self, auth, data):
        review_id = data['id']
        person_reviews_qs = data['person_reviews']

        review = assemble.get_review(
            subject=auth.user,
            filters={'ids': [data['id']]}
        )

        assemble.permissions_review.ensure_has_action(
            review=review,
            actions=[const.REVIEW_ACTIONS.DELETE_ANY_PERSONS, const.REVIEW_ACTIONS.DELETE_PERSONS],
        )

        extra_ids = list(
            person_reviews_qs
            .exclude(review_id=review_id)
            .values_list('id', flat=True)
        )
        review_actions.delete_review_subject(auth.user, review, person_reviews_qs)
        return {
            'extra_ids': extra_ids
        }
