# coding: utf-8

import json
import logging
import math
import requests
import yt.wrapper as yt

from yt.wrapper.errors import YtTokenError
from django.conf import settings
from requests import Response
from typing import Dict, Iterator

from review.lib.tvm2_client import get_service_ticket


logger = logging.getLogger(__name__)


class BIError(Exception):
    """Error during getting data from BI MSTR."""


class NeedAuth(BIError):
    """
    Raises on 401.
    Auth session for BI lives only for some time.
    Need to re-auth in that case.
    """


def _raise_for_status(response):  # type: (Response) -> None
    if response.status_code == 401:
        raise NeedAuth('Session expired or never existed')
    try:
        response.raise_for_status()
    except requests.HTTPError:
        content = json.loads(response.content)
        raise BIError('An %s HTTP error occurred "%s" for url: %s' % (
            response.status_code,
            content.get('message'),
            response.url,
        ))


class BusinessIntelligenceAPI:

    yt_proxy = 'hahn'
    yt_path = settings.BI_YT_PATH

    yt_tables = {
        'income': '/income',
        'detailed_income': '/detailed_income',
        'assignment': '/assignment',
        'vesting': '/vesting',
        'rate': '/rate',
    }

    def _parsed(self, row: dict) -> dict:
        """мимикрируем под ответ ручек bi, где ключи капсом"""
        return {
            key.upper(): value
            for key, value in row.items()
        }

    def __init__(self, instance_name: str):
        assert instance_name in self.yt_tables
        self.instance_name = instance_name

    def get_yt_table(self, instance_name: str):
        client = yt.YtClient(proxy=self.yt_proxy, token=settings.YT_TOKEN)
        try:
            return client.read_table(
                table=self.yt_path + self.yt_tables[instance_name],
                format=yt.JsonFormat(attributes={'encode_utf8': False}),
            )
        except YtTokenError as ex:
            raise NeedAuth(ex.message)

    def get_data(self) -> Iterator[Dict]:
        yt_table = self.get_yt_table(self.instance_name)
        for row in yt_table:
            yield self._parsed(row)
