# coding: utf-8

from decimal import Decimal
import json
import logging
from typing import Dict, Set, Optional
from collections import defaultdict

from django.db import transaction

from review.lib import helpers, encryption
from review.staff.models import Person

from review.bi.api import BusinessIntelligenceAPI
from review.bi.models import (
    BIPersonIncome,
    BIPersonDetailedIncome,
    BIPersonAssignment,
    BIPersonVesting,
    BICurrencyConversionRate,
)


logger = logging.getLogger(__name__)


class BIEntitySync(object):

    datasource = None  # type: str

    def __init__(self, *args, **kwargs):
        self.fetched = 0
        self.to_delete = set()
        self.to_update = []
        self.to_create = []
        self.counters = {
            'fetched': 0,
            'created': 0,
            'updated': 0,
            'deleted': 0,
        }

    def fetched_one_more(self):
        self.counters['fetched'] += 1

    def remove_unused_fields(self, data):
        pass

    def on_unknown_login(self, login):
        pass

    def data_id(self, data, person_id, login):
        raise NotImplementedError

    def do_not_delete_data(self, data_id):
        # type: (int) -> None
        self.to_delete.discard(data_id)

    def data_already_exists(self, data_id):
        # type: (int) -> bool
        raise NotImplementedError

    def existing_data_hash(self, data_id):
        # type: (int) -> int
        raise NotImplementedError

    def make_new(self, data_id, data, data_hash, person_id):
        raise NotImplementedError

    def update_existing(self, data_id, data, data_hash, person_id):
        raise NotImplementedError

    def on_data_has_no_changes(self, data_id, login):
        raise NotImplementedError

    def apply(self):
        raise NotImplementedError


class BIEntitySyncChunked(BIEntitySync):

    chunk_size = 100000

    def its_time_to_flush(self) -> bool:
        self.created_count = 0
        self.updated_count = 0
        return len(self.to_create) >= self.chunk_size or len(self.to_update) >= self.chunk_size

    def apply_chunk(self):
        raise NotImplementedError


@transaction.atomic
def run_sync(entity_sync, dry_run, only_logins=None):
    # type: (BIEntitySync, bool, Optional[Set]) -> None
    chunked_mode = isinstance(entity_sync, BIEntitySyncChunked)
    login_to_person_id_map = dict(Person.objects.values_list('login', 'id'))

    for data in BusinessIntelligenceAPI(entity_sync.datasource).get_data():
        entity_sync.fetched_one_more()
        entity_sync.remove_unused_fields(data)
        login = data.pop('LOGIN')

        if only_logins and login not in only_logins:
            if login in login_to_person_id_map:
                # do not delete old data
                person_id = login_to_person_id_map[login]
                data_id = entity_sync.data_id(data, person_id, login)
                entity_sync.do_not_delete_data(data_id)
            continue

        if login not in login_to_person_id_map:
            entity_sync.on_unknown_login(login)
        else:
            person_id = login_to_person_id_map[login]
            data_id = entity_sync.data_id(data, person_id, login)
            data_hash = hash(json.dumps(data, sort_keys=True))
            if entity_sync.data_already_exists(data_id):
                if data_hash == entity_sync.existing_data_hash(data_id):
                    entity_sync.on_data_has_no_changes(data_id, login)
                else:
                    entity_sync.update_existing(data_id, data, data_hash, person_id)
            else:
                entity_sync.make_new(data_id, data, data_hash, person_id)

            if chunked_mode and not dry_run and entity_sync.its_time_to_flush():
                entity_sync.apply_chunk()

    if not dry_run:
        entity_sync.apply()


class BIAssignmentSync(BIEntitySync):
    datasource = 'assignment'

    def __init__(self):
        super(BIAssignmentSync, self).__init__()
        self.assignment_id_to_hash_map = dict(BIPersonAssignment.objects.values_list('id', 'hash'))
        self.to_delete = set(self.assignment_id_to_hash_map.keys())

    def remove_unused_fields(self, data):
        # Единственное поле с чувствительными данными
        data.pop('ASG_SALARY', None)

    def data_id(self, data, _person_id, login):
        return int(data['ASSIGNMENT_ID'])

    def data_already_exists(self, data_id):
        self.do_not_delete_data(data_id)
        return data_id in self.assignment_id_to_hash_map

    def on_data_has_no_changes(self, data_id, login):
        logger.debug('BI Assignment %s of person `%s` has no changes', data_id, login)

    def update_existing(self, data_id, data, data_hash, person_id):
        self.to_update.append(BIPersonAssignment(id=data_id, data=data, hash=data_hash))

    def make_new(self, data_id, data, data_hash, person_id):
        self.to_create.append(BIPersonAssignment(id=data_id, data=data, hash=data_hash, person_id=person_id))

    def existing_data_hash(self, data_id):
        return self.assignment_id_to_hash_map[data_id]

    def apply(self):
        BIPersonAssignment.objects.filter(id__in=self.to_delete).delete()

        for assignment in self.to_update:
            assignment.save(update_fields=['data', 'hash', 'updated_at'])

        BIPersonAssignment.objects.bulk_create(self.to_create)

        self.counters['updated'] = len(self.to_update)
        self.counters['created'] = len(self.to_create)
        self.counters['deleted'] = len(self.to_delete)


@helpers.timeit_no_args_logging
def sync_bi_assignment_data(dry_run=False):
    assignment_sync = BIAssignmentSync()
    run_sync(assignment_sync, dry_run)
    return assignment_sync.counters


class BIIncomeSync(BIEntitySync):
    datasource = 'income'

    def __init__(self):
        super(BIIncomeSync, self).__init__()
        self.person_id_to_income_hash_map = dict(BIPersonIncome.objects.values_list('person_id', 'hash'))
        self.to_delete = set(self.person_id_to_income_hash_map.keys())

    def data_id(self, _data, person_id, login):
        return person_id

    def data_already_exists(self, data_id):
        self.do_not_delete_data(data_id)
        return data_id in self.person_id_to_income_hash_map

    def existing_data_hash(self, data_id):
        return self.person_id_to_income_hash_map[data_id]

    def on_data_has_no_changes(self, _data_id, login):
        logger.debug('BI Income of person `%s` has no changes', login)

    def update_existing(self, data_id, data, data_hash, person_id):
        data = json.dumps(data, sort_keys=True)
        self.to_update.append(BIPersonIncome(person_id=data_id, data=encryption.encrypt(data), hash=data_hash))

    def make_new(self, data_id, data, data_hash, person_id):
        data = json.dumps(data, sort_keys=True)
        self.to_create.append(BIPersonIncome(
            person_id=person_id,
            data=encryption.encrypt(data),
            hash=data_hash,
        ))

    def apply(self):
        BIPersonIncome.objects.filter(person_id__in=self.to_delete).delete()

        for income in self.to_update:
            income.save(update_fields=['data', 'hash', 'updated_at'])

        BIPersonIncome.objects.bulk_create(self.to_create)

        self.counters['updated'] = len(self.to_update)
        self.counters['created'] = len(self.to_create)
        self.counters['deleted'] = len(self.to_delete)


@helpers.timeit_no_args_logging
def sync_bi_income_data(dry_run=False, only_logins=None):
    income_sync = BIIncomeSync()
    run_sync(income_sync, dry_run, only_logins)
    return income_sync.counters


class BIDetailedIncomeSync(BIEntitySyncChunked):

    datasource = 'detailed_income'

    def __init__(self, *args, **kwargs):
        super(BIDetailedIncomeSync, self).__init__(*args, **kwargs)
        self.data_id_to_detailed_income_hash_map = dict(BIPersonDetailedIncome.objects.values_list('unique', 'hash'))
        self.to_delete = set(self.data_id_to_detailed_income_hash_map)

    def data_id(self, _data, person_id, login):
        month_id = _data['MTH_ID']
        return f'{login}_{month_id}'

    def data_already_exists(self, data_id):
        self.do_not_delete_data(data_id)
        return data_id in self.data_id_to_detailed_income_hash_map

    def existing_data_hash(self, data_id):
        return self.data_id_to_detailed_income_hash_map[data_id]

    def on_data_has_no_changes(self, _data_id, login):
        pass

    def update_existing(self, data_id, data, data_hash, person_id):
        data = json.dumps(data, sort_keys=True)
        assert data_hash == hash(data)
        self.to_update.append(
            BIPersonDetailedIncome(
                unique=data_id,
                person_id=person_id,
                data=encryption.encrypt(data),
                hash=data_hash,
            )
        )

    def make_new(self, data_id, data, data_hash, person_id):
        data = json.dumps(data, sort_keys=True)
        assert data_hash == hash(data)
        self.to_create.append(
            BIPersonDetailedIncome(
                unique=data_id,
                person_id=person_id,
                data=encryption.encrypt(data),
                hash=data_hash,
            )
        )

    def apply_chunk(self):
        """Сохраняет промежуточные данные в базу, обновляет счётчики изменений для общей статистики"""

        # update: теряем логику created_at, но поштучно слишком долго.
        BIPersonDetailedIncome.objects.filter(unique__in=[di.unique for di in self.to_update]).delete()
        BIPersonDetailedIncome.objects.bulk_create(self.to_update, batch_size=5000)

        # create
        BIPersonDetailedIncome.objects.bulk_create(self.to_create, batch_size=5000)

        self.counters['updated'] += len(self.to_update)
        self.counters['created'] += len(self.to_create)

        self.to_update = []
        self.to_create = []

    def apply(self):
        # delete
        BIPersonDetailedIncome.objects.filter(unique__in=self.to_delete).delete()

        # update
        BIPersonDetailedIncome.objects.filter(unique__in=[di.unique for di in self.to_update]).delete()
        BIPersonDetailedIncome.objects.bulk_create(self.to_update, batch_size=5000)

        # create
        BIPersonDetailedIncome.objects.bulk_create(self.to_create, batch_size=5000)

        self.counters['updated'] += len(self.to_update)
        self.counters['created'] += len(self.to_create)
        self.counters['deleted'] = len(self.to_delete)


@helpers.timeit_no_args_logging
def sync_bi_detailed_income_data(dry_run=False, only_logins=None):
    detailed_income_sync = BIDetailedIncomeSync()
    run_sync(detailed_income_sync, dry_run, only_logins)
    return detailed_income_sync.counters


class BIVestingSync(BIEntitySync):
    datasource = 'vesting'

    def __init__(self):
        super(BIVestingSync, self).__init__()
        self.person_id_to_vesting_hash_map = dict(BIPersonVesting.objects.values_list('person_id', 'hash'))
        self.to_delete = set(self.person_id_to_vesting_hash_map.keys())
        self.person_id_to_data = defaultdict(dict)  # {person_id: {vest_le_name: {key: value}}}

    def _person_data_received(self, person_id: int, vesting_data: Dict) -> None:
        vest_le_name = vesting_data.pop('VEST_LE_NAME') or ''
        self.person_id_to_data[person_id][vest_le_name] = vesting_data

    def data_id(self, _data, person_id, login):
        return person_id

    def data_already_exists(self, data_id):
        return data_id in self.person_id_to_vesting_hash_map

    def existing_data_hash(self, data_id):
        self.do_not_delete_data(data_id)
        return self.person_id_to_vesting_hash_map[data_id]

    def on_data_has_no_changes(self, _data_id, login):
        pass

    def update_existing(self, data_id, data, data_hash, person_id):
        self._person_data_received(person_id=data_id, vesting_data=data)

    def make_new(self, data_id, data, data_hash, person_id):
        self._person_data_received(person_id=data_id, vesting_data=data)

    def _parse_persons_data(self):
        self.to_update = []
        self.to_create = []

        for person_id, vesting_data in self.person_id_to_data.items():
            vesting_data_str = json.dumps(vesting_data, sort_keys=True)
            data_hash = hash(vesting_data_str)
            if not self.data_already_exists(data_id=person_id):
                self.to_create.append(
                    BIPersonVesting(
                        person_id=person_id,
                        data=encryption.encrypt(vesting_data_str),
                        hash=data_hash,
                    )
                )
            elif data_hash == self.existing_data_hash(person_id):
                logger.debug(f'BI Vesting of person with id `{person_id}` has no changes')
            else:
                self.to_update.append(
                    BIPersonVesting(
                        person_id=person_id,
                        data=encryption.encrypt(vesting_data_str),
                        hash=data_hash,
                    )
                )

    def apply(self):
        self._parse_persons_data()
        BIPersonVesting.objects.filter(person_id__in=self.to_delete).delete()

        for vesting in self.to_update:
            vesting.save(update_fields=['data', 'hash', 'updated_at'])

        BIPersonVesting.objects.bulk_create(self.to_create, batch_size=50)

        self.counters['updated'] = len(self.to_update)
        self.counters['created'] = len(self.to_create)
        self.counters['deleted'] = len(self.to_delete)


@helpers.timeit_no_args_logging
def sync_bi_vesting_data(dry_run=False, only_logins=None):
    vesting_sync = BIVestingSync()
    run_sync(vesting_sync, dry_run, only_logins)
    return vesting_sync.counters


def _rate_changed(rate_data, rate_record):
    if rate_record.rate != Decimal(rate_data['RATE']):
        return True

    if rate_record.legal_entity_convert_type != rate_data['LEGAL_ENTITY_CONVERT_TYPE']:
        return True

    return False


@helpers.timeit_no_args_logging
def sync_bi_rates(dry_run=False):
    to_create = []
    to_update = []
    fetched = 0

    for rate_data in BusinessIntelligenceAPI('rate').get_data():
        fetched += 1
        key = '{}_{}_{}_{}'.format(
            rate_data['FROM_CURRENCY_ID'],
            rate_data['TO_CURRENCY_ID'],
            rate_data['CONVERSION_DAY'],
            rate_data['LEGAL_ENTITY_ID'],
        )

        try:
            rate_record = BICurrencyConversionRate.objects.get(id=key)
            if _rate_changed(rate_data, rate_record):
                rate_record.rate = Decimal(rate_data['RATE'])
                rate_record.legal_entity_convert_type = rate_data['LEGAL_ENTITY_CONVERT_TYPE']
                to_update.append(rate_record)
        except:
            to_create.append(BICurrencyConversionRate(
                id=key,
                from_currency=rate_data['FROM_CURRENCY_ID'],
                to_currency=rate_data['TO_CURRENCY_ID'],
                conversion_date=rate_data['CONVERSION_DAY'],
                legal_entity_name=rate_data['LEGAL_ENTITY_NAME'],
                organization_id=int(rate_data['LEGAL_ENTITY_ID']),
                rate=rate_data['RATE'],
                legal_entity_convert_type=rate_data['LEGAL_ENTITY_CONVERT_TYPE'],
            ))

    if not dry_run:
        for record in to_update:
            record.save(update_fields=('rate', 'legal_entity_convert_type'))

        BICurrencyConversionRate.objects.bulk_create(to_create)

    return {
        'fetched': fetched,
        'created': len(to_create),
        'updated': len(to_update),
    }
