from collections import defaultdict
import json
import logging
from datetime import date
from decimal import Decimal
from typing import Dict, Iterable, List, Optional, DefaultDict

import ylog

from review.bi import (
    person_income,
    models as bi_models,
    marks
)
from review.bi.person_income import PersonIncome
from review.bi.serializers import AssignmentDataSerializer
from review.bi.utils import bi_bool
from review.lib import encryption


logger = logging.getLogger(__name__)


def get_actual_grades(observables_logins):
    # type: (List[str]) -> Dict[str, Optional[int]]
    income_data = (
        bi_models.BIPersonIncome.objects
        .filter(person__login__in=observables_logins)
        .values_list('person_id', 'data')
    )
    decrpytped = {
        person_id: json.loads(encryption.decrypt(data))
        for person_id, data in income_data
    }
    res = {}
    for person_id, decrpytped_data in decrpytped.items():
        value = decrpytped_data.get('GRADE_MAIN')
        try:
            grade = int(value)
        except Exception:
            logging.warn(
                'Grade for {} looks weird: type {}'.format(person_id, type(value)),
                exc_info=True,
            )
            grade = None
        res[person_id] = grade
    return res


def grade_lte_18(grade):
    # type: (Optional[int]) -> bool
    # 0 and -1 are special values
    # these are handled as >18
    # None is technically an error, but it is better not to fail here
    return grade not in [0, -1, None] and grade <= 18


def get_assignments(person_ids, serializer_context=None):
    # type (List[int]) -> Dict[str, List[Dict]]
    # Output format can be seen in AssignmentDataSerializer
    query = (
        bi_models.BIPersonAssignment.objects
        .filter(person_id__in=person_ids)
    )
    res = {}
    for assignment in query:
        serialized = AssignmentDataSerializer.serialize_many(
            objects=[
                dict(id=assignment.id, **assignment.data)
                for assignment in query
            ],
            context=serializer_context,
        )
        res[assignment.person_id] = serialized
    return res


def get_main_assignment(assignments):
    # type: (List[Dict]) -> Optional[int]
    # assignments format can be seen in AssignmentDataSerializer
    for it in assignments:
        if it['main'] is True:
            return it


def get_rates_for_currency(org_id, currency):
    # type: (int, str) -> Dict[str, Decimal]
    return dict(
        bi_models.BICurrencyConversionRate.objects
        .filter(
            organization_id=org_id,
            conversion_date__lte=date.today(),
            to_currency=currency,
        )
        .order_by('from_currency', '-conversion_date')
        .distinct('from_currency')
        .values_list('from_currency', 'rate')
    )


def get_persons_rates(
    person_ids,  # type: Optional[List[int]]
    currencies,  # type: Optional[List[str]]
):
    # type: (...) -> DefaultDict[int, DefaultDict[str, DefaultDict[str, Decimal]]]
    # Output: person_id -> to_currency -> from_currency -> rate
    org_to_person = defaultdict(list)
    for person_id, assignmetns in get_assignments(person_ids).items():
        main_assignment = get_main_assignment(assignmetns) or {}
        org_id = main_assignment.get('organization_id')
        if org_id:
            org_to_person[org_id].append(person_id)

    query = (
        bi_models.BICurrencyConversionRate.objects
        .filter(
            organization_id__in=org_to_person,
            conversion_date__lte=date.today(),
        )
        .order_by('organization_id', 'from_currency', 'to_currency', '-conversion_date')
        .distinct('organization_id', 'from_currency', 'to_currency')
        .values_list('organization_id', 'from_currency', 'to_currency', 'rate')
    )
    if currencies:
        query = query.filter(
            from_currency__in=currencies,
            to_currency__in=currencies,
        )
    rates = defaultdict(
        lambda: defaultdict(
            lambda: defaultdict(Decimal)
        )
    )
    for org_id, from_, to_, rate in query:
        for person_id in org_to_person[org_id]:
            rates[person_id][to_][from_] = rate
    return rates


def get_person_incomes(
    person_ids,  # type: Iterable[int]
    long_forecast=False,  # type: bool
):
    # type: (...) -> Dict[str, PersonIncome]
    marks_lib = marks.Library()

    incomes = (
        bi_models.BIPersonIncome.objects
        .filter(person_id__in=person_ids)
        .values_list('person__login', 'data')
    )

    vestings = (
        bi_models.BIPersonVesting.objects.filter(person_id__in=person_ids).values_list('person__login', 'data')
    )
    login_to_vesting = {
        login: encryption.decrypt_json(data)
        for login, data in vestings
    }

    res = {}
    for login, encrypted in incomes:
        with ylog.context.LogContext(income_creating_for=login):
            income_obj = person_income.create(
                marks_lib=marks_lib,
                long_forecast=long_forecast,
                income_data=encryption.decrypt_json(encrypted),
                vesting_data=login_to_vesting.get(login),
            )
            if income_obj:
                res[login] = income_obj
    return res


def get_person_income(login, long_forecast=False):
    # type: (str, bool) -> Optional[PersonIncome]
    income = get_person_income_decrypt_data(login)
    detailed_income = get_person_detailed_income_decrypt_data(login)
    vesting = (
        bi_models.BIPersonVesting.objects
        .filter(person__login=login)
        .values_list('data', flat=True)
        .first()
    )
    if vesting:
        vesting = encryption.decrypt_json(vesting)
    with ylog.context.LogContext(income_creating_for=login):
        return person_income.create(
            marks_lib=marks.Library(),
            long_forecast=long_forecast,
            income_data=income,
            detailed_income_data=detailed_income,
            vesting_data=vesting,
        )


def get_person_income_decrypt_data(login) -> dict:
    income = (
        bi_models.BIPersonIncome.objects
        .filter(person__login=login)
        .values_list('data', flat=True)
        .first()
    )
    if income:
        income = encryption.decrypt_json(income)
    return income


def get_person_detailed_income_decrypt_data(login) -> List[dict]:
    detailed_income = (
        bi_models.BIPersonDetailedIncome.objects
        .filter(person__login=login)
        .values_list('data', flat=True)
        .order_by('unique')
    )
    return [
        encryption.decrypt_json(income)
        for income in detailed_income
    ]


def is_person_review_announced(login):
    income_data = get_person_income_decrypt_data(login)
    return bi_bool(income_data.get('ANNOUNCED_FLAG')) if income_data else False
