# coding: utf-8
from typing import Dict, Optional

import attr

from review.core import models as core_models


@attr.s
class Mark(object):
    scale_id = attr.ib(type=str)
    mark = attr.ib(type=str)
    meta = attr.ib(type=Dict, factory=dict)


class Library:
    """
    HACK: can be broken in future
    This class exists only for BI module logic
    It exists due to bi service loses metadata for marks
    Correct way to get meta for a mark is:
    1. Person participates in a review
    2. Person gets mark during the review
    3. Review has MarksScale, that has all available marks (with meta) for this review
    4. By commbining person's mark, review.id and MarksScale we can get correct meta
    5. Some marks are out of scales, like "-" (means "no mark").
    This way of getting meta is broken due to lost review.id in bi data
    So we are just searching for most recent data for every mark
    and hope not to mislead anyone.
    For marks out of scale we assume most recent scale.
    ¯\_(ツ)_/¯
    """

    def __init__(self):
        scales = list(core_models.MarksScale.objects.order_by('-id'))
        self._default_scale_id = scales[-1].id
        self._marks_meta = {}
        for marks_scale in scales:
            for mark, meta in marks_scale.scale.items():
                if mark in self._marks_meta:
                    continue
                meta = meta if isinstance(meta, dict) else {}
                self._marks_meta[mark] = Mark(
                    scale_id=marks_scale.id,
                    mark=mark,
                    meta=meta,
                )

    def get(self, mark_string):
        # type: (Optional[str]) -> Optional[Mark]
        if not mark_string:
            return

        # Some marks are lowercase in review db and uppercase in BI
        # It is highly likely to have this case
        # so we check .lower() first
        meta = self._marks_meta.get(mark_string.lower())
        if meta:
            return meta
        meta = self._marks_meta.get(mark_string)
        if meta:
            return meta
        return Mark(
            scale_id=self._default_scale_id,
            mark=mark_string,
            meta={},
        )
