# coding: utf-8
import logging
import calendar

from datetime import date
from typing import Dict, List, Tuple, Optional

import attr

from review.bi import (
    models as bi_models,
    marks,
)
from review.core import const


log = logging.getLogger(__name__)


@attr.s
class TotalIncome(object):
    from_ = attr.ib(type=str)
    to = attr.ib(type=Optional[str])
    forecast = attr.ib(type=float)
    salary = attr.ib(type=float)
    bonus = attr.ib(type=float)
    other = attr.ib(type=float)
    signup = attr.ib(type=float)
    signup2 = attr.ib(type=float)
    retention = attr.ib(type=float)
    vesting = attr.ib(type=float)
    income = attr.ib(type=float)
    piecerate = attr.ib(type=float)
    welcome = attr.ib(type=float)
    vesting_cash_payment = attr.ib(type=float)
    deferred_payment = attr.ib(type=float)

    vesting_detail = attr.ib(type=Dict[str, float])


@attr.s
class Totals(object):
    per_calendar_year = attr.ib(type=List[TotalIncome], factory=list)
    from_current_date = attr.ib(type=List[TotalIncome], factory=list)
    detailed = attr.ib(type=List[TotalIncome], factory=list)


@attr.s
class PersonIncome(object):

    gold_pay_last = attr.ib(type=bool, default=False)
    gold_pay_curr = attr.ib(type=bool, default=False)
    gold_opt_last = attr.ib(type=bool, default=False)
    gold_opt_curr = attr.ib(type=bool, default=False)

    mark_last = attr.ib(type=Optional[marks.Mark], default=None)
    mark_current = attr.ib(type=Optional[marks.Mark], default=None)

    grade_main = attr.ib(type=Optional[int], default=None)
    grade_current = attr.ib(type=Optional[int], default=None)
    grade_last = attr.ib(type=Optional[int], default=None)
    up_current = attr.ib(type=Optional[int], default=None)
    up_last = attr.ib(type=Optional[int], default=None)

    salary = attr.ib(type=float, default=0)
    avg_salary = attr.ib(type=float, default=0)
    currency = attr.ib(type=str, default='RUB')

    income = attr.ib(type=Totals, factory=Totals)


def dates_range(month_id: int) -> str:
    """month_id задан целым числом вида 201905, содержащим год и номер месяца"""
    year = month_id // 100
    month = month_id % 100
    last_day = calendar.monthrange(year, month)[1]

    return {
        'first_date': date(year, month, day=1),
        'last_date': date(year, month, day=last_day)
    }


def is_forecast_month(month_id: int) -> bool:
    last_date = dates_range(month_id)['last_date']
    return last_date > date.today()


def create(
    marks_lib: marks.Library,
    long_forecast: bool,
    income_data: Optional[dict] = None,
    detailed_income_data: List[dict] = None,
    vesting_data: dict = None,
) -> PersonIncome:
    if income_data is None:
        log.warning('No income data')
        return

    if vesting_data is None:
        log.warning('No vesting data')
        return

    detailed_income_data = detailed_income_data or []

    def to_int(key):
        value = income_data.get(key)
        try:
            return int(value)
        except (TypeError, ValueError):
            log.warning('Fail to convert to int {}'.format(key))
            return

    def to_bool(key):
        try:
            return bool(int(income_data.get(key, '0')))
        except (TypeError, ValueError):
            log.warning('Fail to convert to bool {}'.format(key))
            return False

    def to_float(key, from_=income_data, default=0.0):
        try:
            return float(from_.get(key))
        except (TypeError, ValueError):
            log.warning('Fail to convert to foat {}'.format(key))
            return default

    periods_keys = ['4', '3', '2', '1', '01']
    if long_forecast:
        periods_keys.extend(['02', '03', '04'])

    return PersonIncome(
        mark_last=marks_lib.get(income_data.get('MARK_LAST')),
        mark_current=marks_lib.get(income_data.get('MARK_CURRENT')),

        currency=income_data['MAIN_ASG_CURRENCY'],

        salary=to_float('SALARY'),
        avg_salary=to_float('AVG_SAL'),
        grade_last=to_int('GRADE_LAST'),
        grade_current=to_int('GRADE_CURRENT'),
        grade_main=to_int('GRADE_MAIN'),
        up_current=to_int('UP_CURRENT'),
        up_last=to_int('UP_LAST'),
        gold_opt_curr=to_bool('GOLD_OPT_CURR'),
        gold_opt_last=to_bool('GOLD_OPT_LAST'),
        gold_pay_curr=to_bool('GOLD_PAY_CURR'),
        gold_pay_last=to_bool('GOLD_PAY_LAST'),
        income=Totals(
            per_calendar_year=[
                TotalIncome(
                    from_=income_data['CAL%s_PERIOD' % num],
                    to=None,
                    forecast=num.startswith('0'),
                    salary=to_float('CAL%s_SALARY' % num),
                    bonus=to_float('CAL%s_BONUS' % num),
                    other=to_float('CAL%s_OTHER' % num),
                    signup=to_float('CAL%s_SIGNUP' % num),
                    signup2=to_float('CAL%s_SIGNUP2' % num),
                    retention=to_float('CAL%s_RETENTION' % num),
                    vesting=to_float('CAL%s_VESTING' % num),
                    income=to_float('CAL%s_INCOME' % num),
                    piecerate=to_float('CAL%s_PIECERATE' % num),
                    welcome=to_float('CAL%s_WELCOME' % num),
                    vesting_cash_payment=to_float('CAL%s_VESTCASHPAY' % num),
                    deferred_payment=to_float('CAL%s_DEFPAY' % num),
                    vesting_detail={
                        legal_entity: to_float(
                            'CAL%s_LE_VESTING' % num,
                            vesting_data[legal_entity],
                            default=None,
                        )
                        for legal_entity in vesting_data
                        if legal_entity != 'LOGIN'
                    },
                ) for num in periods_keys
            ],
            from_current_date=[
                TotalIncome(
                    from_=income_data['EFF%s_BEG' % num],
                    to=income_data['EFF%s_END' % num],
                    forecast=num.startswith('0'),
                    salary=to_float('EFF%s_SALARY' % num),
                    bonus=to_float('EFF%s_BONUS' % num),
                    other=to_float('EFF%s_OTHER' % num),
                    signup=to_float('EFF%s_SIGNUP' % num),
                    signup2=to_float('EFF%s_SIGNUP2' % num),
                    retention=to_float('EFF%s_RETENTION' % num),
                    vesting=to_float('EFF%s_VESTING' % num),
                    income=to_float('EFF%s_INCOME' % num),
                    piecerate=to_float('EFF%s_PIECERATE' % num),
                    welcome=to_float('EFF%s_WELCOME' % num),
                    vesting_cash_payment=to_float('EFF%s_VESTCASHPAY' % num),
                    deferred_payment=to_float('EFF%s_DEFPAY' % num),
                    vesting_detail={
                        legal_entity: to_float(
                            'EFF%s_LE_VESTING' % num,
                            vesting_data[legal_entity],
                            default=None,
                        )
                        for legal_entity in vesting_data
                        if legal_entity != 'LOGIN'
                    },
                ) for num in periods_keys
            ],
            detailed=[
                TotalIncome(
                    from_=dates_range(month_income['MTH_ID'])['first_date'].isoformat(),
                    to=dates_range(month_income['MTH_ID'])['last_date'].isoformat(),
                    forecast=is_forecast_month(month_income['MTH_ID']),
                    salary=to_float('SALARY', from_=month_income),
                    bonus=to_float('BONUS_SUM', from_=month_income),
                    other=to_float('OTHER_SUM', from_=month_income),
                    signup=to_float('SIGNUP_SUM', from_=month_income),
                    signup2=to_float('SIGNUP2_SUM', from_=month_income),
                    retention=to_float('RETENTION_SUM', from_=month_income),
                    vesting=to_float('VEST_SUM', from_=month_income),
                    income=to_float('INCOME_SUM', from_=month_income),
                    piecerate=to_float('PIECERATE_SUM', from_=month_income),
                    welcome=to_float('WELCOME_SUM', from_=month_income),
                    vesting_cash_payment=to_float('VESTCASHPAY_SUM', from_=month_income),
                    deferred_payment=to_float('DEFPAY_SUM', from_=month_income),
                    vesting_detail={},
                ) for month_income in detailed_income_data if month_income.get('MTH_ID')
            ],
        ),
    )
