# coding: utf-8
import logging
from django.utils import timezone

from review.bi.utils import bi_bool
from review.lib.serializers import BaseSerializer, F


log = logging.getLogger(__name__)


class AssignmentDataSerializer(BaseSerializer):

    fields = {
        F('id', source='[id]'),
        F('number', source='[ASSIGNMENT_NUMBER]'),
        F('currency', source='[ASG_CURRENCY]'),
        F('report_date', source='[REPORT_DATE]'),
        F('rate', source='[ASG_RATE]', cast_type=float),
        F('rate_fact', source='[ASG_RATE_FACT]', cast_type=float),
        F('contr_end_date', source='[ASG_CONTR_END_DATE]'),
        F('organization', source='[ASG_LEGAL_ENTITY_NAME]'),
        F('organization_id', source='[ASG_LEGAL_ENTITY_ID]', cast_type=int),
        F('position', source='[ASG_JOB_NAME]'),
        F('contract_number', source='[ASG_CONTRACT_NUMBER]'),
        F('main', source='get_main()'),
    }

    @classmethod
    def get_main(cls, obj):
        return bi_bool(obj['MAIN_ASG_FLAG'])

    @classmethod
    def serialize(cls, obj, fields_requested=None, context=None):
        result = super(AssignmentDataSerializer, cls).serialize(obj, fields_requested, context)
        if result.get('main'):
            result['report_date'] = get_announcement_date(result['report_date'], context)
        return result


def get_announcement_date(report_date, context):
    """
    It is dirty hack that shall be changed in future.
    For now (2022 year) review dates was moved, so review results is announced in april/october.
    In these months you still get old pre-review salary, but in finncap post-review salary is
    already showed. To prevent misunderstanding we will change the dates:
    if review has announced already, we change the date to 1st of May/November
    (the first day of month when you start getting post-review salary).
    """
    if context and context.get('review_announced'):
        try:
            report_date = timezone.now().date()
            if 4 <= report_date.month <= 5:
                report_date = report_date.replace(month=5, day=1)
            elif 10 <= report_date.month <= 11:
                report_date = report_date.replace(month=11, day=1)
            report_date = report_date.strftime('%m/%d/%Y')
        except Exception:
            log.exception('Cannot change report_date: %s', report_date)

    return report_date
