import logging
import threading
import time
from decimal import Decimal
from datetime import datetime, date
from django_cryptography.fields import EncryptedMixin
from django.contrib.contenttypes.models import ContentType

from review.compensations.lib import encryption
from review.compensations.models import choices

logger = logging.getLogger(__name__)


class ActionLog:

    def __init__(self):
        self._ctx = threading.local()

    def __enter__(self):
        pass

    def __exit__(self, exp_type, exp_value, exp_tr):
        # Если мы получили ValidationError, то эксепшена здесь не будет, так как мы его
        # обработали ранее. При этом писать в лог пустоту тоже смысла нет.
        # https://st.yandex-team.ru/FEMIDA-2370
        if not exp_type and self.ctx.snapshots:
            self.save()
        else:
            self.reset()

    @property
    def ctx(self):
        return self._ctx

    def get_serialized_obj(self, obj):

        result = {}
        for field in obj._meta.fields:
            name = field.get_attname()
            value = getattr(obj, name)
            if issubclass(field.__class__, EncryptedMixin):
                value = encryption.encrypt(str(value))
            else:
                value = field.get_prep_value(value)

            if isinstance(value, (date, datetime)):
                value = value.isoformat()
            elif isinstance(value, Decimal):
                value = str(value)

            result[name] = value
        return result

    def is_initialized(self):
        return bool(getattr(self.ctx, 'action_name', None))

    def init(self, action_name, user=None):
        if self.is_initialized():
            msg = 'Lost actionlog. action_name: `%s`'
            logger.error(msg, self.ctx.action_name)

        # Если пользователь не авторизован, нельзя записывать это в actionlog
        if user is not None and not user.is_authenticated:
            user = None

        self.ctx.action_name = action_name
        self.ctx.user = user
        self.ctx.snapshots = {}
        self.ctx.context = {}
        self.model_content_type_map = dict(ContentType.objects.values_list('model', 'id'))
        return self

    def reset(self):
        self.ctx.action_name = None
        self.ctx.user = None
        self.ctx.snapshots = {}
        self.ctx.context = {}

    def add_context(self, data):
        self.ctx.context.update(data)

    def add_object(self, obj, reason):
        from review.compensations.models import Snapshot

        obj_id = obj.id
        obj_str = obj.__class__.__name__.lower()
        obj_type_id = self.model_content_type_map.get(obj_str)
        obj_uid = (obj_type_id, obj_str, obj_id)

        snapshot = self.ctx.snapshots.get(obj_uid)
        data = self.get_serialized_obj(obj)

        if not snapshot:
            self.ctx.snapshots[obj_uid] = Snapshot(
                reason=reason,
                data=data,
                obj_type_id=obj_type_id,
                obj_id=obj_id,
                obj_str=obj_str,
            )
        else:
            snapshot.data = data

    def save(self):
        from review.compensations.models import LogRecord, Snapshot

        start = time.time()

        try:
            log_record = LogRecord.objects.create(
                action_name=self.ctx.action_name,
                user=self.ctx.user,
                context=self.ctx.context,
            )
            for snapshot in self.ctx.snapshots.values():
                snapshot.log_record = log_record
            Snapshot.objects.bulk_create(self.ctx.snapshots.values())
        except Exception:
            logger.exception('[ACTION_LOG] Failed to save log record `%s`', self.ctx.action_name)
            raise
        finally:
            self.reset()

        logger.info(
            '[ACTION_LOG] Log record added. Action name: %s, time to save: %s',
            log_record.action_name, time.time() - start
        )


actionlog = ActionLog()
