from django.contrib import admin, messages
from django.db.models import QuerySet
from django.http.request import HttpRequest
from django.urls import reverse
from django.utils.safestring import mark_safe

from review.compensations import tasks, logic, models
from review.compensations.models import choices
from review.compensations.actionlog.decorators import action_logged


class PaymentSchedulesFileAdmin(admin.ModelAdmin):

    model = models.PaymentSchedulesFile

    list_display = (
        'name',
        'file_link',
        'status',
        'payment_type',
        'uploader',
        'schedules_link',
    )
    readonly_fields = (
        'uploader',
        'created',
        'modified',
        'errors',
    )
    fields = (
        'name',
        'file',
        'payment_type',
        'errors',
    )

    def schedules_link(self, obj: models.PaymentSchedulesFile) -> str:
        if obj.status != models.choices.SCHEDULES_FILE_STATUS.successed:
            return ''
        link = reverse('admin:compensations_personpaymentschedule_changelist')
        url = f'<a href="{link}?source={obj.id}">Записи в базе</a>'
        return mark_safe(url)

    def file_link(self, obj: models.PaymentSchedulesFile) -> str:
        if obj.file is None:
            return ''
        link = reverse('compensations_api:payment_schedules_file', kwargs={'file_id': obj.id})
        url = f'<a href="{link}?source={obj.id}">Скачать .xls</a>'
        return mark_safe(url)

    @action_logged('paymentschedulesfile_save')
    def save_model(self, request: HttpRequest, obj: models.PaymentSchedulesFile, form, change) -> None:
        obj.uploader = request.user
        result = super().save_model(request, obj, form, change)
        tasks.process_input_files.delay(file_ids=[obj.id])
        return result

    @action_logged('paymentschedulesfile_parse')
    def parse_files_action(self, request, queryset):
        valid_statuses_for_parsing = (
            models.choices.SCHEDULES_FILE_STATUS.pending,
            models.choices.SCHEDULES_FILE_STATUS.failed,
        )
        if queryset.exclude(status__in=valid_statuses_for_parsing).exists():
            self.message_user(
                request=request,
                message='Only `pending` or `failed` files could be parsed',
                level=messages.ERROR,
            )
            return

        for imported_file_model in queryset:
            logic.parse_compensation_plan_file_to_payment_schedules(imported_file_model)

    parse_files_action.short_description = 'Распарсить эксельку'

    actions = (parse_files_action, )


class PersonPaymentInline(admin.TabularInline):

    model = models.PersonPayment

    readonly_fields = ('amount', 'currency', 'raw_date', 'date', 'status', 'message')
    show_change_link = True
    show_full_result_count = True
    extra = 0
    can_delete = False

    def has_add_permission(self, request: HttpRequest, obj=None) -> bool:
        return False


class PersonPaymentScheduleAdmin(admin.ModelAdmin):

    model = models.PersonPaymentSchedule

    list_display = ('person_login', 'full_name', 'payment_type', 'salary', 'currency', 'payments_start_date')
    search_fields = ('person_login', 'assignment', 'source__name')
    readonly_fields = ('source', 'person_login', )
    list_filter = ('payment_type', 'legal_entity', 'source__name')

    inlines = (PersonPaymentInline, )

    fields = (
        'source',
        'legal_entity',
        ('person_login', 'full_name', 'assignment'),
        'payment_type',
        ('salary', 'currency'),
        ('bonus', 'bonus_absolute'),
        ('financial_reporting_center', 'product', 'gl_service'),
        'payments_start_date',
        ('date_begin', 'date_end'),
        'message',
        'status',
    )

    @action_logged('personpayment_generate')
    def generate_payments_action(self, request, queryset):
        """Из schedule нагенерить выплат"""
        for schedule in queryset.select_related('payment_type__plan'):
            logic.generate_person_payments_from_schedule(schedule)

    generate_payments_action.short_description = 'Нагенерировать выплат из выбранных планов'

    @action_logged('personpayment_delete')
    def delete_payments_action(self, request, queryset):
        queryset.delete()

    delete_payments_action.short_description = 'Удалить выбранные планы выплат!'

    actions = (generate_payments_action, delete_payments_action, )

    def is_processed(self, obj: models.PersonPaymentSchedule) -> bool:
        return obj.processed_at is not None

    @action_logged('personpaymentschedule_save')
    def save_model(self, request: HttpRequest, obj, form, change) -> None:
        return super().save_model(request, obj, form, change)


class PersonPaymentAdmin(admin.ModelAdmin):

    model = models.PersonPayment

    list_display = ('person_login', 'amount', 'raw_date', 'date', 'status')
    ordering = ('schedule', 'raw_date')
    list_filter = ('status', 'schedule__source__name', 'schedule__source__payment_type', 'export__date')
    readonly_fields = ('schedule', 'status', 'export')

    fields = (
        'schedule',
        ('amount', 'currency'),
        ('raw_date', 'date'),
        'export',
        'status',
    )

    date_hierarchy = 'raw_date'

    @action_logged('personpayment_specify_date')
    def specify_payment_dates_action(self, request, queryset):
        """Заполнение уточнённых дат выплат с учётом регулярных выплат"""
        logic.specify_payment_dates(queryset)

    specify_payment_dates_action.short_description = 'Рассчитать фактичесткие даты выплат'

    actions = (specify_payment_dates_action, )

    def get_queryset(self, request: HttpRequest) -> QuerySet:
        return super().get_queryset(request).select_related('schedule__source')

    def person_login(self, obj: models.PersonPayment) -> str:
        return obj.schedule.person_login

    @action_logged('personpayment_save')
    def save_model(self, request: HttpRequest, obj, form, change) -> None:
        return super().save_model(request, obj, form, change)


class ExportPaymentsAdmin(admin.ModelAdmin):

    model = models.ExportPayments

    readonly_fields = ('file', 'author', 'status', 'file')
    list_display = ('date', 'payment_type', 'author', 'export_file_link', 'status')

    def export_file_link(self, obj: models.ExportPayments) -> str:
        valid_statuses = [
            choices.EXPORT_FILE_STATUSES.ready,
            choices.EXPORT_FILE_STATUSES.done,
        ]
        if obj.status not in valid_statuses or obj.file is None:
            return f''

        link = reverse('compensations_api:export_payments_file', kwargs={'export_id': obj.id})
        url = f'<a href="{link}?source={obj.id}">Скачать .xls</a>'
        return mark_safe(url)

    @action_logged('exportpayments_save')
    def save_model(self, request: HttpRequest, obj: models.PaymentSchedulesFile, form, change) -> None:
        obj.author = request.user
        obj.file = None
        result = super().save_model(request, obj, form, change)
        try:
            logic.process_xls_export(obj)
        except RuntimeError as ex:
            self.message_user(
                request=request,
                message=ex.args[0],
                level=messages.ERROR,
            )
        return result

    @action_logged('exportpayments_mark_done')
    def mark_payments_as_done_action(self, request, queryset):
        try:
            for export_instance in queryset:
                logic.mark_payments_as_done(export_instance)
        except RuntimeError as ex:
            self.message_user(
                request=request,
                message=ex.args[0],
                level=messages.ERROR,
            )

    mark_payments_as_done_action.short_description = 'Пометить выплаты как проведённые'

    actions = (mark_payments_as_done_action, )
