import io
import json
import datetime
import logging
import itertools
import openpyxl

from calendar import monthrange
from dateutil.relativedelta import relativedelta
from decimal import Decimal
from typing import Iterable, Iterator, List, Dict, Tuple, Type
from openpyxl.writer.excel import save_virtual_workbook

from django.utils import timezone
from django.core.files.base import File, ContentFile
from django.db import transaction
from django.db.models.query import QuerySet
from django.db.models import Value, CharField
from django import forms as django_forms

from review.compensations import models, forms, exceptions
from review.compensations.actionlog import signals


logger = logging.getLogger(__name__)


def parse_excel(excel_file: File, form_class: Type[django_forms.Form]) -> Iterator[Dict]:
    column_names = list(form_class.base_fields)

    workbook = openpyxl.load_workbook(io.BytesIO(excel_file.read()))
    sheet = workbook.active
    rows = sheet.rows
    # [
    #   'Логин*', 'ФИО', '№ назначения', 'Премия, сумма', 'Премия, %', 'Оклад', 'Валюта',
    #   'ЦФО', 'Продукт', 'Сервис GL', 'Юр. лицо', 'Дата С', 'Дата По', 'Статус',
    #   'Дата отсчёта', 'Сообщение'
    # ]
    captions = [cell.value for cell in next(rows)]
    for row_cells in rows:
        values = [c.value for c in row_cells]
        yield dict(zip(column_names, values))


def parse_compensation_plan_file_to_payment_schedules(input_file: models.PaymentSchedulesFile) -> None:
    """Парсим файл на строчки PersonPaymentSchedule с валидацией формой PersonPaymentForm"""
    errors = {}

    payment_schedule_forms = [
        forms.PersonPaymentScheduleForm(data=row_dict)
        for row_dict in parse_excel(input_file.file, forms.PersonPaymentScheduleForm)
    ]

    errors = {
        num: form.errors
        for num, form in enumerate(payment_schedule_forms, start=1)
        if not form.is_valid()
    }

    if errors:
        input_file.errors = json.dumps(errors, indent=2, ensure_ascii=False)
        input_file.status = models.choices.SCHEDULES_FILE_STATUS.failed
        input_file.save()
        return

    payment_schedule_models = [
        models.PersonPaymentSchedule(
            source=input_file,
            payment_type=input_file.payment_type,
            **form.cleaned_data
        )
        for form in payment_schedule_forms
    ]
    with transaction.atomic():
        models.PersonPaymentSchedule.objects.bulk_create(
            payment_schedule_models,
            batch_size=1000,
        )
        signals.post_bulk_create.send(
            sender=models.PersonPaymentSchedule,
            queryset=payment_schedule_models,
        )
        input_file.status = models.choices.SCHEDULES_FILE_STATUS.successed
        input_file.errors = ''
        input_file.save(update_fields=['status', 'errors'])


def assert_no_payments_done(schedule_id: int) -> None:
    done_payments = models.PersonPayment.objects.filter(
        schedule_id=schedule_id,
        status=models.choices.PAYMENT_STATUSES.done,
    )
    if done_payments.exists():
        raise exceptions.PaymentsUpdateConflictError(
            f'Cannot update payments by schedule {schedule_id}'
            'because it has already done payments'
        )


def generate_payment_details(schedule: models.PersonPaymentSchedule) -> Iterator[Tuple]:
    """Логика рассчёта дат и сумм выплат исходя из плана выплат и дат регулярных выплат"""
    payments_count = len(schedule.payment_type.plan.periods)
    already_counted = Decimal(0)

    payment_date = schedule.payments_start_date - datetime.timedelta(days=50)

    to_add = 0
    payment_plan = schedule.payment_type.plan
    for payment_num, (months, percents) in enumerate(zip(payment_plan.periods, payment_plan.scheme), start=1):
        to_add += months
        payment_date = schedule.payments_start_date + relativedelta(months=to_add)
        payment_amount = Decimal(round(schedule.bonus_absolute * percents) / 100)
        if payment_num == payments_count:
            payment_amount = schedule.bonus_absolute - already_counted
        yield (payment_amount, payment_date)

        already_counted += payment_amount


def _group_by_country(qs: QuerySet, country_field='country_id', order_field='date') -> Dict[str, Iterable]:
    return {
        country_code: list(items)
        for country_code, items in itertools.groupby(
            qs.order_by(country_field, order_field),
            key=lambda item: getattr(item, country_field)
        )
    }


def get_last_possible_payment_date(
        payment: models.PersonPayment,
        regulars: Dict[str, datetime.date],
        payment_type_to_regular_payments: Dict[int, List[str] or None],
) -> datetime.date:
    """
    Возвращает дату последнего регулярного платежа, исходя из того,
    какие регулярные платежи свойственны элементу данного PersonPayment.
    Либо саму расчётную дату raw_date в случае если Элемент не предусматривает привязки к регулярным платежам.
    """
    pt_id = payment.schedule.payment_type_id
    if pt_id not in payment_type_to_regular_payments:
        # Для этой страны не хватает Элемента с таким payment_type и поэтому не понятно как считать
        raise exceptions.ElementNotFoundError(f'Corresponding Element with payment_type_id={pt_id} not found')

    if payment_type_to_regular_payments[pt_id] == []:
        # Если соответствующий Элемент не связан ни с одним видом регулярных платежей
        # то деньги выплатим непосредстванно в расчётную дату.
        return payment.raw_date

    possible_dates = [
        regulars[rp]
        for rp in payment_type_to_regular_payments[pt_id]
        if regulars[rp] and regulars[rp] <= payment.raw_date
    ]

    if possible_dates == []:
        # Нет регулярного платежа (RegularPaymentDate) нужного типа, предшествующего этой выплате
        raise exceptions.RegularPaymentDateNotFoundError(f'Lack of RegularPaymentDate for payment {payment}')

    return max(possible_dates)


def generate_dates_for_payments(
        person_payments: list,
        regular_payment_dates: list,
        country_code: str,
) -> Iterator[Tuple[models.PersonPayment, datetime.date]]:
    """
    Для каждой выплаты (person_payment) поиск ближайшей предшествующей даты регулярных выплат,
    из графика регулярных выплат (regular_payment_dates), в которую деньги фактически должны быть перечислены.
    """
    this_country_elements = models.Element.objects.filter(country__code=country_code)
    payment_type_to_regular_payments = {
        type_id: regular_payments
        for type_id, regular_payments in
        this_country_elements.values_list('type_id', 'regular_payments')
    }

    sorted_by_dates = sorted(
        itertools.chain(person_payments, regular_payment_dates),
        key=lambda d: getattr(d, 'raw_date', None) or getattr(d, 'date')
    )

    current_regular_dates = dict.fromkeys(models.choices.REGULAR_PAYMENT_TYPES._db_values)

    for model_instance in sorted_by_dates:
        if isinstance(model_instance, models.RegularPaymentDate):
            current_regular_dates[model_instance.type] = model_instance.date
            continue
        actual_payment_date = get_last_possible_payment_date(
            payment=model_instance,
            regulars=current_regular_dates,
            payment_type_to_regular_payments=payment_type_to_regular_payments,
        )

        yield model_instance, actual_payment_date


def specify_payment_dates(payments_qs) -> None:
    """Пересчитываем значение date из raw_date по графику регулярных выплат и сохраняем в поле date"""

    # Пока нет привязки к странам (COMP-18) добавляем поле country_id='ru' якобы разметка по странам уже есть
    payments_qs = payments_qs.annotate(
        country_id=Value('ru', output_field=CharField())
    )
    person_payments_by_country = _group_by_country(payments_qs)
    regular_payments_by_country = _group_by_country(models.RegularPaymentDate.objects.all())

    for country_code, person_payments in person_payments_by_country.items():
        regular_payments = regular_payments_by_country[country_code]
        dates_gen = generate_dates_for_payments(person_payments, regular_payments, country_code)
        for person_payment, regular_date in dates_gen:
            person_payment.date = regular_date
            person_payment.save()  # bulk_update в Джанге 2.2


def generate_person_payments_from_schedule(schedule: models.PersonPaymentSchedule) -> None:
    """Идемподентно генерируем PersonPayment'ы согласно плану выплат"""
    assert_no_payments_done(schedule.id)

    payments = []
    for payment_amount, payment_date in generate_payment_details(schedule):
        payments.append(
            models.PersonPayment(
                schedule=schedule,
                amount=payment_amount,
                raw_date=payment_date,
                date=None,
                currency=schedule.currency,
                status=models.choices.PAYMENT_STATUSES.scheduled,
            )
        )

    with transaction.atomic():
        models.PersonPayment.objects.filter(schedule=schedule).delete()
        models.PersonPayment.objects.bulk_create(payments)
        signals.post_bulk_create.send(
            sender=models.PersonPayment,
            queryset=payments,
        )
        schedule.processed_at = timezone.now()
        schedule.save(update_fields=['processed_at', 'modified'])


def assert_no_outdated_payments(export_date: datetime.date, payment_type_id: int) -> None:
    outdated_payments = models.PersonPayment.objects.filter(
        status=models.choices.PAYMENT_STATUSES.scheduled,
        schedule__payment_type_id=payment_type_id,
        raw_date__lte=export_date,
        date=None,
    )
    if outdated_payments.exists():
        raise exceptions.OutdatedPaymentsError(
            f'Some payments are not ready for {export_date}. '
            f'Make sure all RegularPaymentDates exists. Problem size: {outdated_payments.count()}'
        )


def get_payments_for_export(export_instance: models.ExportPayments) -> QuerySet:
    """
    Пока задаём фильтр для отчётов только через `payment_type` и `date` — тип выплат и дату,
        на которую они должны быть проведены
    Остальные поля Экспорта пока не задействованы.
    """
    return (
        models.PersonPayment.objects.filter(
            status=models.choices.PAYMENT_STATUSES.scheduled,
            schedule__payment_type=export_instance.payment_type,
            date__lte=export_instance.date
        )
        .exclude(date=None)
    )


def _get_export_xls_row(payments_qs: QuerySet) -> Iterable[list]:
    for person_payment in payments_qs.select_related('schedule'):
        schedule = person_payment.schedule
        yield [
            schedule.person_login,
            schedule.full_name,
            schedule.assignment,
            person_payment.amount,  # Единственное поле с расчитанными данными.
            schedule.bonus,
            schedule.salary,
            schedule.currency,
            schedule.financial_reporting_center,
            schedule.product,
            schedule.gl_service,
            schedule.legal_entity,
            schedule.date_begin,
            schedule.date_end,
            schedule.status,
            schedule.message,
        ]


def make_xls_report(target_payments: QuerySet) -> bytes:
    workbook = openpyxl.Workbook()
    worksheet = workbook.active
    worksheet.title = 'export example'
    titles_row = [
        'Логин*', 'ФИО', '№ назначения', 'Премия, сумма', 'Премия, %', 'Оклад', 'Валюта',
        'ЦФО', 'Продукт', 'Сервис GL', 'Юр. лицо', 'Дата С', 'Дата По', 'Статус', 'Сообщение',
    ]
    worksheet.append(titles_row)
    for payment_record in _get_export_xls_row(target_payments):
        worksheet.append(payment_record)

    return save_virtual_workbook(workbook)


def process_xls_export(export_instance: models.ExportPayments) -> None:
    assert_no_outdated_payments(export_instance.date, export_instance.payment_type_id)

    target_payments_qs = get_payments_for_export(export_instance)
    report_file_content = make_xls_report(target_payments=target_payments_qs)

    with transaction.atomic():
        export_instance.file.save(name='doesntmatter.xlsx', content=ContentFile(report_file_content))

        target_payments_qs.update(export=export_instance)
        signals.post_update.send(sender=models.PersonPayment, queryset=target_payments_qs)

        export_instance.status = models.choices.EXPORT_FILE_STATUSES.ready
        export_instance.save()


def assert_no_payments_already_done(payments_qs: QuerySet) -> None:
    done_payments = payments_qs.filter(status=models.choices.PAYMENT_STATUSES.done)
    if done_payments.exists():
        raise exceptions.PaymentsExportConflictError(
            f'This export contains {done_payments.count()} payments '
            'that has already been done by another export.'
        )


def mark_payments_as_done(export_instance: models.ExportPayments) -> None:
    payments = models.PersonPayment.objects.filter(export=export_instance)
    assert_no_payments_already_done(payments)
    payments.update(status=models.choices.PAYMENT_STATUSES.done)
    signals.post_update.send(sender=models.PersonPayment, queryset=payments)

    export_instance.status = models.choices.EXPORT_FILE_STATUSES.done
    export_instance.save()
