import logging

from django.db import models
from django.contrib.contenttypes.models import ContentType
from django.contrib.postgres.fields import JSONField
from django.utils import timezone

from review.staff.models import Person
from review.compensations.models import choices

from review.compensations.actionlog.signals import *  # noqa


logger = logging.getLogger(__name__)


class LogRecord(models.Model):

    action_name = models.CharField(max_length=127, db_index=True)
    action_time = models.DateTimeField(default=timezone.now, db_index=True)
    user = models.ForeignKey(
        to=Person,
        on_delete=models.PROTECT,
        null=True,
        blank=True,
    )
    context = JSONField(null=True)

    def __str__(self):
        return f'LogRecord {self.id}, {self.action_name}'


class Snapshot(models.Model):

    log_record = models.ForeignKey(
        to=LogRecord,
        on_delete=models.CASCADE,
        related_name='snapshots',
    )
    reason = models.CharField(max_length=10, choices=choices.SNAPSHOT_REASONS)
    data = JSONField(null=True)

    obj_type = models.ForeignKey(
        to=ContentType,
        on_delete=models.SET_NULL,
        blank=True,
        null=True,
    )
    obj_id = models.IntegerField()
    # ContentType может и не быть, поэтому запишем хотя бы что-то
    obj_str = models.CharField(max_length=127)

    synced_to_yt = models.DateTimeField(null=True, default=None)

    def __str__(self):
        return f'Snapshot {self.id} of log record {self.log_record_id}'

    class Meta:
        indexes = [
            models.Index(fields=['obj_id', 'obj_type'], name='actionlog_snapshot_obj_index')
        ]
