from django.contrib.postgres.fields import ArrayField
from django.db import models

from . import choices
from .fields import ChoiceArrayField


class Country(models.Model):

    code = models.CharField(max_length=2, primary_key=True)
    name_ru = models.CharField(max_length=32)
    name_en = models.CharField(max_length=32)

    def __str__(self) -> str:
        return f'<{self.name_ru} ({self.code})>'

    class Meta:
        verbose_name = 'Страна'
        verbose_name_plural = 'Страны (справочник)'


class PaymentPlan(models.Model):

    slug = models.CharField(max_length=32, unique=True, primary_key=True, verbose_name='Слаг')
    periods = ArrayField(
        base_field=models.IntegerField(),
        default=list,
        size=12,
        verbose_name='Схема выплат по месяцам'
    )
    scheme = ArrayField(
        base_field=models.IntegerField(),
        default=list,
        size=12,
        verbose_name='Схема выплат по процентам'
    )

    def __str__(self) -> str:
        return f'<{self.slug} ({self.scheme})>'

    class Meta:
        verbose_name = 'График выплат'
        verbose_name_plural = 'Графики выплат (справочник)'


class PaymentType(models.Model):
    name_ru = models.CharField(max_length=50, unique=True, verbose_name='Название')
    name_en = models.CharField(max_length=50, unique=True, verbose_name='Название англ.')
    plan = models.ForeignKey(PaymentPlan, db_index=True, on_delete=models.PROTECT, verbose_name='График выплат')
    payment_postponing = models.BooleanField(
        default=False,
        verbose_name='Отсрочка выплат за длительные отсутствия',
    )
    alert_layoff_before_payment = models.BooleanField(
        default=False,
        verbose_name='Уведомление в случае увольнения до выплаты',
    )

    def __str__(self) -> str:
        return f'<{self.name_ru} (plan: {self.plan})>'

    class Meta:
        verbose_name = 'Тип выплат'
        verbose_name_plural = 'Типы выплат (справочник)'


class Element(models.Model):

    name_ru = models.CharField(max_length=50, verbose_name='Название')
    name_en = models.CharField(max_length=50, verbose_name='Название англ.')
    type = models.ForeignKey(PaymentType, on_delete=models.PROTECT, verbose_name='Тип выплат')
    country = models.ForeignKey(Country, db_column='country_code', on_delete=models.CASCADE)

    regular_payments = ChoiceArrayField(
        base_field=models.CharField(max_length=10, choices=choices.REGULAR_PAYMENT_TYPES),
        default=list,
        blank=True,
        verbose_name='Привязка к регулярным платежам',
    )

    class Meta:
        verbose_name = 'Элемент'
        verbose_name_plural = 'Элементы (справочник)'


class RegularPaymentDate(models.Model):

    date = models.DateField(verbose_name='Дата выплаты')
    type = models.CharField(
        max_length=10,
        choices=choices.REGULAR_PAYMENT_TYPES,
        null=True,
        blank=True,
        default=None,
        verbose_name='Тип выплат',
    )
    country = models.ForeignKey(Country, db_column='country_code', on_delete=models.CASCADE)

    def __str__(self) -> str:
        return f'<RegularPayment ({self.country_id}) at {self.date}>'

    class Meta:
        verbose_name = 'Дата выплаты'
        verbose_name_plural = 'График регулярных выплат (справочник)'
