from urllib.parse import MAX_CACHE_SIZE
from uuid import uuid4

from model_utils.models import TimeStampedModel
from django.db import models
from django_cryptography.fields import encrypt

from review.lib.s3 import S3Storage

from . import choices
from .base_entities import PaymentPlan, PaymentType


def _get_xlsx_filename(*args, **kwargs):
    return f'{uuid4().hex}.xlsx'


class PaymentSchedulesFile(TimeStampedModel):

    name = models.CharField(max_length=128, verbose_name='Название выплаты')
    file = models.FileField(
        upload_to=_get_xlsx_filename,
        max_length=512,
        storage=S3Storage(),
        null=True,
        blank=True,
        verbose_name='.xlsx-файл со значениями выплат сотрудникам'
    )
    payment_type = models.ForeignKey(PaymentType, db_index=True, null=True, blank=True, default=None, on_delete=models.PROTECT, verbose_name='Тип выплат')
    payments_start_date = models.DateField(null=True, default=None, verbose_name='Дата отсчёта')
    status = models.CharField(
        choices=choices.SCHEDULES_FILE_STATUS,
        default=choices.SCHEDULES_FILE_STATUS.pending,
        max_length=32,
        verbose_name='Статус',
    )
    processed_at = models.DateTimeField(null=True, blank=True, default=None, verbose_name='Время последней обработки')
    errors = models.TextField(blank=True, verbose_name='Ошибки валидации строк в файле')
    uploader = models.ForeignKey('staff.Person', on_delete=models.PROTECT, verbose_name='Автор файла')

    class Meta:
        verbose_name = 'Файл выплат (Экселька)'
        verbose_name_plural = '1. Файлы планов выплат'


class PersonPaymentSchedule(TimeStampedModel):
    """Одна строка с расписанием выплат содруднику из исходного excel-файла"""
    source = models.ForeignKey(to=PaymentSchedulesFile, on_delete=models.CASCADE, verbose_name='Файл-источник')

    person_login = models.CharField(max_length=50, verbose_name='Логин')
    full_name = models.CharField(max_length=50, verbose_name='ФИО')

    payment_type = models.ForeignKey(PaymentType, db_index=True, null=True, blank=True, default=None, on_delete=models.PROTECT, verbose_name='Тип выплат')
    assignment = models.CharField(max_length=16, null=True, default=None, verbose_name='Назначение')

    bonus = encrypt(models.DecimalField(max_digits=7, decimal_places=2, default=0, null=True, blank=True, verbose_name='Премия (%)'))  # percents
    bonus_absolute = encrypt(models.DecimalField(max_digits=20, decimal_places=2, default=0, null=True, blank=True, verbose_name='Премия (абс)'))

    salary = encrypt(models.DecimalField(max_digits=20, decimal_places=2, default=0, null=True, blank=True, verbose_name='Зарплата'))
    currency = models.CharField(max_length=8, choices=choices.CURRENCIES, verbose_name='Валюта')

    payments_start_date = models.DateField(verbose_name='Дата отсчёта')

    # ЦФО - центр финансовой отчётности
    financial_reporting_center = models.CharField(
        max_length=32,
        verbose_name='Центр финансовой отчётности',
        null=True,
        blank=True,
    )

    product = models.CharField(
        max_length=32,
        blank=True,
        null=True,
        verbose_name='Продукт',
    )

    gl_service = models.CharField(
        max_length=32,
        blank=True,
        null=True,
        verbose_name='GL-сервис',
    )

    legal_entity = models.CharField(
        max_length=120,
        blank=True,
        null=True,
        verbose_name='Юр. лицо',
    )

    date_begin = models.DateField(null=True, blank=True, default=None, verbose_name='Дата с')
    date_end = models.DateField(null=True, blank=True, default=None, verbose_name='Дата по')

    status = models.CharField(max_length=16, verbose_name='Статус')
    payments_start_date = models.DateField(verbose_name='Дата отсчёта')
    message = models.CharField(max_length=128, blank=True, default='', verbose_name='Сообщение')
    processed_at = models.DateTimeField(null=True, blank=True, default=None, verbose_name='Время последней обработки')

    payment_postponement = models.IntegerField(default=0, verbose_name='количество дней, на которые отсрочиваются выплаты по этому расписанию')

    def __str__(self) -> str:
        return f'<PaymentSchedule for {self.person_login}>'

    class Meta:
        unique_together = ('source', 'person')

    class Meta:
        verbose_name = 'План по сотруднику'
        verbose_name_plural = '2. Планы по сотрудникам'


class PersonPayment(models.Model):
    """Выплата сотруднику по конкретному графику в конкретную дату"""
    schedule = models.ForeignKey(
        to=PersonPaymentSchedule,
        related_name='payments',
        on_delete=models.CASCADE,
        db_index=True,
        verbose_name='План выплат',
    )
    amount = encrypt(models.DecimalField(max_digits=20, decimal_places=2, null=True, verbose_name='Сумма выплаты'))
    currency = models.CharField(max_length=8, choices=choices.CURRENCIES, verbose_name='Валюта')
    raw_date = models.DateField(verbose_name='Дата выплаты (расчётная)')
    date = models.DateField(
        null=True,
        blank=True,
        default=None,
        verbose_name='Дата выплаты (фактическая)'
    )
    status = models.CharField(
        max_length=16,
        choices=choices.PAYMENT_STATUSES,
        default=choices.PAYMENT_STATUSES.scheduled,
        verbose_name='Статус',
    )
    export = models.ForeignKey(
        'ExportPayments',
        on_delete=models.PROTECT,
        null=True,
        blank=True,
        related_name='person_payments',
        verbose_name='Выгрузка, в которую попала выплата',
    )
    message = models.CharField(max_length=128, blank=True, default='', verbose_name='Сообщение')

    def __str__(self) -> str:
        return f'<Payment for {self.schedule.person_login} at {self.date or self.raw_date}>'

    class Meta:
        verbose_name = 'Выплата'
        verbose_name_plural = '3. Выплаты'


class ExportPayments(models.Model):
    """Выгрузка выплат по указанным фильтрам"""
    date = models.DateField(verbose_name='На какую дату')
    payment_type = models.ForeignKey(
        to=PaymentType,
        on_delete=models.PROTECT,
        null=True,
        blank=True,
        default=None,
        verbose_name='Тип выплат',
    )
    file = models.FileField(
        upload_to=_get_xlsx_filename,
        max_length=64,
        storage=S3Storage(),
        null=True,
        blank=True,
        verbose_name='.xlsx-файл с выгрузкой'
    )
    status = models.CharField(
        choices=choices.EXPORT_FILE_STATUSES,
        default=choices.EXPORT_FILE_STATUSES.draft,
        max_length=10,
        verbose_name='Статус',
    )
    author = models.ForeignKey(
        'staff.Person',
        on_delete=models.PROTECT,
        verbose_name='Автор выгрузки',
    )

    class Meta:
        verbose_name = 'Выгрузка'
        verbose_name_plural = '4. Выгрузки'
