# -*- coding: utf-8 -*-
from django.contrib import admin
from django.db import models as django_models, transaction
from django.conf import settings

from review.core import models
from review.core import const
from review.core.logic import roles as role_inheritance, review_actions
from review.lib.admin import ReadOnlyProductionModelAdmin, ReadOnlyProductionAdminMixin


def get_instance_roles(instance, verbose_roles):
    roles = instance.roles.all()
    if roles:
        grouped_by_type = {}
        for role in roles:
            group = grouped_by_type.setdefault(role.type, [])
            group.append(role.person.login)
        for key, value in grouped_by_type.items():
            grouped_by_type[key] = ', '.join(value)

        return '; '.join(
            '{}: {}'.format(verbose_roles[role_type], logins)
            for role_type, logins in grouped_by_type.items()
        )
    else:
        return None


@admin.register(models.Review)
class ReviewAdmin(ReadOnlyProductionModelAdmin):
    list_per_page = 20

    fields = (
        'name',
        'author',
        'start_date',
        'finish_date',
        'finish_feedback_date',
        'finish_submission_date',
        'finish_calibration_date',
        'finish_approval_date',
        'evaluation_from_date',
        'evaluation_to_date',
        'product_schema_loaded',

        'mark_mode',
        'level_change_mode',
        'salary_change_mode',
        'bonus_mode',
        'options_rsu_mode',
        'options_rsu_format',
        'deferred_payment_mode',

        'status',
        'notify_reminder_days',
        'notify_reminder_date_from',
        'notify_reminder_date_to',
        'notify_events_other',
        'notify_events_superreviewer',

        'salary_date',
    )

    list_display = ('id', ) + fields + ('roles',)

    raw_id_fields = ('author',)

    list_filter = (
        'start_date',
        'finish_date',
        'status',
        'salary_date',
    )
    search_fields = (
        'name',
        'id',
    )

    def delete_fast(self, request, queryset):
        queryset.delete()

    if not settings.IS_PRODUCTION_DB:
        actions = [delete_fast]

    def get_queryset(self, request):
        queryset = super(ReviewAdmin, self).get_queryset(request)
        return queryset.prefetch_related('roles', 'roles__person')

    def roles(self, instance):
        return get_instance_roles(instance, const.ROLE.REVIEW.VERBOSE)


@admin.register(models.PersonReview)
class PersonReviewAdmin(ReadOnlyProductionModelAdmin):
    list_per_page = 20

    class PersonReviewRoleInline(ReadOnlyProductionAdminMixin, admin.StackedInline):
        model = models.PersonReviewRole
        extra = 0

        fields = ('person', 'type', 'position')

        raw_id_fields = ('person',)
        can_delete = not settings.IS_PRODUCTION_DB

    inlines = [PersonReviewRoleInline]

    fields = [
        'person',
        'review',
        'umbrella',
        'main_product',
    ]

    if not settings.IS_PRODUCTION_DB:
        fields += [
            'status',
            'updated_at',
            'approve_level',
            'mark',
            'goldstar',
            'level_change',
            'salary_change',
            'bonus',
            'options_rsu',
            'options_rsu_legacy',
            'flagged',
        ]

    list_display = [
        'person',
        'review',
        'id',
        'roles',
    ]

    search_fields = (
        'person__login',
        'review__name',
        'review__id',
    )

    raw_id_fields = ('person', 'review', 'umbrella', 'main_product')

    def get_queryset(self, request):
        queryset = super(PersonReviewAdmin, self).get_queryset(request)
        return queryset.prefetch_related('roles', 'roles__person')

    def roles(self, instance):
        return get_instance_roles(instance, const.ROLE.PERSON_REVIEW.VERBOSE)


@admin.register(models.Calibration)
class CalibrationAdmin(ReadOnlyProductionModelAdmin):
    list_per_page = 20

    fields = (
        'name',
        'status',
        'start_date',
        'finish_date',
        'author',
        'notify_users',
    )
    list_display = (
        'id',
    ) + fields
    readonly_fields = () if not settings.IS_PRODUCTION_DB else [
        f for f in fields if f != 'notify_users'
    ]

    list_filter = (
        'start_date',
        'finish_date',
        'status',
    )
    search_fields = (
        'name',
    )
    raw_id_fields = (
        'author',
    )

    def get_queryset(self, request):
        queryset = super(CalibrationAdmin, self).get_queryset(request)
        return queryset.prefetch_related('roles', 'roles__person')

    def roles(self, instance):
        return get_instance_roles(instance, const.ROLE.CALIBRATION.VERBOSE)

    class CalibrationRoleInline(ReadOnlyProductionAdminMixin, admin.StackedInline):
        model = models.CalibrationRole
        extra = 0

        fields = (
            'person',
            'type',
        )
        raw_id_fields = (
            'person',
        )
        can_delete = not settings.IS_PRODUCTION_DB

    inlines = [CalibrationRoleInline]


@admin.register(models.CalibrationPersonReview)
class CalibrationPersonReviewAdmin(ReadOnlyProductionModelAdmin):
    list_per_page = 20

    fields = (
        'calibration',
        'person_review',
        'updated_at',
    )
    list_display = (
        'id',
    ) + fields

    search_fields = (
        'calibration_id',
        'person_review_id',
    )
    raw_id_fields = (
        'calibration',
        'person_review',
    )


@admin.register(models.StoredFilter)
class StoredFilterAdmin(admin.ModelAdmin):

    search_fields = ('owner__login', 'name', 'value')
    fields = ('owner', 'name', 'value')
    list_display = ('id',) + fields
    raw_id_fields = ('owner',)


@admin.register(models.MarksScale)
class MarksScaleAdmin(admin.ModelAdmin):
    fields = ('scale', )
    list_display = ('id', ) + fields


class ReviewRoleAdmin(ReadOnlyProductionModelAdmin):
    fields = ('person', 'review', 'type')
    list_display = ('id',) + fields
    raw_id_fields = ('person',)

    search_fields = ('person__login', 'review__name', 'review__id')


class PersonReviewRoleAdmin(ReadOnlyProductionModelAdmin):
    fields = (
        'person',
        'person_review',
        'type',
        'position',
        'inherited_from',
        'from_calibration_role',
        'from_review_role',
    )
    list_display = ('id',) + fields

    raw_id_fields = ('person', 'person_review')
    search_fields = (
        'person__login',
    )
    list_filter = ('type',)

    def get_search_results(self, request, queryset, search_term):
        queryset, use_distinct = super(PersonReviewRoleAdmin, self).get_search_results(
            request, queryset, search_term
        )
        try:
            int_search = int(search_term)
            foreign_fields = (
                'person_review__id',
                'person_review__review_id',
                'inherited_from',
                'from_calibration_role',
                'from_review_role',
            )
            for field in foreign_fields:
                queryset |= self.model.objects.filter({field: int_search})
        except ValueError:
            pass
        return queryset, use_distinct


class PersonReviewReadersManager(django_models.Manager):
    def get_queryset(self):
        return super(PersonReviewReadersManager, self).get_queryset().filter(
            type__in={const.ROLE.PERSON_REVIEW.READER, const.ROLE.PERSON_REVIEW.SUPERREADER}
        )


class PersonReviewReader(django_models.Model):

    objects = PersonReviewReadersManager()

    class Meta:
        db_table = models.PersonReviewRole._meta.db_table
        managed = False

    person = django_models.ForeignKey(
        to='staff.Person',
        related_name='person_review_reader_roles',
        related_query_name='person_review_reader_role'
    )
    person_review = django_models.ForeignKey(
        to='core.PersonReview',
        related_name='reader_roles',
        related_query_name='reader_role',
        on_delete=django_models.CASCADE
    )
    type = django_models.CharField(
        max_length=2,
        choices=[
            (role, const.ROLE.PERSON_REVIEW.VERBOSE[role])
            for role in (
                const.ROLE.PERSON_REVIEW.READER,
                const.ROLE.PERSON_REVIEW.SUPERREADER,
            )
        ]
    )
    position = django_models.IntegerField(editable=False, default=0)
    created_at_auto = django_models.DateTimeField(auto_now_add=True)
    updated_at_auto = django_models.DateTimeField(auto_now=True)


class ReadersModelAdmin(admin.ModelAdmin):
    fields = ('person', 'person_review', 'type')
    list_display = ('id',) + fields

    raw_id_fields = ('person', 'person_review')
    search_fields = ('person__login', 'person_review__id')
    list_filter = ('type',)

    def save_model(self, request, obj, form, change):
        super(ReadersModelAdmin, self).save_model(request, obj, form, change)
        role_inheritance.denormalize_person_review_roles(
            id__in={obj.person_review_id}
        )

    def delete_model(self, request, obj):
        super(ReadersModelAdmin, self).delete_model(request, obj)
        role_inheritance.denormalize_person_review_roles(
            id__in={obj.person_review_id}
        )


class GlobalRoleAdmin(admin.ModelAdmin):
    fields = ('person', 'type')
    list_display = ('id',) + fields

    search_fields = ('person__login',)
    list_filter = ('type',)
    raw_id_fields = ('person',)

    def has_delete_permission(self, request, obj=None):
        # пока не подключили idm
        return True
        # return not settings.IS_PRODUCTION_DB

    def has_add_permission(self, request):
        # пока не подключили idm
        return True
        # return not settings.IS_PRODUCTION_DB

    @transaction.atomic
    def delete_model(self, request, obj):
        review_actions.delete_from_global_admins(obj)
        super(GlobalRoleAdmin, self).delete_model(request, obj)


class PersonReviewChangeAdmin(ReadOnlyProductionModelAdmin):
    fields = list_display = (
        'subject',
        'person_review',
        'created_at',
        'notified',
    )
    raw_id_fields = ('subject', 'person_review')

    search_fields = ('subject__login', 'person__review__id')


class PersonReviewCommentAdmin(ReadOnlyProductionModelAdmin):
    fields = list_display = (
        'subject',
        'person_review',
        'created_at',
        'notified',
    )
    raw_id_fields = ('subject', 'person_review')

    search_fields = ('subject__login', 'person__review__id')


class ExcelTemplateAdmin(admin.ModelAdmin):
    fields = (
        'review',
        'template_type',
        'file',
    )
    list_display = (
        'review',
        'template_type',
    )
    raw_id_fields = ('review', )
    search_fields = ('review__name', )

    def has_delete_permission(self, request, obj=None):
        if obj and obj.review.status != const.REVIEW_STATUS.ARCHIVE:
            return True
        return False


admin.site.register(models.PersonReviewChange, PersonReviewChangeAdmin)
admin.site.register(models.PersonReviewComment, PersonReviewCommentAdmin)
admin.site.register(models.GlobalRole, GlobalRoleAdmin)
admin.site.register(models.ReviewRole, ReviewRoleAdmin)
admin.site.register(models.PersonReviewRole, PersonReviewRoleAdmin)
admin.site.register(PersonReviewReader, ReadersModelAdmin)
admin.site.register(models.ExcelTemplate, ExcelTemplateAdmin)
