# coding: utf-8

from __future__ import unicode_literals

from collections import defaultdict
from itertools import (
    chain,
    count,
)

from review.staff import const as staff_const


class SPECIAL_VALUE(str):
    def __bool__(self):
        return False


class ConstCreator(object):

    def __init__(self):
        self._created_const = set()

    def create_const(self, s):
        assert s not in self._created_const, 'Const {} is already created'.format(s)
        self._created_const.add(s)
        return s

    def get_all(self):
        return frozenset(self._created_const)


OK = SPECIAL_VALUE('_OK_')
NOT_SET = SPECIAL_VALUE('_NOT_SET_')
NO_ACCESS = SPECIAL_VALUE('_NO_ACCESS_')
NOT_AVAILABLE = SPECIAL_VALUE('_NOT_AVAILABLE_')
EMPTY = SPECIAL_VALUE('_EMPTY_')
ERROR = SPECIAL_VALUE('_ERROR_')
DISABLED = SPECIAL_VALUE('_DISABLED_')
AUTO = SPECIAL_VALUE('_AUTO_')


class REVIEW_STATUS(object):
    DRAFT = 'd'
    IN_PROGRESS = 'i'
    FINISHED = 'f'
    ARCHIVE = 'a'

    VERBOSE = {
        DRAFT: 'draft',
        IN_PROGRESS: 'in_progress',
        FINISHED: 'finished',
        ARCHIVE: 'archive',
    }
    CHOICES = VERBOSE.items()
    ALL = frozenset(VERBOSE)
    VISIBLE = ALL - {DRAFT}
    INACTIVE = frozenset([ARCHIVE])
    ACTIVE = VISIBLE - INACTIVE
    ORDERING = (
        IN_PROGRESS,
        FINISHED,
        ARCHIVE,
        DRAFT,
    )


class REVIEW_OPTIONS_RSU_FORMAT(object):
    ACTUAL = "a"
    LEGACY = "l"
    VERBOSE = {
        ACTUAL: "options_rsu",
        LEGACY: "options_rsu_legacy",
    }
    CHOICES = VERBOSE.items()


class REVIEW_OPTIONS_RSU_UNIT(object):
    QUANTITY = 'q'
    USD = 'u'
    VERBOSE = {
        QUANTITY: 'quantity',
        USD: 'usd',
    }
    CHOICES = VERBOSE.items()


class CALIBRATION_STATUS(object):
    DRAFT = 'd'
    IN_PROGRESS = 'i'
    ARCHIVE = 'a'

    VERBOSE = {
        DRAFT: 'draft',
        IN_PROGRESS: 'in_progress',
        ARCHIVE: 'archive',
    }
    ORDERING = (
        DRAFT,
        IN_PROGRESS,
        ARCHIVE,
    )
    CHOICES = VERBOSE.items()
    ALL = frozenset(VERBOSE)
    VISIBLE = ALL - {DRAFT}
    INACTIVE = frozenset([ARCHIVE])
    ACTIVE = VISIBLE - INACTIVE


CALIBRATION_INHERITANCE_LENGTH = 4


class PERSON_REVIEW_CHANGE_TYPE(object):
    const_creator = ConstCreator()
    cc = const_creator.create_const

    PERSON = cc('p')
    ROBOT = cc('r')
    FILE = cc('f')

    ALL = const_creator.get_all()
    del cc, const_creator

    VERBOSE = {
        PERSON: 'person',
        ROBOT: 'robot',
        FILE: 'file',
    }
    CHOICE = VERBOSE.items()


class MARK(object):
    NO_MARK = '-'
    NOT_SET = '?'
    B_STAR = 'B*'
    DEFAULT = NOT_SET
    TEXT_NO_MARK = 'n/a'

    SPECIAL_VALUES = {
        NO_MARK: None,
        NOT_SET: None,
    }
    TO_NUM = dict(SPECIAL_VALUES)
    TO_NUM[B_STAR] = 2


class GOLDSTAR(object):
    NO = '-'
    BONUS_ONLY = 'b'
    OPTION_ONLY = 'o'
    OPTION_AND_BONUS = 'a'
    DEFAULT = NO

    VERBOSE = {
        NO: 'no_goldstar',
        BONUS_ONLY: 'bonus_only',
        OPTION_ONLY: 'option_only',
        OPTION_AND_BONUS: 'option_and_bonus',
    }
    CHOICES = VERBOSE.items()
    VERBOSE_CHOICES = zip(VERBOSE.values(), VERBOSE.values())
    ALL = frozenset(VERBOSE)


class SALARY_DEPENDENCY_TYPE(object):
    ABSOLUTE = 'a'
    PERCENTAGE = 'p'

    DEFAULT = PERCENTAGE

    VERBOSE = {
        ABSOLUTE: 'absolute',
        PERCENTAGE: 'percentage',
    }
    CHOICES = VERBOSE.items()


class PERSON_REVIEW_STATUS(object):

    WAIT_EVALUATION = 'w'
    # ожидает согласования
    EVALUATION = 'e'

    APPROVAL = 'p'
    # согласовано
    APPROVED = 'a'
    # ожидает объявления
    WAIT_ANNOUNCE = 'n'
    # объявлено
    ANNOUNCED = 'd'

    VERBOSE = {
        WAIT_EVALUATION: 'wait_evaluation',
        EVALUATION: 'evaluation',
        APPROVAL: 'approval',
        APPROVED: 'approved',
        WAIT_ANNOUNCE: 'wait_announce',
        ANNOUNCED: 'announced',
    }
    CHOICES = VERBOSE.items()
    VERBOSE_CHOICES = zip(VERBOSE.values(), VERBOSE.values())
    ALL = frozenset(VERBOSE)


class REVIEW_TYPE(object):
    NORMAL = 'n'
    MIDDLE = 'm'
    BONUS = 'b'
    PROMOTION = 'p'

    VERBOSE = {
        NORMAL: 'normal',
        MIDDLE: 'middle',
        BONUS: 'bonus',
        PROMOTION: 'promotion',
    }
    VERBOSE_REVERSED = {val: key for key, val in VERBOSE.items()}
    CHOICES = VERBOSE.items()
    CHOICES_REVERSED = VERBOSE_REVERSED.items()
    ALL = frozenset(VERBOSE)


class REVIEW_BONUS_TYPE(object):
    MONTHLY = 'm'
    QUARTERLY = 'q'
    HALFYEARLY = 'h'
    PROJECT = 'p'
    SINGLE = 's'
    VERBOSE = {
        MONTHLY: 'monthly',
        QUARTERLY: 'quarterly',
        HALFYEARLY: 'halfyearly',
        PROJECT: 'project',
        SINGLE: 'single',
    }
    VERBOSE_REVERSED = {val: key for key, val in VERBOSE.items()}
    CHOICES = VERBOSE.items()
    CHOICES_REVERSED = VERBOSE_REVERSED.items()
    ALL = frozenset(VERBOSE)


class GLOBAL_ACTIONS(object):
    CREATE_REVIEW = 'create_review'
    CREATE_CALIBRATION = 'create_calibration'
    ADD_PERSON_REVIEWS_TO_CALIBRATION = 'add_person_reviews_to_calibration'
    ALL = frozenset([
        CREATE_REVIEW,
        CREATE_CALIBRATION,
        ADD_PERSON_REVIEWS_TO_CALIBRATION,
    ])


class ROLE(object):

    class GLOBAL(object):
        const_creator = ConstCreator()
        cc = const_creator.create_const

        REVIEW_CREATOR = cc('r')
        CALIBRATION_CREATOR = cc('c')
        SUPPORT = cc('u')
        EXPORTER = cc('e')
        ROBOT = cc('o')
        ADMIN = cc('ag')
        BI_VIEWER = cc('bi')
        LOAN_VIEWER = cc('lo')

        VERBOSE = {
            REVIEW_CREATOR: 'review_creator',
            CALIBRATION_CREATOR: 'calibration_creator',
            SUPPORT: 'support',
            EXPORTER: 'exporter',
            ROBOT: 'robot',
            ADMIN: 'admin',
            BI_VIEWER: 'bi_viewer',
            LOAN_VIEWER: 'loan_viewer',
        }
        ROBOTS = frozenset({EXPORTER, ROBOT})
        CHOICES = VERBOSE.items()
        ALL = const_creator.get_all()

        del const_creator, cc

    class DEPARTMENT(object):
        HEAD = staff_const.STAFF_ROLE.DEPARTMENT.HEAD
        HR_PARTNER = staff_const.STAFF_ROLE.HR.HR_PARTNER
        HR_ANALYST = staff_const.STAFF_ROLE.HR.HR_ANALYST
        FINANCE_VIEWER = staff_const.STAFF_ROLE.HR.FINANCE_VIEWER

        VERBOSE = {
            HEAD: 'head',
            HR_PARTNER: 'hr partner',
            HR_ANALYST: 'hr analytic',
            FINANCE_VIEWER: 'finance_viewer',
        }
        HRS = frozenset({HR_PARTNER, HR_ANALYST})
        ALL = frozenset(VERBOSE)

    class PERSON(object):
        SELF = 'f'

        VERBOSE = {
            SELF: 'self'
        }
        ALL = frozenset(VERBOSE)

    class REVIEW(object):
        ADMIN = 'a'
        SUPERREVIEWER = 's'
        ACCOMPANYING_HR = 'hr'
        VERBOSE = {
            ADMIN: 'review_admin',
            SUPERREVIEWER: 'superreviewer',
            ACCOMPANYING_HR: 'accompanying_hr',
        }
        CHOICES = VERBOSE.items()
        ALL = frozenset(VERBOSE)

    class PERSON_REVIEW(object):
        REVIEWER = 'rv'
        READER = 're'
        SUPERREADER = 'sr'
        # top-ревьюера я сначала пытался сделал через type='tr' & position=0,
        # но код сразу стал корявеньким, так что я выдал ему свою роль
        # (хотя 0 у него будет тоже стоять, что является дублированием,
        # но удобно при сортировке, если вдруг понадобится)
        # у обычных ревьюеров индекс начинается с единицы в сторону уменьшения
        TOP_REVIEWER = 'tr'
        CALIBRATOR_ARCHIVED = 'ar'

        CALIBRATOR_DENORMALIZED = 'cd'
        CALIBRATION_ADMIN_DENORMALIZED = 'ad'
        SUPERREVIEWER_DENORMALIZED = 'sd'

        READER_INHERITED = 'ei'
        SUPERREADER_INHERITED = 'si'
        REVIEWER_INHERITED = 'ri'
        TOP_REVIEWER_INHERITED = 'ti'
        CALIBRATOR_INHERITED = 'ci'
        CALIBRATION_ADMIN_INHERITED = 'ai'
        SUPERREVIEWER_INHERITED = 'pi'

        VERBOSE = {
            REVIEWER: 'reviewer',
            READER: 'reader',
            SUPERREADER: 'superreader',
            TOP_REVIEWER: 'top_reviewer',
            CALIBRATOR_DENORMALIZED: 'calibrator denormalized',
            CALIBRATION_ADMIN_DENORMALIZED: 'calibration admin denormalized',
            SUPERREVIEWER_DENORMALIZED: 'superreviewer denormalized',
            READER_INHERITED: 'reader inherited',
            TOP_REVIEWER_INHERITED: 'top reviewer inherited',
            SUPERREADER_INHERITED: 'superreader inherited',
            REVIEWER_INHERITED: 'reviewer inherited',
            CALIBRATOR_INHERITED: 'calibrator inherited',
            CALIBRATION_ADMIN_INHERITED: 'calibration admin inherited',
            SUPERREVIEWER_INHERITED: 'superreviewer inherited',
            CALIBRATOR_ARCHIVED: 'calibrator archived',
        }
        CHOICES = VERBOSE.items()
        ALL = frozenset(VERBOSE)
        ALL_REVIEWER_ROLES = frozenset([
            REVIEWER,
            REVIEWER_INHERITED,
            TOP_REVIEWER,
            TOP_REVIEWER_INHERITED,
        ])

    class CALIBRATION(object):
        ADMIN = 'ca'
        # дурацкое название, но я не придумал лучше
        CALIBRATOR = 'cc'
        VERBOSE = {
            ADMIN: 'admin',
            CALIBRATOR: 'calibrator'
        }
        CHOICES = VERBOSE.items()
        ALL = frozenset(VERBOSE)

    INHERITANCE = {
        PERSON_REVIEW.REVIEWER: PERSON_REVIEW.REVIEWER_INHERITED,
        PERSON_REVIEW.TOP_REVIEWER: PERSON_REVIEW.TOP_REVIEWER_INHERITED,
        PERSON_REVIEW.READER: PERSON_REVIEW.READER_INHERITED,
        PERSON_REVIEW.SUPERREADER: PERSON_REVIEW.SUPERREADER_INHERITED,
        CALIBRATION.CALIBRATOR: PERSON_REVIEW.CALIBRATOR_INHERITED,
        CALIBRATION.ADMIN: PERSON_REVIEW.CALIBRATION_ADMIN_INHERITED,
        REVIEW.SUPERREVIEWER: PERSON_REVIEW.SUPERREVIEWER_INHERITED,
    }
    INHERITABLE = frozenset(INHERITANCE)
    INHERITED = frozenset(INHERITANCE.values())

    # используем только при получении данных из базы — дальше заменяем
    # на обычные роли
    DENORMALIZATION = {
        REVIEW.SUPERREVIEWER: PERSON_REVIEW.SUPERREVIEWER_DENORMALIZED,
        CALIBRATION.CALIBRATOR: PERSON_REVIEW.CALIBRATOR_DENORMALIZED,
        CALIBRATION.ADMIN: PERSON_REVIEW.CALIBRATION_ADMIN_DENORMALIZED,
    }
    DENORMALIZATION_REVERSED = {
        val: key for key, val in DENORMALIZATION.items()
    }

    INHERITANCE_WITH_DENORMALIZATION = {}
    for key, value in INHERITANCE.items():
        if key in DENORMALIZATION:
            key = DENORMALIZATION[key]
        INHERITANCE_WITH_DENORMALIZATION[key] = value
    INHERITABLE_WITH_DENORMALIZATION = frozenset(INHERITANCE_WITH_DENORMALIZATION)

    # проверим, что у нас нет коллизий для символьного обозначения ролей,
    # которые могут использоваться вместе
    ROLES_THAT_CAN_BE_USED_TOGETHER = (
        PERSON.SELF,
        REVIEW.ADMIN,
        REVIEW.SUPERREVIEWER,
        REVIEW.ACCOMPANYING_HR,
        DEPARTMENT.HEAD,
        DEPARTMENT.HR_PARTNER,
        DEPARTMENT.HR_ANALYST,
        PERSON_REVIEW.REVIEWER,
        PERSON_REVIEW.TOP_REVIEWER,
        PERSON_REVIEW.READER,
        PERSON_REVIEW.SUPERREADER,
        PERSON_REVIEW.CALIBRATOR_DENORMALIZED,
        PERSON_REVIEW.SUPERREVIEWER_DENORMALIZED,
        PERSON_REVIEW.CALIBRATION_ADMIN_DENORMALIZED,
        PERSON_REVIEW.READER_INHERITED,
        PERSON_REVIEW.SUPERREADER_INHERITED,
        PERSON_REVIEW.REVIEWER_INHERITED,
        PERSON_REVIEW.TOP_REVIEWER_INHERITED,
        PERSON_REVIEW.CALIBRATOR_INHERITED,
        PERSON_REVIEW.SUPERREVIEWER_INHERITED,
        CALIBRATION.CALIBRATOR,
        CALIBRATION.ADMIN,
    ) + tuple(GLOBAL.ALL)
    assert len(ROLES_THAT_CAN_BE_USED_TOGETHER) == len(
        set(ROLES_THAT_CAN_BE_USED_TOGETHER))

    # роли, которые участвуют в формировании списка PersonReview
    PERSON_REVIEW_LIST_RELATED = frozenset([
        PERSON.SELF,
        GLOBAL.ROBOT,
        DEPARTMENT.HEAD,
        DEPARTMENT.HR_ANALYST,
        DEPARTMENT.HR_PARTNER,
        REVIEW.ADMIN,
        REVIEW.SUPERREVIEWER,
        REVIEW.ACCOMPANYING_HR,
        PERSON_REVIEW.REVIEWER,
        PERSON_REVIEW.TOP_REVIEWER,
        PERSON_REVIEW.READER,
        PERSON_REVIEW.SUPERREADER,
        PERSON_REVIEW.READER_INHERITED,
        PERSON_REVIEW.SUPERREADER_INHERITED,
        PERSON_REVIEW.REVIEWER_INHERITED,
        PERSON_REVIEW.TOP_REVIEWER_INHERITED,
        PERSON_REVIEW.SUPERREVIEWER_INHERITED,
    ])

    CALIBRATION_PERSON_REVIEWS_ROLES_DEFAULT = {
        CALIBRATION.CALIBRATOR,
        CALIBRATION.ADMIN,
        PERSON_REVIEW.CALIBRATOR_ARCHIVED,
    }

    CALIBRATION_GLOBAL_ROLES = frozenset([
        GLOBAL.SUPPORT,
    ])

    VERBOSE = {}
    for scope in (
        GLOBAL,
        REVIEW,
        PERSON_REVIEW,
        CALIBRATION,
        PERSON,
        DEPARTMENT,
    ):
        for role, verbose in scope.VERBOSE.items():
            VERBOSE[role] = verbose

    ORDER_OF_IMPORTANCE = {
        PERSON_REVIEW.REVIEWER: 0,
        PERSON_REVIEW.TOP_REVIEWER: 0,
        REVIEW.SUPERREVIEWER: 1,
        PERSON_REVIEW.SUPERREVIEWER_DENORMALIZED: 1,
        CALIBRATION.CALIBRATOR: 2,
        PERSON_REVIEW.CALIBRATOR_DENORMALIZED: 2,
        CALIBRATION.ADMIN: 3,
        PERSON_REVIEW.READER: 4,
        PERSON_REVIEW.SUPERREADER: 4,
    }
    CUSTOM_IMPORTANCE_ROLES = set(chain(
        ORDER_OF_IMPORTANCE,
        (
            REVIEW.ADMIN,
            PERSON.SELF,
            DEPARTMENT.HR_ANALYST,
        ),
    ))
    importance_vals = count(max(ORDER_OF_IMPORTANCE.values()) + 1)
    for role in set(VERBOSE) - CUSTOM_IMPORTANCE_ROLES:
        ORDER_OF_IMPORTANCE[role] = next(importance_vals)
    ORDER_OF_IMPORTANCE[REVIEW.ADMIN] = next(importance_vals)
    ORDER_OF_IMPORTANCE[DEPARTMENT.HR_ANALYST] = next(importance_vals)
    ORDER_OF_IMPORTANCE[PERSON.SELF] = next(importance_vals)
    del importance_vals

    SCOPES_VERBOSE = {
        GLOBAL: 'Глобальный',
        REVIEW: 'Ревью',
        PERSON_REVIEW: 'Ревьюируемый',
        CALIBRATION: 'Калибровка',
        PERSON: 'Сотрудник',
        DEPARTMENT: 'Подразделение',
    }


class FETCHERS(object):

    PREFIX = 'fetch:'

    COLLECT = PREFIX + 'collect'
    PERSONS = PREFIX + 'persons'
    REVIEWS = PREFIX + 'reviews'
    MARKS_SCALES = PREFIX + 'marks_scales'
    PERSON_REVIEW_HISTORY = PREFIX + 'person_reviews_history'
    GOODIES = PREFIX + 'goodies'
    REVIEWERS = PREFIX + 'reviewers'
    SUBORDINATION = PREFIX + 'subordination'
    CHIEF = PREFIX + 'chief'
    FINANCE_EVENTS = PREFIX + 'finance_events'
    DEPARTMENT_CHAIN = PREFIX + 'department_chain'
    CURRENT_SALARIES = PREFIX + 'current_salaries'
    COMMENTS = PREFIX + 'comments'
    CHANGES = PREFIX + 'changes'
    TAGS = PREFIX + 'tags'
    UMBRELLA = PREFIX + 'umbrella'
    MAIN_PRODUCT = PREFIX + 'main_product'
    PRODUCT_SCHEMA_LOADED = PREFIX + 'product_schema_loaded'

    ALL = frozenset([
        COLLECT,
        PERSONS,
        REVIEWS,
        MARKS_SCALES,
        PERSON_REVIEW_HISTORY,
        FINANCE_EVENTS,
        GOODIES,
        REVIEWERS,
        SUBORDINATION,
        CHIEF,
        DEPARTMENT_CHAIN,
        CURRENT_SALARIES,
        COMMENTS,
        CHANGES,
        UMBRELLA,
        MAIN_PRODUCT,
        PRODUCT_SCHEMA_LOADED,
    ])


class FIELDS(object):
    """
    Константы, связанные с полями domain_obj.PersonReview
    """
    ID = 'id'
    STATUS = 'status'
    MARK = 'mark'
    GOLDSTAR = 'goldstar'
    LEVEL_CHANGE = 'level_change'
    SALARY_CHANGE = 'salary_change'
    SALARY_CHANGE_ABSOLUTE = 'salary_change_absolute'
    SALARY_CHANGE_TYPE = 'salary_change_type'
    SALARY_CHANGE_IS_DIFFER_FROM_DEFAULT = 'salary_change_is_differ_from_default'
    SALARY_AFTER_REVIEW = 'salary_after_review'
    BONUS = 'bonus'
    BONUS_ABSOLUTE = 'bonus_absolute'
    BONUS_TYPE = 'bonus_type'
    BONUS_IS_DIFFER_FROM_DEFAULT = 'bonus_is_differ_from_default'
    BONUS_RSU = 'bonus_rsu'
    DEFERRED_PAYMENT = 'deferred_payment'
    OPTIONS_RSU = 'options_rsu'
    OPTIONS_RSU_LEGACY = 'options_rsu_legacy'
    OPTIONS_RSU_IS_DIFFER_FROM_DEFAULT = 'options_rsu_is_differ_from_default'
    FLAGGED = 'flagged'
    FLAGGED_POSITIVE = 'flagged_positive'
    REVIEWERS = 'reviewers'
    ACTION_AT = 'action_at'
    APPROVE_LEVEL = 'approve_level'
    TAG_AVERAGE_MARK = 'tag_average_mark'
    TAKEN_IN_AVERAGE = 'taken_in_average'
    UPDATED_AT = 'updated_at'

    # данные сотрудника на текущий момент
    PERSON_ID = 'person_id'
    PERSON_LOGIN = 'person_login'
    PERSON_FIRST_NAME = 'person_first_name'
    PERSON_LAST_NAME = 'person_last_name'
    PERSON_GENDER = 'person_gender'
    PERSON_IS_DISMISSED = 'person_is_dismissed'
    PERSON_JOIN_AT = 'person_join_at'
    PERSON_POSITION = 'person_position'
    PERSON_DEPARTMENT_ID = 'person_department_id'
    PERSON_DEPARTMENT_SLUG = 'person_department_slug'
    PERSON_DEPARTMENT_NAME = 'person_department_name'
    PERSON_DEPARTMENT_PATH = 'person_department_path'
    PERSON_DEPARTMENT_CHAIN_SLUGS = 'person_department_chain_slugs'
    PERSON_DEPARTMENT_CHAIN_NAMES = 'person_department_chain_names'
    PERSON_CHIEF = 'person_chief'
    PERSON_CITY_NAME = 'person_city_name'

    # данные ревью
    REVIEW_ID = 'review_id'
    REVIEW_TYPE = 'review_type'
    REVIEW_NAME = 'review_name'
    REVIEW_STATUS = 'review_status'
    REVIEW_START_DATE = 'review_start_date'
    REVIEW_FINISH_DATE = 'review_finish_date'
    REVIEW_EVALUATION_FROM_DATE = 'review_evaluation_from_date'
    REVIEW_EVALUATION_TO_DATE = 'review_evaluation_to_date'
    REVIEW_FEEDBACK_FROM_DATE = 'review_feedback_from_date'
    REVIEW_FEEDBACK_TO_DATE = 'review_feedback_to_date'
    REVIEW_GOALS_FROM_DATE = 'review_goals_from_date'
    REVIEW_GOALS_TO_DATE = 'review_goals_to_date'
    REVIEW_OPTIONS_RSU_FORMAT = 'review_options_rsu_format'
    REVIEW_OPTIONS_RSU_UNIT = 'review_options_rsu_unit'
    REVIEW_SCALE_ID = 'review_scale_id'
    REVIEW_MARKS_SCALE = 'review_marks_scale'
    REVIEW_SALARY_DATE = 'review_salary_date'
    REVIEW_KPI_LOADED = 'review_kpi_loaded'

    # данные зонта и vs
    UMBRELLA_ID = 'umbrella_id'
    UMBRELLA = 'umbrella'
    MAIN_PRODUCT_ID = 'main_product_id'
    MAIN_PRODUCT = 'main_product'
    PRODUCT_SCHEMA_LOADED = 'product_schema_loaded'

    # оебс-данные сотрудника к началу ревью
    FTE = 'fte'
    LEVEL = 'level'
    PROFESSION = 'profession'
    SALARY_VALUE = 'salary_value'
    SALARY_CURRENCY = 'salary_currency'
    SHORT_HISTORY = 'short_history'
    MARK_LEVEL_HISTORY = 'mark_level_history'

    # искуственные поля, чтобы не парсить grade и salary history несколько раз
    GRADE_CLOSEST_TO_REVIEW_START = 'grade_closest_to_review_start'
    SALARY_CLOSEST_TO_REVIEW_START = 'salary_closest_to_review_start'

    # связанные сущности
    COMMENTS = 'comments'
    CHANGES = 'changes'

    # данные смотрящего
    SUBORDINATION = 'subordination'
    ROLES = 'roles'
    MODIFIERS = 'modifiers'
    PERMISSIONS_WRITE = 'permissions_write'
    PERMISSIONS_READ = 'permissions_read'
    ACTION_MARK = 'action_mark'
    ACTION_GOLDSTAR = 'action_goldstar'
    ACTION_LEVEL_CHANGE = 'action_level_change'
    ACTION_SALARY_CHANGE = 'action_salary_change'
    ACTION_SALARY_CHANGE_ABSOLUTE = 'action_salary_change_absolute'
    ACTION_SALARY_CHANGE_TYPE = 'action_salary_change_type'
    ACTION_BONUS = 'action_bonus'
    ACTION_BONUS_ABSOLUTE = 'action_bonus_absolute'
    ACTION_BONUS_TYPE = 'action_bonus_type'
    ACTION_BONUS_RSU = 'action_bonus_rsu'
    ACTION_DEFERRED_PAYMENT = 'action_deferred_payment'
    ACTION_OPTIONS_RSU = 'action_options_rsu'
    ACTION_FLAGGED = 'action_flagged'
    ACTION_FLAGGED_POSITIVE = 'action_flagged_positive'
    ACTION_APPROVE = 'action_approve'
    ACTION_UNAPPROVE = 'action_unapprove'
    ACTION_ALLOW_ANNOUNCE = 'action_allow_announce'
    ACTION_ANNOUNCE = 'action_announce'
    ACTION_COMMENT = 'action_comment'
    ACTION_REVIEWERS = 'action_reviewers'
    ACTION_TAG_AVERAGE_MARK = 'action_tag_average_mark'
    ACTION_UMBRELLA = 'action_umbrella'
    ACTION_MAIN_PRODUCT = 'action_main_product'
    ACTION_TAKEN_IN_AVERAGE = 'action_taken_in_average'

    # внешние сущности
    GOALS_URL = 'goals_url'
    ST_GOALS_URL = 'st_goals_url'

    ALL = frozenset([
        ID,
        STATUS,
        MARK,
        GOLDSTAR,
        LEVEL_CHANGE,
        SALARY_CHANGE,
        SALARY_CHANGE_ABSOLUTE,
        SALARY_CHANGE_TYPE,
        SALARY_AFTER_REVIEW,
        BONUS,
        BONUS_ABSOLUTE,
        BONUS_TYPE,
        BONUS_RSU,
        DEFERRED_PAYMENT,
        OPTIONS_RSU,
        OPTIONS_RSU_LEGACY,
        FLAGGED,
        FLAGGED_POSITIVE,
        TAG_AVERAGE_MARK,
        TAKEN_IN_AVERAGE,
        UPDATED_AT,
        REVIEWERS,
        SALARY_CHANGE_IS_DIFFER_FROM_DEFAULT,
        BONUS_IS_DIFFER_FROM_DEFAULT,
        OPTIONS_RSU_IS_DIFFER_FROM_DEFAULT,
        ACTION_AT,
        APPROVE_LEVEL,
        PERSON_ID,
        PERSON_LOGIN,
        PERSON_FIRST_NAME,
        PERSON_LAST_NAME,
        PERSON_IS_DISMISSED,
        PERSON_JOIN_AT,
        PERSON_POSITION,
        PERSON_DEPARTMENT_ID,
        PERSON_DEPARTMENT_SLUG,
        PERSON_DEPARTMENT_NAME,
        PERSON_DEPARTMENT_PATH,
        PERSON_DEPARTMENT_CHAIN_SLUGS,
        PERSON_DEPARTMENT_CHAIN_NAMES,
        PERSON_CHIEF,
        PERSON_GENDER,
        PERSON_CITY_NAME,
        REVIEW_ID,
        REVIEW_NAME,
        REVIEW_TYPE,
        REVIEW_SCALE_ID,
        REVIEW_MARKS_SCALE,
        REVIEW_SALARY_DATE,
        REVIEW_STATUS,
        REVIEW_START_DATE,
        REVIEW_FINISH_DATE,
        REVIEW_EVALUATION_FROM_DATE,
        REVIEW_EVALUATION_TO_DATE,
        REVIEW_FEEDBACK_FROM_DATE,
        REVIEW_FEEDBACK_TO_DATE,
        REVIEW_GOALS_FROM_DATE,
        REVIEW_GOALS_TO_DATE,
        REVIEW_OPTIONS_RSU_FORMAT,
        REVIEW_OPTIONS_RSU_UNIT,
        REVIEW_KPI_LOADED,
        FTE,
        LEVEL,
        PROFESSION,
        SALARY_VALUE,
        SALARY_CURRENCY,
        SHORT_HISTORY,
        MARK_LEVEL_HISTORY,
        COMMENTS,
        CHANGES,
        SUBORDINATION,
        ROLES,
        MODIFIERS,
        PERMISSIONS_WRITE,
        PERMISSIONS_READ,
        ACTION_MARK,
        ACTION_GOLDSTAR,
        ACTION_LEVEL_CHANGE,
        ACTION_SALARY_CHANGE,
        ACTION_SALARY_CHANGE_ABSOLUTE,
        ACTION_SALARY_CHANGE_TYPE,
        ACTION_BONUS,
        ACTION_BONUS_ABSOLUTE,
        ACTION_BONUS_TYPE,
        ACTION_BONUS_RSU,
        ACTION_DEFERRED_PAYMENT,
        ACTION_OPTIONS_RSU,
        ACTION_FLAGGED,
        ACTION_FLAGGED_POSITIVE,
        ACTION_APPROVE,
        ACTION_UNAPPROVE,
        ACTION_ALLOW_ANNOUNCE,
        ACTION_ANNOUNCE,
        ACTION_COMMENT,
        ACTION_REVIEWERS,
        ACTION_TAG_AVERAGE_MARK,
        ACTION_TAKEN_IN_AVERAGE,
        GOALS_URL,
        ST_GOALS_URL,
        UMBRELLA,
        UMBRELLA_ID,
        MAIN_PRODUCT,
        MAIN_PRODUCT_ID,
        ACTION_UMBRELLA,
        ACTION_MAIN_PRODUCT,
        PRODUCT_SCHEMA_LOADED,
    ])
    INTERNAL_FIELDS = frozenset([
        GRADE_CLOSEST_TO_REVIEW_START,
        SALARY_CLOSEST_TO_REVIEW_START,
    ])
    MINIMUM = frozenset([
        ID,
        PERSON_ID,
        REVIEW_ID,
    ])
    FROM_COLLECT_STAGE = frozenset([
        STATUS,
        MARK,
        GOLDSTAR,
        LEVEL_CHANGE,
        SALARY_CHANGE,
        SALARY_CHANGE_ABSOLUTE,
        SALARY_CHANGE_TYPE,
        BONUS,
        BONUS_ABSOLUTE,
        BONUS_TYPE,
        BONUS_RSU,
        DEFERRED_PAYMENT,
        OPTIONS_RSU,
        OPTIONS_RSU_LEGACY,
        FLAGGED,
        FLAGGED_POSITIVE,
        APPROVE_LEVEL,
        TAG_AVERAGE_MARK,
        TAKEN_IN_AVERAGE,
        UPDATED_AT,
        UMBRELLA_ID,
        MAIN_PRODUCT_ID,
    ]) | MINIMUM
    DEFAULT_FOR_MODE_REVIEW = MINIMUM | frozenset([
        STATUS,
        MARK,
        GOLDSTAR,
        LEVEL_CHANGE,
        SALARY_CHANGE,
        SALARY_CHANGE_ABSOLUTE,
        SALARY_CHANGE_TYPE,
        SALARY_AFTER_REVIEW,
        BONUS,
        BONUS_ABSOLUTE,
        BONUS_TYPE,
        BONUS_RSU,
        DEFERRED_PAYMENT,
        OPTIONS_RSU,
        OPTIONS_RSU_LEGACY,
        FLAGGED,
        FLAGGED_POSITIVE,
        APPROVE_LEVEL,
        REVIEWERS,
        PERSON_LOGIN,
        PERSON_GENDER,
        PERSON_FIRST_NAME,
        PERSON_LAST_NAME,
        PERSON_IS_DISMISSED,
        PERSON_CITY_NAME,
        PERSON_JOIN_AT,
        PERSON_CHIEF,
        PERSON_POSITION,
        PERSON_DEPARTMENT_SLUG,
        PERSON_DEPARTMENT_NAME,
        PERSON_DEPARTMENT_CHAIN_SLUGS,
        PERSON_DEPARTMENT_CHAIN_NAMES,
        PROFESSION,
        REVIEW_ID,
        REVIEW_NAME,
        REVIEW_TYPE,
        REVIEW_START_DATE,
        REVIEW_OPTIONS_RSU_FORMAT,
        REVIEW_OPTIONS_RSU_UNIT,
        REVIEW_GOALS_FROM_DATE,
        REVIEW_GOALS_TO_DATE,
        REVIEW_FEEDBACK_FROM_DATE,
        REVIEW_FEEDBACK_TO_DATE,
        REVIEW_SCALE_ID,
        REVIEW_KPI_LOADED,
        LEVEL,
        SALARY_VALUE,
        SALARY_CURRENCY,
        GOALS_URL,
        ST_GOALS_URL,
        MARK_LEVEL_HISTORY,
        ACTION_AT,
        TAG_AVERAGE_MARK,
        TAKEN_IN_AVERAGE,
        UMBRELLA,
        MAIN_PRODUCT,
        PRODUCT_SCHEMA_LOADED,
    ])
    DEFAULT_FOR_MODE_CALIBRATION = MINIMUM | frozenset([
        PERSON_LOGIN,
        PERSON_GENDER,
        PERSON_FIRST_NAME,
        PERSON_LAST_NAME,
        PERSON_IS_DISMISSED,
        PERSON_CITY_NAME,
        PERSON_JOIN_AT,
        PERSON_CHIEF,
        PERSON_POSITION,
        PERSON_DEPARTMENT_SLUG,
        PERSON_DEPARTMENT_NAME,
        PERSON_DEPARTMENT_CHAIN_SLUGS,
        PERSON_DEPARTMENT_CHAIN_NAMES,
        REVIEW_ID,
        REVIEW_NAME,
        REVIEW_TYPE,
        REVIEW_START_DATE,
        REVIEW_OPTIONS_RSU_FORMAT,
        REVIEW_OPTIONS_RSU_UNIT,
        REVIEW_GOALS_FROM_DATE,
        REVIEW_GOALS_TO_DATE,
        REVIEW_FEEDBACK_FROM_DATE,
        REVIEW_FEEDBACK_TO_DATE,
        REVIEW_SCALE_ID,
        REVIEW_KPI_LOADED,
        STATUS,
        ACTION_AT,
        APPROVE_LEVEL,
        REVIEWERS,
        LEVEL,
        LEVEL_CHANGE,
        SALARY_CHANGE,
        BONUS,
        BONUS_ABSOLUTE,
        BONUS_TYPE,
        BONUS_RSU,
        DEFERRED_PAYMENT,
        OPTIONS_RSU,
        OPTIONS_RSU_LEGACY,
        MARK,
        GOLDSTAR,
        SALARY_VALUE,
        SALARY_CURRENCY,
        SALARY_AFTER_REVIEW,
        FLAGGED,
        FLAGGED_POSITIVE,
        GOALS_URL,
        ST_GOALS_URL,
        MARK_LEVEL_HISTORY,
        TAKEN_IN_AVERAGE,
        ACTION_MARK,
        ACTION_GOLDSTAR,
        ACTION_LEVEL_CHANGE,
        ACTION_FLAGGED,
        ACTION_FLAGGED_POSITIVE,
        ACTION_COMMENT,
        ACTION_UMBRELLA,
        ACTION_MAIN_PRODUCT,
        UMBRELLA,
        MAIN_PRODUCT,
        PRODUCT_SCHEMA_LOADED,
        TAG_AVERAGE_MARK,
    ])
    DEFAULT_FOR_MODE_CALIBRATION_EDIT = DEFAULT_FOR_MODE_CALIBRATION | frozenset([
        PROFESSION,
        PERSON_DEPARTMENT_NAME,
    ])
    DEFAULT_FOR_REVIEWS_MODE_HISTORY = MINIMUM | frozenset([
        STATUS,
        MARK,
        GOLDSTAR,
        LEVEL_CHANGE,
        SALARY_CHANGE,
        SALARY_CHANGE_ABSOLUTE,
        SALARY_CHANGE_TYPE,
        SALARY_AFTER_REVIEW,
        BONUS,
        BONUS_ABSOLUTE,
        BONUS_TYPE,
        BONUS_RSU,
        DEFERRED_PAYMENT,
        OPTIONS_RSU,
        OPTIONS_RSU_LEGACY,
        FLAGGED,
        FLAGGED_POSITIVE,
        APPROVE_LEVEL,
        PERSON_LOGIN,
        PERSON_GENDER,
        PERSON_POSITION,
        PERSON_FIRST_NAME,
        PERSON_LAST_NAME,
        PERSON_IS_DISMISSED,
        PERSON_JOIN_AT,
        PERSON_CITY_NAME,
        PERSON_DEPARTMENT_NAME,
        PERSON_DEPARTMENT_SLUG,
        PERSON_DEPARTMENT_CHAIN_NAMES,
        PROFESSION,
        PERSON_CHIEF,
        REVIEW_ID,
        REVIEW_NAME,
        REVIEW_START_DATE,
        REVIEW_SALARY_DATE,
        REVIEW_FEEDBACK_FROM_DATE,
        REVIEW_FEEDBACK_TO_DATE,
        REVIEW_GOALS_FROM_DATE,
        REVIEW_GOALS_TO_DATE,
        REVIEW_TYPE,
        REVIEW_SCALE_ID,
        REVIEW_OPTIONS_RSU_FORMAT,
        REVIEW_OPTIONS_RSU_UNIT,
        REVIEW_KPI_LOADED,
        LEVEL,
        SALARY_VALUE,
        SALARY_CURRENCY,
        REVIEWERS,
        ACTION_MARK,
        ACTION_GOLDSTAR,
        ACTION_LEVEL_CHANGE,
        ACTION_SALARY_CHANGE,
        ACTION_SALARY_CHANGE_ABSOLUTE,
        ACTION_SALARY_CHANGE_TYPE,
        ACTION_BONUS,
        ACTION_BONUS_ABSOLUTE,
        ACTION_BONUS_TYPE,
        ACTION_BONUS_RSU,
        ACTION_DEFERRED_PAYMENT,
        ACTION_OPTIONS_RSU,
        ACTION_FLAGGED,
        ACTION_FLAGGED_POSITIVE,
        ACTION_APPROVE,
        ACTION_UNAPPROVE,
        ACTION_ALLOW_ANNOUNCE,
        ACTION_ANNOUNCE,
        ACTION_COMMENT,
        ACTION_AT,  # for actions which is stupid
        ACTION_UMBRELLA,
        ACTION_MAIN_PRODUCT,
        GOALS_URL,
        ST_GOALS_URL,
        TAG_AVERAGE_MARK,
        TAKEN_IN_AVERAGE,
        UMBRELLA,
        MAIN_PRODUCT,
        PRODUCT_SCHEMA_LOADED,
    ])
    EDIT_ACTION_FIELDS = frozenset([
        ACTION_MARK,
        ACTION_GOLDSTAR,
        ACTION_LEVEL_CHANGE,
        ACTION_SALARY_CHANGE,
        ACTION_SALARY_CHANGE_ABSOLUTE,
        ACTION_SALARY_CHANGE_TYPE,
        ACTION_BONUS,
        ACTION_BONUS_ABSOLUTE,
        ACTION_BONUS_TYPE,
        ACTION_BONUS_RSU,
        ACTION_DEFERRED_PAYMENT,
        ACTION_OPTIONS_RSU,
        ACTION_FLAGGED,
        ACTION_FLAGGED_POSITIVE,
        ACTION_COMMENT,
        ACTION_REVIEWERS,
        ACTION_TAG_AVERAGE_MARK,
        ACTION_TAKEN_IN_AVERAGE,
        ACTION_UMBRELLA,
        ACTION_MAIN_PRODUCT,
    ])
    WORKFLOW_ACTION_FIELDS = frozenset([
        ACTION_APPROVE,
        ACTION_UNAPPROVE,
        ACTION_ALLOW_ANNOUNCE,
        ACTION_ANNOUNCE,
    ])
    ALL_ACTION_FIELDS = EDIT_ACTION_FIELDS | WORKFLOW_ACTION_FIELDS
    FOR_DOING_ACTION = MINIMUM | ALL_ACTION_FIELDS | frozenset([
        STATUS,
        LEVEL,
        MARK,
        GOLDSTAR,
        LEVEL_CHANGE,
        SALARY_CHANGE,
        SALARY_CHANGE_ABSOLUTE,
        SALARY_CHANGE_TYPE,
        BONUS,
        BONUS_ABSOLUTE,
        BONUS_TYPE,
        BONUS_RSU,
        DEFERRED_PAYMENT,
        OPTIONS_RSU,
        PERSON_LOGIN,
        APPROVE_LEVEL,
        MODIFIERS,
        FLAGGED,
        FLAGGED_POSITIVE,
        REVIEW_MARKS_SCALE,
        REVIEWERS,
        SALARY_VALUE,
        TAG_AVERAGE_MARK,
        TAKEN_IN_AVERAGE,
        UMBRELLA,
        UMBRELLA_ID,
        MAIN_PRODUCT,
        MAIN_PRODUCT_ID,
    ])

    FOR_ROBOT = frozenset([
        STATUS,
        MARK,
        APPROVE_LEVEL,
        REVIEWERS,
    ]) | ALL_ACTION_FIELDS | MINIMUM

    FLAG_FIELDS = frozenset([
        FLAGGED,
        FLAGGED_POSITIVE,
    ])

    GROUPS = {
        '_all_': ALL,
        '_minimum_': MINIMUM,
        '_mode_review_': DEFAULT_FOR_MODE_REVIEW,
        '_mode_calibration_': DEFAULT_FOR_MODE_CALIBRATION,
        '_mode_history_': DEFAULT_FOR_REVIEWS_MODE_HISTORY,
    }

    CHOICES = [(x, x) for x in ALL | set(GROUPS)]

    DB_PERSON_FIELDS_MAP = {
        PERSON_ID: 'id',
        PERSON_LOGIN: 'login',
        PERSON_FIRST_NAME: 'first_name',
        PERSON_LAST_NAME: 'last_name',
        PERSON_IS_DISMISSED: 'is_dismissed',
        PERSON_JOIN_AT: 'join_at',
        PERSON_POSITION: 'position',
        PERSON_DEPARTMENT_ID: 'department__id',
        PERSON_DEPARTMENT_SLUG: 'department__slug',
        PERSON_DEPARTMENT_PATH: 'department__path',
        PERSON_DEPARTMENT_NAME: 'department__name',
        PERSON_GENDER: 'gender',
        PERSON_CITY_NAME: 'city_name'
    }

    DB_REVIEW_FIELDS_MAP = {
        REVIEW_ID: 'id',
        REVIEW_TYPE: 'type',
        REVIEW_NAME: 'name',
        REVIEW_STATUS: 'status',
        REVIEW_START_DATE: 'start_date',
        REVIEW_FINISH_DATE: 'finish_date',
        REVIEW_FEEDBACK_FROM_DATE: 'feedback_from_date',
        REVIEW_FEEDBACK_TO_DATE: 'feedback_to_date',
        REVIEW_EVALUATION_FROM_DATE: 'evaluation_from_date',
        REVIEW_EVALUATION_TO_DATE: 'evaluation_to_date',
        REVIEW_OPTIONS_RSU_FORMAT: 'options_rsu_format',
        REVIEW_OPTIONS_RSU_UNIT: 'options_rsu_unit',
        REVIEW_SCALE_ID: 'scale_id',
        REVIEW_SALARY_DATE: 'salary_date',
        REVIEW_KPI_LOADED: 'kpi_loaded',
    }

    DB_REVIEW_FIELDS = {
        'id',
        'name',
        'type',
        'status',
        'finish_date',
        'start_date',
        'include_dismissed_after_date',
        'product_schema_loaded',
    }

    DB_REVIEW_WITH_DATES = {
        'evaluation_from_date',
        'evaluation_to_date',
        'feedback_from_date',
        'feedback_to_date',
        'finish_approval_date',
        'finish_calibration_date',
        'finish_feedback_date',
        'finish_submission_date',
        'salary_date',
        'product_schema_loaded',
        'kpi_loaded',
    } | DB_REVIEW_FIELDS

    DB_REVIEW_WITH_PARAMETERS = {
        'bonus_type',
        'bonus_reason',
        'scale_id',
        'bonus_mode',
        'goldstar_mode',
        'mark_mode',
        'options_rsu_mode',
        'options_rsu_unit',
        'deferred_payment_mode',
        'salary_change_mode',
        'level_change_mode',
        'notify_reminder_days',
        'notify_reminder_date_from',
        'notify_reminder_date_to',
        'notify_events_other',
        'notify_events_superreviewer',
    } | DB_REVIEW_FIELDS

    DB_REVIEW_ROLES_FIELDS = {
        'admins',
        'super_reviewers',
        'accompanying_hrs',
    }

    DB_REVIEW_ALL_FIELDS = DB_REVIEW_ROLES_FIELDS | DB_REVIEW_WITH_DATES | DB_REVIEW_WITH_PARAMETERS

    DB_CALIBRATION_FIELDS = {
        'id',
        'name',
        'status',
        'finish_date',
        'start_date'
    }

    DB_UMBRELLA_FIELDS = {
        'id',
        'issue_key',
        'main_product_id',
        'name',
    }

    DB_MAIN_PRODUCT_FIELDS = {
        'id',
        'issue_key',
        'abc_service_id',
        'name',
    }

    PERSON_REVIEW_IMPORT_FIELDS = frozenset([
        MARK,
        GOLDSTAR,
        LEVEL_CHANGE,
        SALARY_CHANGE,
        SALARY_CHANGE_ABSOLUTE,
        SALARY_CHANGE_TYPE,
        BONUS,
        BONUS_ABSOLUTE,
        BONUS_TYPE,
        BONUS_RSU,
        DEFERRED_PAYMENT,
        OPTIONS_RSU,
        TAG_AVERAGE_MARK,
        TAKEN_IN_AVERAGE,
        UMBRELLA,
        MAIN_PRODUCT,
    ])

    INPUT_FIELDS = frozenset([
        MARK,
        GOLDSTAR,
        LEVEL_CHANGE,
    ])
    GOODIE_FIELDS = frozenset([
        SALARY_CHANGE,
        SALARY_CHANGE_ABSOLUTE,
        BONUS,
        BONUS_ABSOLUTE,
        OPTIONS_RSU,
    ])

    DEFAULT_GOODIE_FIELDS = frozenset([
        ID,
        LEVEL,
        MARK,
        GOLDSTAR,
        LEVEL_CHANGE,
        SALARY_CHANGE,
        BONUS,
        OPTIONS_RSU,
    ])

    DEFAULT_REVIEWER_FIELDS = frozenset([
        'person_review_id',
        'position',
        'login',
        'is_dismissed',
        'first_name',
        'last_name',
        'gender',
    ])

    DEFAULT_REVIEW_ROLE_PERSON_FIELDS = frozenset([
        'login',
        'last_name',
        'first_name',
        'position',
        'is_dismissed',
        'department_name',
        'department_slug',
        'gender',
    ])

    DEFAULT_DEPARTMENT_CHAIN_FIELDS = frozenset([
        'id',
        'slug',
        'name',
    ])
    BONUS_ACTIONS = frozenset([
        ACTION_BONUS,
        ACTION_BONUS_ABSOLUTE,
    ])
    SALARY_ACTIONS = frozenset([
        ACTION_SALARY_CHANGE,
        ACTION_SALARY_CHANGE_ABSOLUTE,
    ])


MARK_LEVEL_HISTORY_RANGE = 3


class ERROR_CODES(object):
    const_creator = ConstCreator()
    cc = const_creator.create_const

    ACTION_UNAVAILABLE = cc('ACTION_UNAVAILABLE')
    SHOULD_NOT_BEFORE_MARK = cc('SHOULD_NOT_BEFORE_MARK')
    REVIEW_WORKFLOW_EMPTY_CHAINS = cc('REVIEW_WORKFLOW_EMPTY_CHAINS')
    DELETING_LAST_REVIEWER = cc('DELETING_LAST_REVIEWER')
    PERMISSION_DENIED = cc('PERMISSION_DENIED')
    ALREADY_EXISTS = cc('ALREADY_EXISTS')
    PERSONS_ARE_CALIBRATING = cc('PERSONS_ARE_CALIBRATING')
    PERSONS_ARE_CALIBRATORS = cc('PERSONS_ARE_CALIBRATORS')
    PERSONS_ARE_CALIBRATION_ADMINS = cc('PERSONS_ARE_CALIBRATION_ADMINS')
    CHANGE_REVIEWERS_BELOW_APPROVE_LEVEL = cc('CHANGE_REVIEWERS_BELOW_APPROVE_LEVEL')
    MARK_NOT_FROM_SCALE = cc('MARK_NOT_FROM_SCALE')

    del cc, const_creator


class FILTERS(object):

    PREFIX = 'filter:'

    IDS = PREFIX + 'ids'
    REVIEWS = PREFIX + 'reviews'
    SCALE = PREFIX + 'scale'
    CALIBRATION = PREFIX + 'calibration_id'
    REVIEW_ACTIVITY = PREFIX + 'review_activity'
    REVIEW_STATUSES = PREFIX + 'review_statuses'
    PERSONS = PREFIX + 'persons'
    CALIBRATIONS = PREFIX + 'calibrations'
    MARK = PREFIX + 'mark'
    GOLDSTAR = PREFIX + 'goldstar'
    LEVEL_CHANGE = PREFIX + 'level_change'
    LEVEL_CHANGED = PREFIX + 'level_changed'
    SALARY_CHANGE = PREFIX + 'salary_change'
    BONUS = PREFIX + 'bonus'
    OPTIONS_RSU = PREFIX + 'options_rsu'
    DEPARTMENTS = PREFIX + 'departments'
    PROFESSION = PREFIX + 'profession'
    LEVEL = PREFIX + 'level'
    SALARY_VALUE = PREFIX + 'salary_value'
    SALARY_CURRENCY = PREFIX + 'salary_currency'
    PERSON_JOIN_AT = PREFIX + 'person_join_at'
    STATUS = PREFIX + 'status'
    FLAGGED = PREFIX + 'flagged'
    FLAGGED_POSITIVE = PREFIX + 'flagged_positive'
    TAG_AVERAGE_MARK = PREFIX + 'tag_average_mark'
    TAKEN_IN_AVERAGE = PREFIX + 'taken_in_average'
    ACTION_AT = PREFIX + 'action_at'
    # CIA-833 not used now
    IS_DIFFER_FROM_DEFAULT = PREFIX + 'is_differ_from_default'
    SUBORDINATION = PREFIX + 'subordination'
    REVIEWER = PREFIX + 'reviewer'
    MAIN_PRODUCT = PREFIX + 'main_product'
    MAIN_PRODUCT_ISNULL = PREFIX + 'main_product__isnull'
    UMBRELLA = PREFIX + 'umbrella'
    UMBRELLA_ISNULL = PREFIX + 'umbrella__isnull'
    PRODUCT_SCHEMA_LOADED = PREFIX + 'product_schema_loaded'

    ALL = frozenset([
        IDS,
        REVIEWS,
        SCALE,
        CALIBRATIONS,
        REVIEW_ACTIVITY,
        REVIEW_STATUSES,
        PERSONS,
        MARK,
        GOLDSTAR,
        LEVEL_CHANGE,
        LEVEL_CHANGED,
        SALARY_CHANGE,
        BONUS,
        OPTIONS_RSU,
        DEPARTMENTS,
        PROFESSION,
        LEVEL,
        SALARY_VALUE,
        SALARY_CURRENCY,
        PERSON_JOIN_AT,
        STATUS,
        FLAGGED,
        FLAGGED_POSITIVE,
        ACTION_AT,
        IS_DIFFER_FROM_DEFAULT,
        SUBORDINATION,
        REVIEWER,
        TAG_AVERAGE_MARK,
        TAKEN_IN_AVERAGE,
        MAIN_PRODUCT,
        MAIN_PRODUCT_ISNULL,
        UMBRELLA,
        UMBRELLA_ISNULL,
        PRODUCT_SCHEMA_LOADED,
    ])

    # фильтры, применимые на этапе collect
    FOR_COLLECT = frozenset([
        IDS,
        REVIEWS,
        REVIEW_ACTIVITY,
        REVIEW_STATUSES,
        PERSONS,
        SCALE,
        TAG_AVERAGE_MARK,
        TAKEN_IN_AVERAGE,
        CALIBRATIONS,
        MAIN_PRODUCT,
        MAIN_PRODUCT_ISNULL,
        UMBRELLA,
        UMBRELLA_ISNULL,
    ])

    DEFAULT_FOR_MODE_REVIEW = {
        REVIEW_ACTIVITY: True,
    }

    REVIEW_FILTERS = frozenset([
        'ids',
        'activity',
        'statuses',
    ])

    CALIBRATION_FILTERS = frozenset([
        'ids',
        'statuses',
        'discuss',
    ])

    DEFAULT_FOR_MODE_CALIBRATION = {}


class CALIBRATION_PERSON_REVIEW_ACTIONS(object):
    DISCUSS = 'discuss'
    ALL = frozenset([
        DISCUSS
    ])


class CALIBRATION_PERSON_REVIEW_PERMS(object):
    DISCUSS = 'discuss'

    ROLE_PERMS = {
        ROLE.CALIBRATION.CALIBRATOR: frozenset({
            CALIBRATION_PERSON_REVIEW_ACTIONS.DISCUSS,
        }),
    }


class REVIEW_PERMS(object):
    const_creator = ConstCreator()
    cc = const_creator.create_const

    ADD_PERSONS = cc('add_persons')
    DELETE_PERSONS = cc('delete_persons')
    ADD_ANY_PERSONS = cc('add_any_persons')
    DELETE_ANY_PERSONS = cc('delete_any_persons')
    EDIT_ADMINS = cc('edit_admins')
    EDIT_SUPERREVIEWERS = cc('edit_superreviewers')
    EDIT_ACCOMPANYING_HRS = cc('edit_accompanying_hrs')
    EDIT = cc('edit')
    LIST_ADMINS = cc('list_admins')
    LIST_GOODIES = cc('list_goodies')
    LIST_SUPERREVIEWERS = cc('list_super_reviewers')
    LIST_ACCOMPANYING_HRS = cc('list_accompanying_hrs')
    STATUS_ARCHIVE = cc('archive')
    STATUS_UNARCHIVE = cc('unarchive')
    STATUS_IN_DRAFT = cc('in_draft')
    STATUS_PUBLISH = cc('publish')
    LOAD_PRODUCT_SCHEME = cc('load_product_scheme')
    VIEW_PARTICIPANTS = cc('view_participants')
    LOAD_SECRET_TEMPLATE = cc('load_secret_template')
    LOAD_KPI = cc('load_kpi')

    ALL = const_creator.get_all()
    del cc, const_creator

    ROLE_PERMS = defaultdict(frozenset)
    ROLE_PERMS.update({
        ROLE.REVIEW.ADMIN: frozenset({
            ADD_ANY_PERSONS,
            DELETE_ANY_PERSONS,
            EDIT,
            EDIT_ADMINS,
            EDIT_SUPERREVIEWERS,
            EDIT_ACCOMPANYING_HRS,
            LIST_ADMINS,
            LIST_GOODIES,
            LIST_SUPERREVIEWERS,
            LIST_ACCOMPANYING_HRS,
            STATUS_PUBLISH,
            STATUS_IN_DRAFT,
            STATUS_ARCHIVE,
            STATUS_UNARCHIVE,
            LOAD_KPI,
        }),
        ROLE.REVIEW.SUPERREVIEWER: frozenset({
            DELETE_ANY_PERSONS,
            LIST_ADMINS,
            LIST_SUPERREVIEWERS,
            LIST_ACCOMPANYING_HRS,
            VIEW_PARTICIPANTS,
        }),
        ROLE.REVIEW.ACCOMPANYING_HR: frozenset({
            ADD_PERSONS,
            DELETE_PERSONS,
            EDIT_ACCOMPANYING_HRS,
            EDIT_SUPERREVIEWERS,
            EDIT,
            LIST_ADMINS,
            LIST_SUPERREVIEWERS,
            LIST_ACCOMPANYING_HRS,
            STATUS_PUBLISH,
            STATUS_IN_DRAFT,
            STATUS_ARCHIVE,
            LOAD_SECRET_TEMPLATE,
        }),
    })


class REVIEW_ACTIONS(object):
    const_creator = ConstCreator()
    cc = const_creator.create_const

    ADD_PERSONS = cc('add_persons')
    DELETE_PERSONS = cc('delete_persons')
    ADD_ANY_PERSONS = cc('add_any_persons')
    DELETE_ANY_PERSONS = cc('delete_any_persons')
    EDIT = cc('edit')
    EDIT_ADMINS = cc('edit_admins')
    EDIT_SUPERREVIEWERS = cc('edit_superreviewers')
    EDIT_ACCOMPANYING_HRS = cc('edit_accompanying_hrs')
    LIST_GOODIES = cc('list_goodies')
    STATUS_ARCHIVE = cc('archive')
    STATUS_UNARCHIVE = cc('unarchive')
    STATUS_IN_DRAFT = cc('in_draft')
    STATUS_PUBLISH = cc('publish')
    LOAD_PRODUCT_SCHEME = cc('load_product_scheme')
    VIEW_PARTICIPANTS = cc('view_participants')
    LOAD_SECRET_TEMPLATE = cc('load_secret_template')
    LOAD_KPI = cc('load_kpi')

    ALL = const_creator.get_all()
    del const_creator, cc

    STATUS = frozenset([
        STATUS_ARCHIVE,
        STATUS_IN_DRAFT,
        STATUS_PUBLISH,
        STATUS_UNARCHIVE,
    ])

    STATUS_TO_TRANSITIONS = {
        REVIEW_STATUS.DRAFT: (STATUS_PUBLISH, ),
        REVIEW_STATUS.IN_PROGRESS: (STATUS_ARCHIVE, STATUS_IN_DRAFT),
        REVIEW_STATUS.FINISHED: (STATUS_ARCHIVE, ),
        REVIEW_STATUS.ARCHIVE: (STATUS_UNARCHIVE, ),
    }

    EDIT_TO_ROLE = {
        EDIT_ADMINS: ROLE.REVIEW.ADMIN,
        EDIT_SUPERREVIEWERS: ROLE.REVIEW.SUPERREVIEWER,
        EDIT_ACCOMPANYING_HRS: ROLE.REVIEW.ACCOMPANYING_HR,
    }


class PERSON_REVIEW_PERMS(object):

    MARK_READ = 'mark_read'
    GOLDSTAR_READ = 'goldstar_read'
    STATUS_READ = 'status_read'
    MARK_LEVEL_HISTORY_READ = 'mark_level_history_read'
    APPROVE_LEVEL_READ = 'approve_level_read'
    LEVEL_CHANGE_READ = 'level_change_read'
    SALARY_CHANGE_READ = 'salary_change_read'
    BONUS_READ = 'bonus_read'
    DEFERRED_PAYMENT_READ = 'deferred_payment_read'
    OPTIONS_RSU_READ = 'options_rsu_read'
    FLAGGED_READ = 'flagged_read'
    COMMENTS_READ = 'comments_read'
    CHANGES_READ = 'changes_read'
    FTE_READ = 'fte_read'
    SALARY_READ = 'salary_read'
    LEVEL_READ = 'level_read'
    REVIEWERS_READ = 'reviewers_read'
    TAG_AVERAGE_MARK_READ = 'tag_average_mark_read'
    UPDATED_AT_READ = 'updated_at_read'
    TAKEN_IN_AVERAGE_READ = 'taken_in_average_read'
    ACTION_AT_READ = 'action_at_read'
    MAIN_PRODUCT_READ = 'main_product_read'
    UMBRELLA_READ = 'umbrella_read'

    READ = frozenset([
        MARK_READ,
        GOLDSTAR_READ,
        STATUS_READ,
        MARK_LEVEL_HISTORY_READ,
        APPROVE_LEVEL_READ,
        LEVEL_CHANGE_READ,
        SALARY_CHANGE_READ,
        BONUS_READ,
        DEFERRED_PAYMENT_READ,
        OPTIONS_RSU_READ,
        FLAGGED_READ,
        REVIEWERS_READ,
        FTE_READ,
        SALARY_READ,
        SALARY_READ,
        LEVEL_READ,
        COMMENTS_READ,
        CHANGES_READ,
        TAG_AVERAGE_MARK_READ,
        UPDATED_AT_READ,
        TAKEN_IN_AVERAGE_READ,
        ACTION_AT_READ,
        MAIN_PRODUCT_READ,
        UMBRELLA_READ,
    ])

    MARK_WRITE = 'mark_write'
    GOLDSTAR_WRITE = 'goldstar_write'
    GOLDSTAR_WRITE_CHOSEN = 'goldstar_write_chosen'
    LEVEL_CHANGE_WRITE = 'level_change_write'
    SALARY_CHANGE_WRITE = 'salary_change_write'
    BONUS_WRITE = 'bonus_write'
    DEFERRED_PAYMENT_WRITE = 'deferred_payment_write'
    OPTIONS_RSU_WRITE = 'options_rsu_write'
    SALARY_CHANGE_AUTO_WRITE = 'salary_change_auto_write'
    BONUS_AUTO_WRITE = 'bonus_auto_write'
    DEFERRED_PAYMENT_AUTO_WRITE = 'deferred_payment_auto_write'
    OPTIONS_RSU_AUTO_WRITE = 'options_rsu_auto_write'
    FLAGGED_WRITE = 'flagged_write'
    APPROVE = 'approve'
    UNAPPROVE = 'unapprove'
    ALLOW_ANNOUNCE = 'allow_announce'
    ANNOUNCE = 'announce'
    COMMENTS_WRITE = 'comments_write'
    REVIEWERS_WRITE = 'reviewers_write'
    TAG_AVERAGE_MARK_WRITE = 'tag_average_mark_write'
    UMBRELLA_WRITE = 'umbrella_write'
    MAIN_PRODUCT_WRITE = 'main_product_write'
    TAKEN_IN_AVERAGE_WRITE = 'taken_in_average_write'

    WRITE = frozenset([
        MARK_WRITE,
        GOLDSTAR_WRITE_CHOSEN,
        GOLDSTAR_WRITE,
        LEVEL_CHANGE_WRITE,
        SALARY_CHANGE_WRITE,
        BONUS_WRITE,
        DEFERRED_PAYMENT_WRITE,
        DEFERRED_PAYMENT_AUTO_WRITE,
        OPTIONS_RSU_WRITE,
        SALARY_CHANGE_AUTO_WRITE,
        BONUS_AUTO_WRITE,
        OPTIONS_RSU_AUTO_WRITE,
        FLAGGED_WRITE,
        REVIEWERS_WRITE,
        COMMENTS_WRITE,
        APPROVE,
        UNAPPROVE,
        ALLOW_ANNOUNCE,
        ANNOUNCE,
        TAG_AVERAGE_MARK_WRITE,
        UMBRELLA_WRITE,
        MAIN_PRODUCT_WRITE,
        TAKEN_IN_AVERAGE_WRITE,
    ])

    ALL = READ | WRITE

    FIELD_TO_PERMISSION = {
        FIELDS.MARK: MARK_READ,
        FIELDS.GOLDSTAR: GOLDSTAR_READ,
        FIELDS.STATUS: STATUS_READ,
        FIELDS.MARK_LEVEL_HISTORY: MARK_LEVEL_HISTORY_READ,
        FIELDS.APPROVE_LEVEL: APPROVE_LEVEL_READ,
        FIELDS.LEVEL_CHANGE: LEVEL_CHANGE_READ,
        FIELDS.SALARY_CHANGE: SALARY_CHANGE_READ,
        FIELDS.SALARY_CHANGE_ABSOLUTE: SALARY_CHANGE_READ,
        FIELDS.SALARY_CHANGE_TYPE: SALARY_CHANGE_READ,
        FIELDS.SALARY_AFTER_REVIEW: SALARY_CHANGE_READ,
        FIELDS.BONUS: BONUS_READ,
        FIELDS.BONUS_ABSOLUTE: BONUS_READ,
        FIELDS.BONUS_TYPE: BONUS_READ,
        FIELDS.BONUS_RSU: BONUS_READ,
        FIELDS.DEFERRED_PAYMENT: DEFERRED_PAYMENT_READ,
        FIELDS.OPTIONS_RSU: OPTIONS_RSU_READ,
        FIELDS.OPTIONS_RSU_LEGACY: OPTIONS_RSU_READ,
        FIELDS.FLAGGED: FLAGGED_READ,
        FIELDS.FLAGGED_POSITIVE: FLAGGED_READ,
        FIELDS.REVIEWERS: REVIEWERS_READ,
        FIELDS.SALARY_VALUE: SALARY_READ,
        FIELDS.SALARY_CURRENCY: SALARY_READ,
        FIELDS.LEVEL: LEVEL_READ,
        FIELDS.FTE: FTE_READ,
        FIELDS.TAG_AVERAGE_MARK: TAG_AVERAGE_MARK_READ,
        FIELDS.UPDATED_AT: UPDATED_AT_READ,
        FIELDS.TAKEN_IN_AVERAGE: TAKEN_IN_AVERAGE_READ,
        FIELDS.ACTION_AT: ACTION_AT_READ,
        FIELDS.ACTION_MARK: MARK_WRITE,
        FIELDS.ACTION_GOLDSTAR: GOLDSTAR_WRITE,
        FIELDS.ACTION_LEVEL_CHANGE: LEVEL_CHANGE_WRITE,
        FIELDS.ACTION_SALARY_CHANGE: SALARY_CHANGE_WRITE,
        FIELDS.ACTION_SALARY_CHANGE_ABSOLUTE: SALARY_CHANGE_WRITE,
        FIELDS.ACTION_SALARY_CHANGE_TYPE: SALARY_CHANGE_WRITE,
        FIELDS.ACTION_BONUS: BONUS_WRITE,
        FIELDS.ACTION_BONUS_ABSOLUTE: BONUS_WRITE,
        FIELDS.ACTION_BONUS_TYPE: BONUS_WRITE,
        FIELDS.ACTION_BONUS_RSU: BONUS_WRITE,
        FIELDS.ACTION_DEFERRED_PAYMENT: DEFERRED_PAYMENT_WRITE,
        FIELDS.ACTION_OPTIONS_RSU: OPTIONS_RSU_WRITE,
        FIELDS.ACTION_FLAGGED: FLAGGED_WRITE,
        FIELDS.ACTION_FLAGGED_POSITIVE: FLAGGED_WRITE,
        FIELDS.ACTION_REVIEWERS: REVIEWERS_WRITE,
        FIELDS.ACTION_TAG_AVERAGE_MARK: TAG_AVERAGE_MARK_WRITE,
        FIELDS.ACTION_TAKEN_IN_AVERAGE: TAKEN_IN_AVERAGE_WRITE,
        FIELDS.ACTION_COMMENT: COMMENTS_WRITE,
        FIELDS.ACTION_APPROVE: APPROVE,
        FIELDS.ACTION_UNAPPROVE: UNAPPROVE,
        FIELDS.ACTION_ALLOW_ANNOUNCE: ALLOW_ANNOUNCE,
        FIELDS.ACTION_ANNOUNCE: ANNOUNCE,
        FIELDS.ACTION_UMBRELLA: UMBRELLA_WRITE,
        FIELDS.ACTION_MAIN_PRODUCT: MAIN_PRODUCT_WRITE,
    }

    HISTORY_RESTRICTED_ROLES = frozenset([
        ROLE.PERSON_REVIEW.CALIBRATOR_INHERITED,
        ROLE.PERSON_REVIEW.READER_INHERITED,
    ])


class PERSON_REVIEW_ACTIONS(object):
    MARK = 'mark'
    GOLDSTAR = 'goldstar'
    LEVEL_CHANGE = 'level_change'
    SALARY_CHANGE = 'salary_change'
    SALARY_CHANGE_ABSOLUTE = 'salary_change_absolute'
    SALARY_CHANGE_TYPE = 'salary_change_type'
    BONUS = 'bonus'
    BONUS_ABSOLUTE = 'bonus_absolute'
    BONUS_TYPE = 'bonus_type'
    BONUS_RSU = 'bonus_rsu'
    DEFERRED_PAYMENT = 'deferred_payment'
    OPTIONS_RSU = 'options_rsu'
    FLAGGED = 'flagged'
    FLAGGED_POSITIVE = 'flagged_positive'
    APPROVE = 'approve'
    UNAPPROVE = 'unapprove'
    ALLOW_ANNOUNCE = 'allow_announce'
    ANNOUNCE = 'announce'
    COMMENT = 'comment'
    REVIEWERS = 'reviewers'
    TAG_AVERAGE_MARK = 'tag_average_mark'
    UMBRELLA = 'umbrella'
    MAIN_PRODUCT = 'main_product'
    TAKEN_IN_AVERAGE = 'taken_in_average'

    # https://wiki.yandex-team.ru/staff/pool/CIA/Revju-2.0/flow/
    TRANSITIONS_MARKS_ENABLED = {
        PERSON_REVIEW_STATUS.EVALUATION: {APPROVE},
        PERSON_REVIEW_STATUS.APPROVAL: {APPROVE, UNAPPROVE},
        PERSON_REVIEW_STATUS.APPROVED: {ALLOW_ANNOUNCE, UNAPPROVE},
        PERSON_REVIEW_STATUS.WAIT_ANNOUNCE: {ANNOUNCE},
    }
    TRANSITIONS_MARKS_DISABLED = {
        PERSON_REVIEW_STATUS.WAIT_EVALUATION: {APPROVE},
        PERSON_REVIEW_STATUS.EVALUATION: {APPROVE},
        PERSON_REVIEW_STATUS.APPROVAL: {APPROVE, UNAPPROVE},
        PERSON_REVIEW_STATUS.APPROVED: {ALLOW_ANNOUNCE, UNAPPROVE},
        PERSON_REVIEW_STATUS.WAIT_ANNOUNCE: {ANNOUNCE},
    }

    ALL = frozenset([
        MARK,
        GOLDSTAR,
        LEVEL_CHANGE,
        SALARY_CHANGE,
        SALARY_CHANGE_ABSOLUTE,
        BONUS,
        BONUS_ABSOLUTE,
        BONUS_RSU,
        DEFERRED_PAYMENT,
        OPTIONS_RSU,
        FLAGGED,
        FLAGGED_POSITIVE,
        APPROVE,
        UNAPPROVE,
        ALLOW_ANNOUNCE,
        ANNOUNCE,
        COMMENT,
        REVIEWERS,
        TAG_AVERAGE_MARK,
        TAKEN_IN_AVERAGE,
        UMBRELLA,
    ])

    EDIT = 'edit'
    BONUS_EDIT = 'bonus_edit'
    SALARY_EDIT = 'salary_edit'
    FLAGS = 'flags'
    WORKFLOW_GROUP = 'workflow'
    MODE_REVIEW = 'mode_review'
    MODE_REVIEW_EDIT = 'mode_review_edit'
    GROUPS = {
        EDIT: frozenset([
            MARK,
            GOLDSTAR,
            LEVEL_CHANGE,
            BONUS_RSU,
            DEFERRED_PAYMENT,
            OPTIONS_RSU,
            TAG_AVERAGE_MARK,
            TAKEN_IN_AVERAGE,
        ]),
        SALARY_EDIT: frozenset([
            SALARY_CHANGE,
            SALARY_CHANGE_ABSOLUTE,
        ]),
        BONUS_EDIT: frozenset([
            BONUS,
            BONUS_ABSOLUTE,
        ]),
        WORKFLOW_GROUP: frozenset([
            APPROVE,
            UNAPPROVE,
            ALLOW_ANNOUNCE,
            ANNOUNCE,
        ]),
        FLAGS: frozenset([
            FLAGGED,
            FLAGGED_POSITIVE,
        ]),
        MODE_REVIEW: ALL - {REVIEWERS},
        MODE_REVIEW_EDIT: {REVIEWERS},
    }

    EXECUTE_ORDER = [
        BONUS_ABSOLUTE,
        SALARY_CHANGE_ABSOLUTE,
    ]  # prioritized actions

    AFFECTED_FIELDS = {
        MARK: FIELDS.MARK,
        GOLDSTAR: FIELDS.GOLDSTAR,
        LEVEL_CHANGE: FIELDS.LEVEL_CHANGE,
        SALARY_CHANGE: FIELDS.SALARY_CHANGE,
        SALARY_CHANGE_ABSOLUTE: FIELDS.SALARY_CHANGE_ABSOLUTE,
        BONUS: FIELDS.BONUS,
        BONUS_ABSOLUTE: FIELDS.BONUS_ABSOLUTE,
        BONUS_RSU: FIELDS.BONUS_RSU,
        DEFERRED_PAYMENT: FIELDS.DEFERRED_PAYMENT,
        OPTIONS_RSU: FIELDS.OPTIONS_RSU,
        FLAGGED: FIELDS.FLAGGED,
        FLAGGED_POSITIVE: FIELDS.FLAGGED_POSITIVE,
        APPROVE: FIELDS.STATUS,
        UNAPPROVE: FIELDS.STATUS,
        ALLOW_ANNOUNCE: FIELDS.STATUS,
        ANNOUNCE: FIELDS.STATUS,
        REVIEWERS: FIELDS.REVIEWERS,
        TAG_AVERAGE_MARK: FIELDS.TAG_AVERAGE_MARK,
        UMBRELLA: FIELDS.UMBRELLA,
        TAKEN_IN_AVERAGE: FIELDS.TAKEN_IN_AVERAGE,
    }
    TO_ACTION_FIELDS = {
        MARK: FIELDS.ACTION_MARK,
        GOLDSTAR: FIELDS.ACTION_GOLDSTAR,
        LEVEL_CHANGE: FIELDS.ACTION_LEVEL_CHANGE,
        SALARY_CHANGE: FIELDS.ACTION_SALARY_CHANGE,
        SALARY_CHANGE_ABSOLUTE: FIELDS.ACTION_SALARY_CHANGE_ABSOLUTE,
        SALARY_CHANGE_TYPE: FIELDS.ACTION_SALARY_CHANGE_TYPE,
        BONUS: FIELDS.ACTION_BONUS,
        BONUS_ABSOLUTE: FIELDS.ACTION_BONUS_ABSOLUTE,
        BONUS_TYPE: FIELDS.ACTION_BONUS_TYPE,
        BONUS_RSU: FIELDS.ACTION_BONUS_RSU,
        DEFERRED_PAYMENT: FIELDS.ACTION_DEFERRED_PAYMENT,
        OPTIONS_RSU: FIELDS.ACTION_OPTIONS_RSU,
        FLAGGED: FIELDS.ACTION_FLAGGED,
        FLAGGED_POSITIVE: FIELDS.ACTION_FLAGGED,
        APPROVE: FIELDS.ACTION_APPROVE,
        UNAPPROVE: FIELDS.ACTION_UNAPPROVE,
        ALLOW_ANNOUNCE: FIELDS.ACTION_ALLOW_ANNOUNCE,
        ANNOUNCE: FIELDS.ACTION_ANNOUNCE,
        REVIEWERS: FIELDS.ACTION_REVIEWERS,
        COMMENT: FIELDS.ACTION_COMMENT,
        TAG_AVERAGE_MARK: FIELDS.ACTION_TAG_AVERAGE_MARK,
        UMBRELLA: FIELDS.ACTION_UMBRELLA,
        MAIN_PRODUCT: FIELDS.ACTION_MAIN_PRODUCT,
        TAKEN_IN_AVERAGE: FIELDS.ACTION_TAKEN_IN_AVERAGE,
    }
    TO_ACTION_FIELDS_REVERSED = {
        action_field: field
        for field, action_field in TO_ACTION_FIELDS.items()
    }

    REVIEW_CHAIN_ADD = 'add'
    REVIEW_CHAIN_REPLACE = 'replace'
    REVIEW_CHAIN_DELETE = 'delete'
    REVIEW_CHAIN_REPLACE_ALL = 'replace_all'

    ADD_POSITION_START = 'start'
    ADD_POSITION_END = 'end'
    ADD_POSITION_BEFORE = 'before'
    ADD_POSITION_AFTER = 'after'
    ADD_POSITION_SAME = 'same'

    ADD_POSITION_CHOICE = frozenset([
        ADD_POSITION_START,
        ADD_POSITION_END,
        ADD_POSITION_BEFORE,
        ADD_POSITION_AFTER,
        ADD_POSITION_SAME
    ])


class PERSON_REVIEW_COMMENT_ACTIONS(object):

    EDIT = 'edit'
    ALL = frozenset([
        EDIT
    ])


WEEK_DAYS = ['mo', 'tu', 'we', 'th', 'fr', 'sa', 'su']


class REVIEW_NOTIFICATION_SETTINGS(object):
    NO = 'no'
    ONCE = 'once'
    PERIOD = 'period'
    ALL = frozenset([
        NO,
        ONCE,
        PERIOD,
    ])


class REVIEW_MODE(object):
    MODE_DISABLED = 'd'
    MODE_AUTO = 'a'
    MODE_MANUAL = 'm'
    MODE_MANUAL_BY_CHOSEN = 'c'

    VERBOSE = {
        MODE_DISABLED: 'disabled',
        MODE_AUTO: 'auto',
        MODE_MANUAL: 'manual',
        MODE_MANUAL_BY_CHOSEN: 'chosen',
    }
    ALL = frozenset(VERBOSE)

    CHOICES_FOR_MARK = []

    for key, value in VERBOSE.items():
        if key in {
            MODE_DISABLED,
            MODE_MANUAL
        }:
            CHOICES_FOR_MARK.append((key, value))

    CHOICES_FOR_GOLDSTAR = []

    for key, value in VERBOSE.items():
        if key in {
            MODE_DISABLED,
            MODE_MANUAL,
            MODE_MANUAL_BY_CHOSEN,
        }:
            CHOICES_FOR_GOLDSTAR.append((key, value))

    CHOICES_FOR_LEVEL_CHANGE = CHOICES_FOR_MARK

    CHOICES_FOR_GOODIE = []

    for key, value in VERBOSE.items():
        if key in {
            MODE_DISABLED,
            MODE_AUTO,
            MODE_MANUAL,
        }:
            CHOICES_FOR_GOODIE.append((key, value))

    FORM_MARK_MODE_DISABLED = 'disabled'
    FORM_MARK_MODE_GOLDSTAR_DISABLED = 'goldstar_disabled'
    FORM_MARK_MODE_GOLDSTAR_ENABLED = 'goldstar_enabled'
    FORM_MARK_MODE_GOLDSTAR_RESTRICTED = 'goldstar_restricted'
    FORM_CHOICE_FOR_MARK_MODE = frozenset((
        FORM_MARK_MODE_DISABLED,
        FORM_MARK_MODE_GOLDSTAR_DISABLED,
        FORM_MARK_MODE_GOLDSTAR_ENABLED,
        FORM_MARK_MODE_GOLDSTAR_RESTRICTED,
    ))

    MARK_MODE = 'mark_mode'
    GOLDSTAR_MODE = 'goldstar_mode'
    LEVEL_CHANGE_MODE = 'level_change_mode'
    SALARY_CHANGE_MODE = 'salary_change_mode'
    BONUS_MODE = 'bonus_mode'
    OPTIONS_RSU_MODE = 'options_rsu_mode'
    DEFERRED_PAYMENT_MODE = 'deferred_payment_mode'
    FIELDS_TO_MODIFIERS = {
        FIELDS.MARK: MARK_MODE,
        FIELDS.GOLDSTAR: GOLDSTAR_MODE,
        FIELDS.LEVEL_CHANGE: LEVEL_CHANGE_MODE,
        FIELDS.SALARY_CHANGE: SALARY_CHANGE_MODE,
        FIELDS.SALARY_CHANGE_ABSOLUTE: SALARY_CHANGE_MODE,
        FIELDS.BONUS: BONUS_MODE,
        FIELDS.BONUS_ABSOLUTE: BONUS_MODE,
        FIELDS.BONUS_RSU: BONUS_MODE,
        FIELDS.DEFERRED_PAYMENT: DEFERRED_PAYMENT_MODE,
        FIELDS.OPTIONS_RSU: OPTIONS_RSU_MODE,
        FIELDS.OPTIONS_RSU_LEGACY: OPTIONS_RSU_MODE,
    }
    MODIFIERS_FOR_AUTO_GOODIES = {
        value: key for key, value in FIELDS_TO_MODIFIERS.items()
        if key not in (
            FIELDS.OPTIONS_RSU_LEGACY,
            FIELDS.BONUS_RSU,
            FIELDS.BONUS_ABSOLUTE,
            FIELDS.SALARY_CHANGE_ABSOLUTE,
        )
    }
    MODIFIERS_INPUT = frozenset([
        MARK_MODE,
        GOLDSTAR_MODE,
        LEVEL_CHANGE_MODE,
    ])
    MODIFIERS_OUTPUT = frozenset([
        SALARY_CHANGE_MODE,
        BONUS_MODE,
        OPTIONS_RSU_MODE,
        DEFERRED_PAYMENT_MODE,
    ])
    MODIFIERS_ALL = MODIFIERS_INPUT | MODIFIERS_OUTPUT

    MODIFIER_TO_SPECIAL_VALUE = {
        MODE_DISABLED: DISABLED,
        MODE_AUTO: AUTO,
    }

    DEFAULTS = {
        FIELDS.MARK: MARK.DEFAULT,
        FIELDS.GOLDSTAR: GOLDSTAR.DEFAULT,
        FIELDS.LEVEL_CHANGE: 0
    }


class VALIDATION:
    LEVEL_CHANGE_MIN = -100
    LEVEL_CHANGE_MAX = 100

    SALARY_CHANGE_MIN = -1000
    SALARY_CHANGE_MAX = 1000

    SALARY_MIN = 0
    SALARY_MAX = 10 ** 9

    SALARY_CHANGE_ABSOLUTE_MIN = -100000000
    SALARY_CHANGE_ABSOLUTE_MAX = 100000000

    BONUS_MIN = 0
    BONUS_MAX = 1000

    BONUS_ABSOLUTE_MIN = 0
    BONUS_ABSOLUTE_MAX = 100000000

    BONUS_RSU_MIN = 0
    BONUS_RSU_MAX = 10000000

    DEFERRED_PAYMENT_MIN = 0
    DEFERRED_PAYMENT_MAX = 10000000000

    OPTIONS_RSU_MIN = 0
    OPTIONS_RSU_MAX = 10000000


class EXCEL_TEMPLATE_TYPE:
    SECRET = 's'
    PUBLIC = 'p'

    VERBOSE = {
        SECRET: 'secret',
        PUBLIC: 'public',
    }

    CHOICES = VERBOSE.items()
    REVERSE_VERBOSE = {val: key for key, val in VERBOSE.items()}
