# coding=utf-8
from django import forms
from django.core.exceptions import ValidationError
from django.core import paginator

from django_idm_api import hooks
from django_idm_api import exceptions

from review.core import const as core_const
from review.core import models as core_models
from review.staff import models as staff_models
from review.lib import forms as lib_forms


class GlobalRolesHook(hooks.BaseHooks):
    def info_impl(self, **kwargs):
        return {
            core_const.ROLE.GLOBAL.VERBOSE[core_const.ROLE.GLOBAL.REVIEW_CREATOR]: 'Review creator',
            core_const.ROLE.GLOBAL.VERBOSE[core_const.ROLE.GLOBAL.CALIBRATION_CREATOR]: 'Calibration creator',
            core_const.ROLE.GLOBAL.VERBOSE[core_const.ROLE.GLOBAL.SUPPORT]: 'Support',
            core_const.ROLE.GLOBAL.VERBOSE[core_const.ROLE.GLOBAL.EXPORTER]: 'Exporter',
        }

    def add_role_impl(self, login, role, fields, **kwargs):
        role = role.get('role')
        valid_form = GlobalRoleValidationForm(data={'role': role, 'person': login})
        if not valid_form.is_valid():
            raise ValidationError(valid_form.errors)
        cleaned = valid_form.clean()
        core_models.GlobalRole.objects.create(
            person=cleaned['person'],
            type=cleaned['role'],
        )

    def remove_role_impl(self, login, role, data, is_fired, **kwargs):
        role = role.get('role')
        valid_form = GlobalRoleValidationForm(data={'role': role, 'person': login})
        if not valid_form.is_valid():
            raise ValidationError(valid_form.errors)
        cleaned = valid_form.clean()
        person = cleaned['person']
        role_type = cleaned['role']
        role = core_models.GlobalRole.objects.filter(person=person, type=role_type).first()
        if role is None:
            raise exceptions.RoleNotFound('Person {} has no such role {}'.format(person.login, role_type))
        role.delete()

    def get_user_roles_impl(self, login, **kwargs):
        roles = core_models.GlobalRole.objects.filter(
            person__login=login
        ).values('type')
        return [core_const.ROLE.GLOBAL.VERBOSE[r['type']] for r in roles]

    def get_all_roles_impl(self, **kwargs):
        roles = core_models.GlobalRole.objects.values('person__login', 'type')
        grouped_by_person = {}
        for role in roles:
            person = grouped_by_person.setdefault(role['person__login'], [])
            person.append(core_const.ROLE.GLOBAL.VERBOSE[role['type']])
        return list(grouped_by_person.items())

    def get_roles(self, request):
        queryset = core_models.GlobalRole.objects.values('person__login', 'type').order_by('id')
        page_num = int(request.GET.get('page_num', '1'))
        per_page = int(request.GET.get('per_page', '100'))
        query_paginator = paginator.Paginator(queryset, per_page=per_page)
        try:
            page = query_paginator.page(page_num)
        except paginator.EmptyPage:
            page = query_paginator.page(query_paginator.num_pages)
        roles = [{'login': it['person__login'], 'role': core_const.ROLE.GLOBAL.VERBOSE[it['type']]} for it in page]
        result = {
            'code': 0,
            'roles': roles,
        }
        if page.has_next():
            result['next-url'] = '{}?page_num={}&per_page={}'.format(
                request.path,
                page.next_page_number(),
                per_page,
            )
        return result


class GlobalRoleValidationForm(forms.Form):
    person = forms.ModelChoiceField(
        queryset=staff_models.Person.objects.all(),
        to_field_name="login",
        required=True
    )
    role = lib_forms.VerboseChoiceField(
        choices=core_const.ROLE.GLOBAL.VERBOSE,
        required=True,
    )
