# coding: utf-8
from collections import defaultdict
from typing import Iterable, Dict, List, Set

from review.core import (
    const as core_const,
    models as core_models,
)
from review.core.logic import (
    helpers as core_helpers,
    calibration_rights,
)
from review.core.logic import (
    domain_objs,
)
from review.lib import (
    errors,
    helpers,
)
from review.staff import logic as staff_logic
from review.staff import models as staff_models
from review.staff import const as staff_const

from . import (
    collect_calibrations,
    collect_calibration_person_reviews,
    collect_goodies,
    collect_person_reviews,
    collect_person_review_changes,
    collect_person_review_comments,
    collect_reviews,
    deps,
    fetch_reviews,
    fetch_person_reviews,
    fetched_obj,
    filters,
    permissions_review,
    setters_person_review,
)
from review.core.logic.domain_objs import (
    Calibration,
    CalibrationPersonReview,
)


@helpers.timeit_no_args_logging
def get_person_reviews(
    subject,
    fields_requested,
    filters_chosen=None,
    role_types=None,
):
    filters_chosen = filters_chosen or {}
    subject = staff_logic.ensure_person_model(subject)
    extra_params = {
        'subject': subject,
    }
    if core_const.FILTERS.SUBORDINATION in filters_chosen:
        subordination_subject_login = filters_chosen.pop('subordination_subject', subject.login)
        extra_params['subordination_subject_login'] = subordination_subject_login
    elif 'subordination_subject' in filters_chosen:
        filters_chosen.setdefault(core_const.FILTERS.SUBORDINATION,
                                  staff_const.SUBORDINATION.DIRECT)
        extra_params['subordination_subject_login'] = filters_chosen.pop('subordination_subject')
    steps = deps.get_steps(fields=fields_requested, filters=filters_chosen)
    fetched = fetched_obj.Fetched()
    person_reviews = collect_person_reviews.get_allowed_person_reviews_flat(
        subject=subject,
        fetched=fetched,
        role_types=role_types,
        filters=steps.collect_step_filters,
        db_fields=steps.collect_step_params.get('db_fields', ()),
    )

    for step in steps.non_collect:
        if step.is_filter:
            filters.apply(
                person_reviews=person_reviews,
                filters={step.id: step.value}
            )
        elif step.is_field:
            setters_person_review.set_field(
                field=step.id,
                person_reviews=person_reviews,
                fetched=fetched,
                role_types=role_types,
                **extra_params
            )
        elif step.is_fetcher:
            fetcher_params = dict(extra_params)
            fetcher_params.update(step.params)
            fetch_person_reviews.fetch(
                person_reviews=person_reviews,
                fetcher=step.id,
                fetched=fetched,
                **fetcher_params
            )
    return person_reviews


def get_person_review(subject, fields_requested, id, role_types=None):
    person_reviews = list(get_person_reviews(
        subject=subject,
        fields_requested=fields_requested,
        filters_chosen={core_const.FILTERS.IDS: [id]},
        role_types=role_types,
    ))
    return person_reviews and person_reviews[0] or None


def get_person_review_ids(subject, filters, role_types=None) -> Set[int]:
    return collect_person_reviews.get_allowed_person_review_ids(
        subject=subject,
        filters=filters,
        role_types=role_types,
    )


def get_reviews(subject, filters, requested_fields=None):
    chosen_filters = core_helpers.select_review_filters(filters)
    reviews = collect_reviews.get_allowed_reviews_flat(
        subject=subject,
        filters=chosen_filters,
        requested_fields=requested_fields,
    )
    fetch_reviews.fetch_review_roles(reviews, requested_fields, language=subject.language)
    for review in reviews:
        permissions_review.set_permissions(review)
        permissions_review.set_actions(review)
        permissions_review.clear_unavailable_fields(review)
    return list(reviews)


def get_review(subject, filters, requested_fields=None):
    reviews = get_reviews(
        subject=subject,
        filters=filters,
        requested_fields=requested_fields,
    )
    return reviews and reviews[0] or None


def _get_file_from_template(template):
    return template.file


def get_review_template_file(subject, review_id, template_type):
    review = get_review(subject, {'ids': [review_id]})
    if not review:
        return None

    # тут можно как-то пробросить наверх 403, но сейчас этого не делаем
    not_available = (
        template_type == core_const.EXCEL_TEMPLATE_TYPE.SECRET and
        review.actions[core_const.REVIEW_ACTIONS.LOAD_SECRET_TEMPLATE] != core_const.OK
    )
    if not_available:
        return None

    template_model = core_models.ExcelTemplate.objects.filter(
        review_id=review.id,
        template_type=template_type,
    ).first()
    if not template_model:
        return None

    return _get_file_from_template(template_model)


def get_reviews_goodies(subject, review_ids, requested_fields=None):
    return collect_goodies.get_goodies(
        subject,
        review_ids,
        requested_fields,
    )


def get_review_goodies(subject, review_id, requested_fields=None):
    return get_reviews_goodies(
        subject,
        [review_id],
        requested_fields
    )[review_id]


def get_person_review_comments(subject, filters, person_reviews=None, role_types=None):
    comments = collect_person_review_comments.get_comments(
        subject=subject,
        filters=filters,
        person_reviews=person_reviews,
        role_types=role_types,
    )
    return comments


def get_person_review_comment(subject, id, person_review=None, role_types=None):
    comments = get_person_review_comments(
        subject=subject,
        filters={'id': id},
        person_reviews=[person_review] if person_review else None,
        role_types=role_types,
    )
    return comments.get(id)


def get_person_review_changes(subject, filters, person_reviews=None):
    changes = collect_person_review_changes.get_changes(
        subject=subject,
        filters=filters,
        person_reviews=person_reviews,
    )
    return changes


def get_calibrations(subject, filters, requested_fields=None, role_types=None):
    chosen_filters = core_helpers.select_calibration_filters(filters)
    calibrations = collect_calibrations.get_allowed_calibrations_flat(
        subject=subject,
        filters=chosen_filters,
        role_types=role_types,
        requested_fields=requested_fields,
    )
    for calibration in calibrations:
        calibration.permissions = list(calibration_rights.get_permissions(calibration))
        calibration.actions = calibration_rights.get_actions_state(calibration)
    return list(calibrations)


def get_calibration(subject, id, requested_fields=None):
    calibrations = get_calibrations(
        subject=subject,
        filters={'ids': [id]},
        requested_fields=requested_fields,
    )
    res = calibrations and calibrations[0] or None
    if not res:
        if not core_models.Calibration.objects.filter(id=id).exists():
            raise errors.Error(
                msg_tpl='Calibration {id} does not exist',
                id=id,
                code=404,
            )
        else:
            raise errors.PermissionDenied(id=id)
    return res


def get_calibrations_review_info(calibration_list, filters):
    # type: (List[Calibration], Dict) -> Dict[int, List[Dict]]
    ids = [c.id for c in calibration_list]
    review_info = (
        core_models.Review.objects
        .filter(person_review__calibration_person_review__calibration_id__in=ids)
        .values_list(
            'id',
            'person_review__calibration_person_review__calibration_id',
            'name',
        )
        .distinct('person_review__calibration_person_review__calibration_id', 'id')
    )
    if filters.get('reviews'):
        review_info = review_info.filter(id__in=filters['reviews'])
    review_activity = filters.get('review_activity')
    if review_activity is not None:
        if review_activity:
            review_info = review_info.filter(status__in=core_const.REVIEW_STATUS.ACTIVE)
        else:
            review_info = review_info.filter(status__in=core_const.REVIEW_STATUS.INACTIVE)

    calibration_id_to_reviews = defaultdict(list)
    for review_id, calibration_id, review_name in review_info:
        calibration_id_to_reviews[calibration_id].append(
            {'id': review_id, 'name': review_name}
        )
    return calibration_id_to_reviews


def get_calibration_person_reviews(
    subject,
    filters_chosen,
    requested_person_review_fields,
):
    # type: (staff_models.Person, Dict, Iterable) -> List[CalibrationPersonReview]
    cpr_filters_keys = {
        'ids',
        'calibration_id',
        'discuss',
        'person_review_ids',
        'calibration_status',
        'persons',
    }
    cpr_filters, pr_filters = {}, {}
    for key, value in filters_chosen.items():
        filters_dict = cpr_filters if key in cpr_filters_keys else pr_filters
        filters_dict[key] = value

    calibration_person_reviews = collect_calibration_person_reviews.get_all(subject, filters=cpr_filters)
    if not calibration_person_reviews:
        return []
    pr_filters[core_const.FILTERS.IDS] = {it['person_review_id'] for it in calibration_person_reviews}

    person_reviews = get_person_reviews(
        subject=subject,
        fields_requested=set(requested_person_review_fields) | {
            core_const.FIELDS.PERMISSIONS_WRITE,
            core_const.FIELDS.PERMISSIONS_READ,
        },
        filters_chosen=pr_filters,
        role_types=core_const.ROLE.CALIBRATION_PERSON_REVIEWS_ROLES_DEFAULT
    )
    id_to_person_review = {pr.id: pr for pr in person_reviews}

    def get_actions(pr):
        is_admin = core_const.ROLE.CALIBRATION.ADMIN in pr.roles
        state = core_const.OK if is_admin else core_const.NO_ACCESS
        return {core_const.CALIBRATION_PERSON_REVIEW_ACTIONS.DISCUSS: state}

    res = []
    for cpr in calibration_person_reviews:
        person_review = id_to_person_review.get(cpr['person_review_id'])
        if person_review:
            res.append(domain_objs.CalibrationPersonReview(
                id=cpr['id'],
                calibration_id=cpr['calibration_id'],
                person_review=person_review,
                discuss=cpr['discuss'],
                roles=set(person_review.roles) & core_const.ROLE.CALIBRATION.ALL,
                actions=get_actions(person_review),
                updated_at=cpr['updated_at_auto'],
            ))
    return res


def get_calibration_person_review(subject, id, requested_person_review_fields=None):
    calibration_person_reviews = get_calibration_person_reviews(
        subject=subject,
        filters_chosen={'ids': [id]},
        requested_person_review_fields=requested_person_review_fields,
    )
    res = calibration_person_reviews and calibration_person_reviews[0] or None
    if not res:
        if not core_models.CalibrationPersonReview.objects.filter(id=id).exists():
            raise errors.Error(
                msg_tpl='CalibrationPersonReview {id} does not exist',
                id=id,
                code=404,
            )
        else:
            raise errors.PermissionDenied(id=id)
    return res


def get_marks_scales():
    return dict(core_models.MarksScale.objects.values_list('id', 'scale'))
