from django.db.models import Q, F

from review.core import const
from review.core import models
from review.core.logic.roles import has_global_roles
from review.staff.logic import ensure_person_model

from review.core.logic import domain_objs


def get_allowed_calibrations_flat(*args, **kwargs):
    grouped = get_allowed_calibrations_grouped(*args, **kwargs)
    flat = set()
    calibration_to_roles = {}
    for role_type, calibration_group in grouped.items():
        for calibration in calibration_group:
            calibration_to_roles.setdefault(calibration, [])
            calibration_to_roles[calibration].append(role_type)

    for calibration, roles in calibration_to_roles.items():
        calibration.roles = list(set(roles))
        flat.add(calibration)
    return flat


def get_allowed_calibrations_grouped(subject, filters=None, role_types=None, requested_fields=None):
    role_types = role_types and set(role_types) or const.ROLE.CALIBRATION.ALL
    filters = filters or {}
    requested_fields = requested_fields or const.FIELDS.DB_CALIBRATION_FIELDS
    grouped_by_role = {role: [] for role in role_types}
    calibration_roles = role_types & const.ROLE.CALIBRATION.ALL
    grouped_by_role.update(
        _get_calibration_for_calibration_roles(
            subject=subject,
            filters=filters,
            roles=calibration_roles,
            fields=requested_fields,
        )
    )
    return grouped_by_role


def _get_calibration_for_calibration_roles(subject, filters, roles, fields):
    subject = ensure_person_model(subject)
    filter_q = Q()
    requested_no_admin_roles = roles - {const.ROLE.CALIBRATION.ADMIN}
    if const.ROLE.CALIBRATION.ADMIN in roles:
        filter_q |= Q(
            _prepare_calibrations_filter_q(filters, allow_drafts=True),
            role__type=const.ROLE.CALIBRATION.ADMIN,
            role__person=subject,
        )
    if requested_no_admin_roles:
        filter_q |= Q(
            _prepare_calibrations_filter_q(filters, allow_drafts=False),
            role__type__in=requested_no_admin_roles,
            role__person=subject,

        )
    fields = {'role__type'} | set(fields)
    calibrations = _fetch_calibrations(filter_q, subject.language, fields)

    grouped_by_role = {role: [] for role in roles}
    for calibration in calibrations:
        role = calibration.pop('role__type')
        grouped_by_role[role].append(domain_objs.Calibration(**calibration))
    return grouped_by_role


def _fetch_calibrations(filter_q, language, fields):
    query = models.Calibration.objects.filter(filter_q)
    has_author = 'author' in fields
    person_fields = dict(
        login=F('author__login'),
        is_dismissed=F('author__is_dismissed'),
        first_name=F('author__first_name_' + language),
        last_name=F('author__last_name_' + language),
        gender=F('author__gender'),
    )
    if has_author:
        query = query.annotate(**person_fields)
        fields = set(fields) | set(person_fields.keys())
    fetched = query.values(*fields)
    if has_author:
        for obj in fetched:
            obj['author'] = {
                field: obj.pop(field)
                for field in person_fields
            }
    return fetched


def _prepare_calibrations_filter_q(filters, allow_drafts=False):
    query_params = {
        'status__in': const.CALIBRATION_STATUS.ALL,
    }
    calibrations = filters.get('ids')
    calibration_statuses = filters.get('statuses')

    if calibrations:
        query_params['id__in'] = calibrations

    if calibration_statuses:
        query_params['status__in'] = calibration_statuses

    if not allow_drafts:
        query_params['status__in'] = set(query_params['status__in']) - {const.CALIBRATION_STATUS.DRAFT}

    return Q(**query_params)
