# coding: utf-8
from django.db.models import F

from review.core.logic import domain_objs
from review.core import models as core_models
from review.core import const as core_const
from review.staff import const as staff_const


def get_comments(subject, filters, person_reviews=None, role_types=None):
    from review.core.logic import assemble

    renames = {
        'subject_' + field_name: F('subject__' + field_name + '_' + subject.language)
        for field_name in staff_const.LOCALIZED_PERSON_FIELDS
    }
    renames.update({
        'subject_login': F('subject__login'),
        'subject_gender': F('subject__gender'),
    })

    filters_to_apply = {}
    if 'person_review_ids' in filters:
        filters_to_apply['person_review_id__in'] = filters['person_review_ids']
    if 'id' in filters:
        filters_to_apply['id'] = filters['id']

    fields_to_fetch = (
        'id',
        'person_review_id',
        'subject_id',
        'subject_login',
        'subject_first_name',
        'subject_last_name',
        'subject_gender',
        'subject_type',
        'text_wiki',
        'created_at',
        'updated_at',
    )
    comments = core_models.PersonReviewComment.objects.filter(
        **filters_to_apply
    ).annotate(**renames).values(*fields_to_fetch)

    required_person_review_ids = {
        comment['person_review_id']
        for comment in comments
    }
    if person_reviews:
        person_review_ids_to_fetch = required_person_review_ids - {
            pre.id for pre in person_reviews
        }
        fetched_person_reviews = {pre.id: pre for pre in person_reviews}
    else:
        person_review_ids_to_fetch = required_person_review_ids
        fetched_person_reviews = {}

    if person_review_ids_to_fetch:
        fetched = assemble.get_person_reviews(
            subject=subject,
            filters_chosen={
                core_const.FILTERS.IDS: list(person_review_ids_to_fetch),
            },
            fields_requested=(
                core_const.FIELDS.PERMISSIONS_READ,
                core_const.FIELDS.REVIEW_STATUS,
            ),
            role_types=role_types,
        )
        fetched_person_reviews.update({pre.id: pre for pre in fetched})

    result = {}
    for comment in comments:
        person_review_id = comment['person_review_id']
        person_review = fetched_person_reviews.get(person_review_id)
        if person_review is None:
            continue
        if not person_review.permissions_read[core_const.PERSON_REVIEW_PERMS.COMMENTS_READ]:
            continue
        comment_extended = domain_objs.PersonReviewComment(**comment)
        set_actions(
            subject=subject,
            comment=comment_extended,
            person_review=person_review,
        )
        result[comment['id']] = comment_extended
    return result


def set_actions(subject, comment, person_review):
    comment.actions = {
        core_const.PERSON_REVIEW_COMMENT_ACTIONS.EDIT: all([
            subject.id == comment.subject_id,
            person_review.review_status in core_const.REVIEW_STATUS.ACTIVE,
        ])
    }
