# coding: utf-8
"""
Функции для получения списков доступных PersonReview (с минимальным необходимым
набором полей) для каждой из ролей запрашивающего.
"""

import logging

from django.db.models import Q

import yenv

from review.core.logic import domain_objs
from review.lib import helpers
from review.staff import logic as staff_logic
from review.core import const
from review.core import models as core_models


log = logging.getLogger(__name__)


def get_allowed_person_reviews_flat(subject, fetched, **kwargs):
    role_to_data = collect_allowed_person_reviews_by_roles(subject, **kwargs)
    for role, person_review_data in role_to_data:
        _set_in_fetched(
            fetched=fetched,
            role=role,
            person_review_data=person_review_data,
        )
    flat = set()
    for role_type, person_reviews_group in fetched.grouped_by_roles.items():
        for person_review in person_reviews_group:
            flat.add(person_review)
    return flat


def get_allowed_person_review_ids(subject, filters, role_types):
    role_to_data = collect_allowed_person_reviews_by_roles(
        subject,
        role_types,
        filters,
        {const.FIELDS.ID},
    )
    return {data['id'] for _, data in role_to_data}


def collect_allowed_person_reviews_by_roles(
    subject,
    role_types=None,
    filters=None,
    db_fields=None,
):
    subject = staff_logic.ensure_person_model(subject)
    filters = filters or {}
    role_types = role_types and set(role_types) or set(const.ROLE.PERSON_REVIEW_LIST_RELATED)
    # используем _DENORMALIZED-типы только внутри этого модуля, принимаем и
    # отдаем те роли, от которых они денормализованы.
    for type in list(role_types):
        if type in const.ROLE.DENORMALIZATION:
            role_types.remove(type)
            role_types.add(const.ROLE.DENORMALIZATION[type])

    default_db_fields = const.FIELDS.FROM_COLLECT_STAGE
    if db_fields is None:
        db_fields = default_db_fields
    else:
        db_fields = set(db_fields) & default_db_fields
    db_fields |= const.FIELDS.MINIMUM
    result = []
    if const.ROLE.PERSON.SELF in role_types:
        result += collect_for_self(
            subject=subject,
            db_fields=db_fields,
            filters=filters,
        )

    if const.ROLE.DEPARTMENT.HEAD in role_types:
        result += collect_for_head(
            subject=subject,
            db_fields=db_fields,
            filters=filters,
        )

    hr_roles = role_types & const.ROLE.DEPARTMENT.HRS
    if hr_roles:
        result += collect_for_hr(
            subject=subject,
            db_fields=db_fields,
            filters=filters,
            role_types=hr_roles,
        )

    review_roles = const.ROLE.REVIEW.ALL & role_types
    if review_roles:
        result += collect_for_review_roles(
            subject=subject,
            db_fields=db_fields,
            filters=filters,
            role_types=review_roles,
        )

    person_review_stored_roles = const.ROLE.PERSON_REVIEW.ALL & role_types
    if person_review_stored_roles:
        result += collect_for_person_review_roles(
            subject=subject,
            db_fields=db_fields,
            filters=filters,
            role_types=person_review_stored_roles,
        )

    robot_roles = const.ROLE.GLOBAL.ROBOTS & role_types
    if robot_roles:
        result += collect_for_robot(
            db_fields=db_fields,
            filters=filters,
            role_types=robot_roles & {it.type for it in subject.global_roles.all()},
        )
    return result


@helpers.timeit_no_args_logging
def collect_for_robot(
    role_types,
    db_fields=const.FIELDS.FOR_ROBOT,
    filters=None,
):
    if not role_types:
        return []
    filter_q = _prepare_person_reviews_filter_q(filters or {}, allow_drafts=True)
    queryset = core_models.PersonReview.objects.filter(filter_q).values(*db_fields)
    res = []
    for data in queryset:
        for role in role_types:
            res.append((role, data))
    return res


@helpers.timeit_no_args_logging
def collect_for_review_roles(
    subject,
    db_fields=const.FIELDS.FROM_COLLECT_STAGE,
    filters=None,
    role_types=None,
):
    subject = staff_logic.ensure_person_model(subject)
    filter_q = _prepare_person_reviews_filter_q(filters or {}, allow_drafts=True)

    if role_types is None:
        role_types = const.ROLE.REVIEW.ALL
    db_fields |= {'review__role__type'}

    queryset = core_models.PersonReview.objects.filter(
        filter_q,
        review__role__type__in=role_types,
        review__role__person=subject,
    ).exclude(
        person=subject,
    ).values(*db_fields)

    return [
        (data.pop('review__role__type'), data)
        for data in queryset
    ]


@helpers.timeit_no_args_logging
def collect_for_head(
    subject,
    db_fields=const.FIELDS.FROM_COLLECT_STAGE,
    filters=None,
):
    subject = staff_logic.ensure_person_model(subject)
    filter_q = _prepare_person_reviews_filter_q(filters or {}, allow_drafts=False)

    subordinate_persons_q = staff_logic.get_all_subordinate_persons_q(
        subject,
        prefix='person__',
    )

    paths = const.staff_const.ROOT_DEPARTMENT_PATHS
    ext_or_outstaff = (
        staff_logic.get_persons_from_department_root_q(paths.get('ext', 'WRONG'), prefix='person__') |
        staff_logic.get_persons_from_department_root_q(paths.get('outstaff', 'WRONG'), prefix='person__')
    )

    # For heads of ext and outstaff employees show person reviews only
    # if the head is also their reviewer: it means that he is the actual head, not the formal one.
    person_is_reviewer_q = Q(role__type__in=const.ROLE.PERSON_REVIEW.ALL_REVIEWER_ROLES, role__person=subject)
    queryset = (
        core_models.PersonReview.objects
        .filter(filter_q, subordinate_persons_q)
        .filter(~ext_or_outstaff | (ext_or_outstaff & person_is_reviewer_q))
        .values(*db_fields)
        .order_by('id')
        .distinct('id')
    )

    return [
        (const.ROLE.DEPARTMENT.HEAD, person_review_data)
        for person_review_data in queryset
    ]


@helpers.timeit_no_args_logging
def collect_for_hr(
    subject,
    role_types,
    db_fields=const.FIELDS.FROM_COLLECT_STAGE,
    filters=None,
):
    subject = staff_logic.ensure_person_model(subject)

    queryset = core_models.PersonReview.objects.filter(
        _prepare_person_reviews_filter_q(filters or {}, allow_drafts=True),
        person__hr__hr_person=subject,
        person__hr__type__in=role_types,
    ).values('person__hr__type', *db_fields)
    return [
        (data.pop('person__hr__type'), data)
        for data in queryset
    ]


@helpers.timeit_no_args_logging
def collect_for_self(
    subject,
    db_fields=const.FIELDS.FROM_COLLECT_STAGE,
    filters=None,
):
    subject = staff_logic.ensure_person_model(subject)
    filter_q = _prepare_person_reviews_filter_q(filters or {}, allow_drafts=False)
    queryset = core_models.PersonReview.objects.filter(
        filter_q,
        person=subject,
        status=const.PERSON_REVIEW_STATUS.ANNOUNCED,
    ).values(*db_fields)

    return [
        (const.ROLE.PERSON.SELF, data)
        for data in queryset
    ]


@helpers.timeit_no_args_logging
def collect_for_person_review_roles(
    subject,
    db_fields=const.FIELDS.FROM_COLLECT_STAGE,
    filters=None,
    role_types=None
):
    subject = staff_logic.ensure_person_model(subject)
    filter_q = _prepare_person_reviews_filter_q(filters or {}, allow_drafts=True)

    if role_types is None:
        role_types = const.ROLE.PERSON_REVIEW.ALL

    db_fields |= {
        'role__type',
    }
    person_reviews = core_models.PersonReview.objects.filter(
        filter_q,
        role__person=subject,
        role__type__in=role_types,
    ).exclude(
        person=subject,
    ).values(*db_fields)

    return [
        (data.pop('role__type'), data)
        for data in person_reviews
    ]


@helpers.timeit_no_args_logging
def collect_for_calibration_roles(
    subject,
    fetched,
    db_fields=const.FIELDS.FROM_COLLECT_STAGE,
    filters=None,
    role_types=None,
):
    subject = staff_logic.ensure_person_model(subject)
    filter_q = _prepare_person_reviews_filter_q(filters or {}, allow_drafts=True)

    db_fields |= {'calibration_person_review__calibration__role__type'}

    queryset = core_models.PersonReview.objects.filter(
        filter_q,
        calibration_person_review__calibration__role__type__in=role_types,
        calibration_person_review__calibration__role__person=subject,
    ).exclude(
        person=subject,
    ).values(*db_fields)

    return [
        (data.pop('calibration_person_review__calibration__role__type'), data)
        for data in queryset
    ]


def _set_in_fetched(fetched, role, person_review_data):
    id = person_review_data.pop('id')
    person_id = person_review_data.pop('person_id')
    review_id = person_review_data.pop('review_id')
    umbrella_id = person_review_data.pop('umbrella_id', None)
    main_product_id = person_review_data.pop('main_product_id', None)
    if role in const.ROLE.DENORMALIZATION_REVERSED:
        role = const.ROLE.DENORMALIZATION_REVERSED[role]
    group = fetched.grouped_by_roles.setdefault(role, set())
    group.add(domain_objs.PersonReview(
        id=id,
        person_id=person_id,
        review_id=review_id,
        umbrella_id=umbrella_id,
        main_product_id=main_product_id,
    ))
    if id not in fetched.person_reviews:
        fetched.person_reviews[id] = dict(person_review_data)


def _prepare_person_reviews_filter_q(filters, allow_drafts):
    query_params = {
        'review__status__in': const.REVIEW_STATUS.VISIBLE,
    }
    reviews = filters.get(const.FILTERS.REVIEWS)
    scales = filters.get(const.FILTERS.SCALE)
    review_activity = filters.get(const.FILTERS.REVIEW_ACTIVITY)
    review_statuses = filters.get(const.FILTERS.REVIEW_STATUSES)
    persons = filters.get(const.FILTERS.PERSONS)
    calibrations = filters.get(const.FILTERS.CALIBRATIONS)
    ids = filters.get(const.FILTERS.IDS)
    calibration_id = filters.get(const.FILTERS.CALIBRATION)
    tag_average_mark = filters.get(const.FILTERS.TAG_AVERAGE_MARK)
    taken_in_average = filters.get(const.FILTERS.TAKEN_IN_AVERAGE)
    umbrella = filters.get(const.FILTERS.UMBRELLA)
    umbrella__isnull = filters.get(const.FILTERS.UMBRELLA_ISNULL)
    main_product = filters.get(const.FILTERS.MAIN_PRODUCT)
    main_product__isnull = filters.get(const.FILTERS.MAIN_PRODUCT_ISNULL)

    if ids:
        query_params['id__in'] = ids
    if reviews:
        query_params['review_id__in'] = reviews
    if scales:
        query_params['review__scale_id__in'] = scales
    if review_activity is not None:
        if review_activity:
            query_params['review__status__in'] = const.REVIEW_STATUS.ACTIVE
        else:
            query_params['review__status__in'] = const.REVIEW_STATUS.INACTIVE
    if review_statuses is not None:
        query_params['review__status__in'] = review_statuses
    if calibrations:
        query_params['calibration_person_review__calibration_id__in'] = calibrations
    if not allow_drafts:
        query_params['review__status__in'] = set(query_params['review__status__in']) - {
            const.REVIEW_STATUS.DRAFT
        }

    if persons:
        query_params['person__login__in'] = persons
    if calibration_id:
        statuses = const.CALIBRATION_STATUS.VISIBLE
        query_params.update(
            calibration_person_review__calibration_id=calibration_id,
            calibration_person_review__calibration__status__in=statuses,
        )
    if tag_average_mark:
        query_params['tag_average_mark__in'] = tag_average_mark

    if taken_in_average is not None:
        query_params['taken_in_average'] = taken_in_average

    if yenv.type == 'development':
        assert query_params, 'Empty filter looks like mistake'

    filters_q = Q(**query_params)

    if umbrella or umbrella__isnull is not None:
        subfilter = Q()
        if umbrella:
            subfilter |= Q(umbrella__in=umbrella)
        if umbrella__isnull is not None:
            subfilter |= Q(umbrella__isnull=umbrella__isnull)
        filters_q &= subfilter

    if main_product or main_product__isnull is not None:
        subfilter = Q()
        if main_product:
            subfilter |= Q(main_product__in=main_product)
        if main_product__isnull is not None:
            subfilter |= Q(main_product__isnull=main_product__isnull)
        filters_q &= subfilter

    return filters_q
