from collections import defaultdict

from django.db.models import Q

from review.core import const
from review.core import models
from review.core.logic.roles import has_global_roles
from review.staff.logic import ensure_person_model, get_all_subordinate_persons_q
from review.staff import models as staff_models
from review.core.logic import domain_objs


DEFAULT_REVIEW_LIST_ROLES = (
    const.ROLE.PERSON_REVIEW_LIST_RELATED |
    {
        const.ROLE.GLOBAL.REVIEW_CREATOR,
        const.ROLE.GLOBAL.EXPORTER,
        const.ROLE.GLOBAL.ROBOT,
        const.ROLE.REVIEW.ADMIN,
        const.ROLE.REVIEW.ACCOMPANYING_HR,
    }
) - {
    # denormalized review roles are excessive
    const.ROLE.PERSON_REVIEW.SUPERREVIEWER_DENORMALIZED,
    const.ROLE.PERSON_REVIEW.SUPERREVIEWER_INHERITED,
}


def get_allowed_reviews_flat(*args, **kwargs):
    grouped = get_allowed_reviews_grouped(*args, **kwargs)
    flat = set()
    review_to_roles = {}
    for role_type, reviews_group in grouped.items():
        for review in reviews_group:
            review_to_roles.setdefault(review, [])
            review_to_roles[review].append(role_type)

    for review, roles in review_to_roles.items():
        review.roles = list(set(roles))
        flat.add(review)
    return flat


def get_allowed_reviews_grouped(subject, filters=None, role_types=None, requested_fields=None):
    role_types = role_types and set(role_types) or DEFAULT_REVIEW_LIST_ROLES
    filters = filters or {}
    allowed_review_fields = const.FIELDS.DB_REVIEW_WITH_PARAMETERS | const.FIELDS.DB_REVIEW_WITH_DATES
    if requested_fields is not None:
        requested_fields = set(requested_fields) & allowed_review_fields
    requested_fields = requested_fields or const.FIELDS.DB_REVIEW_FIELDS
    grouped_by_role = {role: [] for role in role_types}
    review_roles = role_types & const.ROLE.REVIEW.ALL
    grouped_by_role.update(
        _get_review_for_review_roles(
            subject=subject,
            filters=dict(filters),
            roles=review_roles,
            fields=requested_fields,
        )
    )
    global_roles = role_types & const.ROLE.GLOBAL.ALL
    grouped_by_role.update(
        _get_review_for_global_roles(
            subject=subject,
            filters=dict(filters),
            roles=global_roles,
            fields=requested_fields,
        )
    )
    person_review_roles = role_types & const.ROLE.PERSON_REVIEW.ALL
    grouped_by_role.update(
        _get_review_for_person_review_roles(
            subject=subject,
            filters=dict(filters),
            roles=person_review_roles,
            fields=requested_fields,
        )
    )
    if const.ROLE.DEPARTMENT.HEAD in role_types:
        grouped_by_role.update(
            _get_reviews_for_head(
                subject=subject,
                filters=dict(filters),
                fields=requested_fields,
            )
        )
    hr_roles = role_types & const.ROLE.DEPARTMENT.HRS
    if hr_roles:
        grouped_by_role.update(
            _get_reviews_for_hrs(
                subject=subject,
                filters=dict(filters),
                fields=requested_fields,
                roles=hr_roles,
            )
        )
    if const.ROLE.PERSON.SELF in role_types:
        grouped_by_role.update(
            _get_reviews_for_self(
                subject=subject,
                filters=dict(filters),
                fields=requested_fields,
            )
        )
    return grouped_by_role


def _get_reviews_for_hrs(subject, filters, fields, roles):
    subject = ensure_person_model(subject)
    cared_persons = list(staff_models.HR.objects.filter(
        hr_person=subject
    ).values('cared_person_id', 'type'))
    if not cared_persons:
        return {}
    person_id_to_roles = defaultdict(list)
    for person in cared_persons:
        person_id_to_roles[person['cared_person_id']].append(person['type'])
    reviews = models.Review.objects.filter(
        _prepare_reviews_filter_q(filters, allow_drafts=True),
        person_review__person_id__in=person_id_to_roles,
    ).values('person_review__person_id', *fields)
    grouped_by_role = defaultdict(list)
    for review in reviews:
        person_id = review.pop('person_review__person_id')
        for role in person_id_to_roles[person_id]:
            grouped_by_role[role].append(domain_objs.Review(**review))
    return grouped_by_role


def _get_review_for_review_roles(subject, filters, roles, fields):
    subject = ensure_person_model(subject)
    filter_q = _prepare_reviews_filter_q(filters, allow_drafts=True)
    reviews = models.Review.objects.filter(
        filter_q,
        role__type__in=roles,
        role__person=subject
    ).values('role__type', *fields)
    grouped_by_role = {role: [] for role in roles}
    for review in reviews.iterator():
        role = review.pop('role__type')
        grouped_by_role[role].append(domain_objs.Review(**review))
    return grouped_by_role


def _get_review_for_global_roles(subject, filters, roles, fields):
    if not has_global_roles(
        person=subject,
        global_roles=roles
    ):
        return {role: [] for role in roles}
    filter_q = _prepare_reviews_filter_q(filters, allow_drafts=True)
    reviews = models.Review.objects.filter(filter_q).values(*fields)
    extended_reviews = [domain_objs.Review(**review) for review in reviews.iterator()]
    return {role: extended_reviews for role in roles}


def _get_review_for_person_review_roles(subject, filters, roles, fields):
    subject = ensure_person_model(subject)
    roles_q = (
        models.PersonReviewRole.objects
        .filter(type__in=roles, person=subject)
        .values_list('person_review__review_id', 'type')
        .distinct('person_review__review_id', 'type')
        .order_by('person_review__review_id', 'type')
    )
    ids = filters.pop('ids', None)
    if ids:
        roles_q = roles_q.filter(person_review__review_id__in=ids)
    review_id_to_roles = defaultdict(list)
    for review_id, role_type in roles_q:
        review_id_to_roles[review_id].append(role_type)

    filter_q = _prepare_reviews_filter_q(filters, allow_drafts=False)
    reviews = (
        models.Review.objects
        .filter(id__in=review_id_to_roles)
        .filter(filter_q)
        .values(*fields)
    )
    grouped_by_role = {role: [] for role in roles}
    for review in reviews:
        for role in review_id_to_roles[review['id']]:
            grouped_by_role[role].append(domain_objs.Review(**review))
    return grouped_by_role


def _get_reviews_for_head(subject, filters, fields):
    subject = ensure_person_model(subject)
    filter_q = _prepare_reviews_filter_q(filters, allow_drafts=False)
    subordinate_persons_q = get_all_subordinate_persons_q(
        subject,
        prefix='person_review__person__',
    )
    reviews = models.Review.objects.filter(
        filter_q,
        subordinate_persons_q
    ).values(*fields)
    extended_reviews = [domain_objs.Review(**review) for review in reviews.iterator()]
    return {const.ROLE.DEPARTMENT.HEAD: extended_reviews}


def _get_reviews_for_self(subject, filters, fields):
    subject = ensure_person_model(subject)
    filter_q = _prepare_reviews_filter_q(filters, allow_drafts=False)
    reviews = models.Review.objects.filter(
        filter_q,
        person_review__person=subject,
        person_review__status=const.PERSON_REVIEW_STATUS.ANNOUNCED,
    ).values(*fields)
    extended_reviews = [domain_objs.Review(**review) for review in reviews.iterator()]
    return {const.ROLE.PERSON.SELF: extended_reviews}


def _prepare_reviews_filter_q(filters, allow_drafts):
    query_params = {
        'status__in': const.REVIEW_STATUS.ALL,
    }
    reviews = filters.get('ids')
    review_activity = filters.get('activity')
    review_statuses = filters.get('statuses')

    assert not (review_activity and review_statuses)

    if reviews:
        query_params['id__in'] = reviews
    if review_activity is not None:
        if review_activity:
            query_params['status__in'] = const.REVIEW_STATUS.ACTIVE
        else:
            query_params['status__in'] = const.REVIEW_STATUS.INACTIVE

    if review_statuses is not None:
        query_params['status__in'] = review_statuses

    if not allow_drafts:
        query_params['status__in'] = query_params['status__in'] - {const.REVIEW_STATUS.DRAFT}

    return Q(**query_params)
