# coding: utf-8
"""
 * Поле может зависеть от других полей и фетчеров с параметрами (обычно
   фетчер может достать из одной таблицы больше или меньше полей)
 * Фильтр может зависеть только от полей.
 * Фетчеры для простоты зависят от безусловно выставленных полей
   id, person_id, review_id.

Задача разрешения зависимостей состоит в том, чтобы заранее определить полный
сет полей, который нужно будет получить для выбранных полей и фильтров.
После этого полный сет расширяется полями, от которых рекурсивно зависят
выбранные и формируется сет необходимых фетчеров с параметрами.

После этого запускаем цикл, который работает пока не собраны все поля и не
применены все фильтры. На каждой итерации пытаемся применить фильтры,
зависимости, которых удовлетворены. Если таких фильтров нет — выбираем первый
неудовлетворенный фильтр в соотвествии с приоритетом, удовлетворяем его
зависимости. Если неудовлетворенных фильтров нет, но есть непроставленные поля —
ставим их.
"""
from review.core import const
from review.oebs import const as oebs_const

FIELDS = const.FIELDS
FILTERS = const.FILTERS
FETCHERS = const.FETCHERS

DEPS_FIELD = FIELDS.ALL | FIELDS.INTERNAL_FIELDS


def get_steps(fields, filters, deps=None):
    deps = deps or DEPS
    steps = Steps()
    state = State(
        deps=deps,
        fields=fields,
        filters=filters,
    )
    while True:
        while state.has_settable_fields or state.has_applicable_filters:
            applicable_filters = state.get_applicable_filters()
            for filter in applicable_filters:
                steps.add_apply_filter_step(filter, value=filters[filter])
                state.mark_filter_applied(filter)

            settable_fields = state.get_settable_fields()
            for field in settable_fields:
                steps.add_set_field_step(field)
                state.mark_field_set(field)

        if not state.has_unhandled:
            break

        fetcher = state.get_cheapest_fetcher()
        if fetcher:
            steps.add_call_fetcher_step(fetcher.id, params=fetcher.params)
            state.mark_fetcher_called(fetcher)
        else:
            raise RuntimeError("Failed to resolve deps %s", state)
    return steps


class Dependency(object):

    def __init__(self, id, **params):
        self.id = id
        self.params = params
        self.normalize_params()

    @property
    def is_field(self):
        return self.id in DEPS_FIELD

    @property
    def is_filter(self):
        return self.id in const.FILTERS.ALL

    @property
    def is_fetcher(self):
        return self.id in const.FETCHERS.ALL

    def normalize_params(self):
        for key, values in self.params.items():
            if not isinstance(values, (list, tuple, set)):
                values = [values]
            self.params[key] = tuple(sorted(values))

    def __hash__(self):
        return hash(self.id) ^ hash(tuple(self.params.items()))

    def __eq__(self, other):
        return hash(self) == hash(other)

    def __repr__(self):
        res = '<D %s' % self.id
        if self.params:
            res += ':' + ', '.join([
                '%s=%s' % (key, value) for key, value in self.params.items()
            ])
        res += '>'
        return res


D = Dependency


class DEPS(object):

    @classmethod
    def wrap(cls, item):
        if not isinstance(item, D):
            return D(item)
        else:
            return item

    @classmethod
    def get_field_deps(cls, field):
        deps = cls.FIELDS_DEPS.get(field) or []
        if not isinstance(deps, (list, tuple, set)):
            deps = [deps]
        return {
            cls.wrap(dep) for dep in deps
        }

    @classmethod
    def get_filter_deps(cls, filter):
        deps = cls.FILTERS_DEPS.get(filter) or []
        if not isinstance(deps, (list, tuple, set, frozenset)):
            deps = [deps]
        return set(cls.wrap(dep) for dep in deps)

    FIELDS_DEPS = {
        # subject fields
        FIELDS.ROLES: (
            FETCHERS.COLLECT,
        ),
        FIELDS.MODIFIERS: (
            FIELDS.ROLES,
            D(FETCHERS.REVIEWS, db_fields=list(const.REVIEW_MODE.FIELDS_TO_MODIFIERS.values()))
        ),
        FIELDS.PERMISSIONS_READ: (
            FIELDS.ROLES,
            D(FETCHERS.COLLECT, db_fields='status'),
            D(FETCHERS.REVIEWS, db_fields='status'),
        ),
        FIELDS.PERMISSIONS_WRITE: (
            FIELDS.ROLES,
            FIELDS.MODIFIERS,
            D(FETCHERS.COLLECT, db_fields='status'),
            D(FETCHERS.REVIEWS, db_fields='status'),
            D(FETCHERS.COLLECT, db_fields='approve_level'),
            FETCHERS.REVIEWERS,
            FIELDS.ACTION_AT,
            FIELDS.PRODUCT_SCHEMA_LOADED,
        ),
        FIELDS.SUBORDINATION: (
            FETCHERS.SUBORDINATION,
        ),

        # безусловно ставятся, не нужно от них зависеть
        FIELDS.ID: (
            D(FETCHERS.COLLECT, db_fields='id'),
        ),
        FIELDS.PERSON_ID: (
            D(FETCHERS.COLLECT, db_fields='person_id'),
        ),
        FIELDS.REVIEW_ID: (
            D(FETCHERS.COLLECT, db_fields='review_id'),
        ),
        FIELDS.UMBRELLA_ID: (
            D(FETCHERS.COLLECT, db_fields='umbrella_id'),
        ),
        FIELDS.UMBRELLA: (
            FIELDS.UMBRELLA_ID,
            D(FETCHERS.UMBRELLA),
            D(FETCHERS.MAIN_PRODUCT),
        ),
        FIELDS.MAIN_PRODUCT_ID: (
            D(FETCHERS.COLLECT, db_fields='main_product_id'),
        ),
        FIELDS.MAIN_PRODUCT: (
            FIELDS.MAIN_PRODUCT_ID,
            D(FETCHERS.UMBRELLA),
            D(FETCHERS.MAIN_PRODUCT),
        ),
        FIELDS.PRODUCT_SCHEMA_LOADED: (
            D(FETCHERS.REVIEWS),
            D(FETCHERS.PRODUCT_SCHEMA_LOADED),
        ),

        # collect
        FIELDS.STATUS: (
            FIELDS.PERMISSIONS_READ,
            D(FETCHERS.COLLECT, db_fields='status'),
        ),
        FIELDS.UPDATED_AT: (
            FIELDS.PERMISSIONS_READ,
            D(FETCHERS.COLLECT, db_fields='updated_at'),
        ),
        FIELDS.MARK: (
            FIELDS.PERMISSIONS_READ,
            FIELDS.MODIFIERS,
            D(FETCHERS.COLLECT, db_fields='mark'),
        ),
        FIELDS.GOLDSTAR: (
            FIELDS.PERMISSIONS_READ,
            FIELDS.MODIFIERS,
            D(FETCHERS.COLLECT, db_fields='goldstar'),
        ),
        FIELDS.LEVEL_CHANGE: (
            FIELDS.PERMISSIONS_READ,
            FIELDS.MODIFIERS,
            D(FETCHERS.COLLECT, db_fields='level_change'),
        ),
        FIELDS.SALARY_CHANGE_TYPE: (
            FIELDS.PERMISSIONS_READ,
            FIELDS.MODIFIERS,
            D(FETCHERS.COLLECT, db_fields='salary_change_type'),
        ),
        FIELDS.SALARY_CHANGE: (
            FIELDS.PERMISSIONS_READ,
            FIELDS.PERMISSIONS_WRITE,
            FIELDS.MODIFIERS,
            FIELDS.SALARY_CHANGE_TYPE,
            D(FETCHERS.COLLECT, db_fields='salary_change'),
        ),
        FIELDS.SALARY_CHANGE_ABSOLUTE: (
            FIELDS.PERMISSIONS_READ,
            FIELDS.PERMISSIONS_WRITE,
            FIELDS.MODIFIERS,
            FIELDS.SALARY_CHANGE_TYPE,
            D(FETCHERS.COLLECT, db_fields='salary_change_absolute'),
        ),
        FIELDS.SALARY_AFTER_REVIEW: (
            FIELDS.PERMISSIONS_READ,
            FIELDS.SALARY_CHANGE_ABSOLUTE,
            FIELDS.SALARY_VALUE,
        ),
        FIELDS.BONUS_TYPE: (
            FIELDS.PERMISSIONS_READ,
            FIELDS.MODIFIERS,
            D(FETCHERS.COLLECT, db_fields='bonus_type'),
        ),
        FIELDS.BONUS: (
            FIELDS.PERMISSIONS_READ,
            FIELDS.PERMISSIONS_WRITE,
            FIELDS.MODIFIERS,
            FIELDS.BONUS_TYPE,
            D(FETCHERS.COLLECT, db_fields='bonus'),
        ),
        FIELDS.BONUS_ABSOLUTE: (
            FIELDS.PERMISSIONS_READ,
            FIELDS.PERMISSIONS_WRITE,
            FIELDS.MODIFIERS,
            FIELDS.BONUS_TYPE,
            D(FETCHERS.COLLECT, db_fields='bonus_absolute'),
        ),
        FIELDS.BONUS_RSU: (
            FIELDS.PERMISSIONS_READ,
            FIELDS.MODIFIERS,
            D(FETCHERS.COLLECT, db_fields='bonus_rsu'),
        ),
        FIELDS.DEFERRED_PAYMENT: (
            FIELDS.PERMISSIONS_READ,
            FIELDS.MODIFIERS,
            D(FETCHERS.COLLECT, db_fields='deferred_payment'),
        ),
        FIELDS.OPTIONS_RSU: (
            FIELDS.PERMISSIONS_READ,
            FIELDS.MODIFIERS,
            D(FETCHERS.COLLECT, db_fields='options_rsu'),
        ),
        FIELDS.OPTIONS_RSU_LEGACY: (
            FIELDS.PERMISSIONS_READ,
            FIELDS.MODIFIERS,
            D(FETCHERS.COLLECT, db_fields='options_rsu_legacy'),
        ),
        FIELDS.FLAGGED: (
            FIELDS.PERMISSIONS_READ,
            D(FETCHERS.COLLECT, db_fields='flagged'),
        ),
        FIELDS.FLAGGED_POSITIVE: (
            FIELDS.PERMISSIONS_READ,
            D(FETCHERS.COLLECT, db_fields='flagged_positive'),
        ),
        FIELDS.APPROVE_LEVEL: (
            FIELDS.PERMISSIONS_READ,
            D(FETCHERS.COLLECT, db_fields='approve_level'),
        ),
        FIELDS.TAG_AVERAGE_MARK: (
            FIELDS.PERMISSIONS_READ,
            D(FETCHERS.COLLECT, db_fields='tag_average_mark'),
        ),
        FIELDS.TAKEN_IN_AVERAGE: (
            FIELDS.PERMISSIONS_READ,
            D(FETCHERS.COLLECT, db_fields='taken_in_average'),
        ),

        # person data
        FIELDS.PERSON_LOGIN: (
            D(FETCHERS.PERSONS, db_fields='login'),
        ),
        FIELDS.PERSON_FIRST_NAME: (
            D(FETCHERS.PERSONS, db_fields='first_name'),
        ),
        FIELDS.PERSON_LAST_NAME: (
            D(FETCHERS.PERSONS, db_fields='last_name'),
        ),
        FIELDS.PERSON_IS_DISMISSED: (
            D(FETCHERS.PERSONS, db_fields='is_dismissed'),
        ),
        FIELDS.PERSON_JOIN_AT: (
            D(FETCHERS.PERSONS, db_fields='join_at'),
        ),
        FIELDS.PERSON_POSITION: (
            D(FETCHERS.PERSONS, db_fields='position'),
        ),
        FIELDS.PERSON_DEPARTMENT_ID: (
            D(FETCHERS.PERSONS, db_fields='department__id'),
        ),
        FIELDS.PERSON_DEPARTMENT_SLUG: (
            D(FETCHERS.PERSONS, db_fields='department__slug'),
        ),
        FIELDS.PERSON_DEPARTMENT_NAME: (
            D(FETCHERS.PERSONS, db_fields='department__name'),
        ),
        FIELDS.PERSON_DEPARTMENT_PATH: (
            D(FETCHERS.PERSONS, db_fields='department__path'),
        ),
        FIELDS.PERSON_DEPARTMENT_CHAIN_SLUGS: (
            FIELDS.PERSON_DEPARTMENT_PATH,
            D(FETCHERS.DEPARTMENT_CHAIN, db_fields='slug')
        ),
        FIELDS.PERSON_DEPARTMENT_CHAIN_NAMES: (
            FIELDS.PERSON_DEPARTMENT_PATH,
            D(FETCHERS.DEPARTMENT_CHAIN, db_fields='name')
        ),
        FIELDS.PERSON_GENDER: (
            D(FETCHERS.PERSONS, db_fields='gender'),
        ),
        FIELDS.PERSON_CITY_NAME: (
            D(FETCHERS.PERSONS, db_fields='city_name'),
        ),
        FIELDS.PERSON_CHIEF: (
            FIELDS.PERSON_ID,
            D(FETCHERS.CHIEF)
        ),

        # finance data
        FIELDS.GRADE_CLOSEST_TO_REVIEW_START: (
            FIELDS.REVIEW_START_DATE,
            FIELDS.PERMISSIONS_READ,
            D(FETCHERS.FINANCE_EVENTS, event_types=oebs_const.GRADE_HISTORY),
        ),
        FIELDS.SALARY_CLOSEST_TO_REVIEW_START: (
            FIELDS.REVIEW_SALARY_DATE,
            FIELDS.REVIEW_START_DATE,
            FIELDS.PERMISSIONS_READ,
            D(FETCHERS.FINANCE_EVENTS, event_types=oebs_const.SALARY_HISTORY),
        ),
        FIELDS.FTE: (
            FIELDS.PERSON_ID,
            FIELDS.PERMISSIONS_READ,
            D(FETCHERS.CURRENT_SALARIES),
        ),
        FIELDS.LEVEL: (
            FIELDS.GRADE_CLOSEST_TO_REVIEW_START,
        ),
        FIELDS.PROFESSION: (
            FIELDS.GRADE_CLOSEST_TO_REVIEW_START,
        ),
        FIELDS.SALARY_VALUE: (
            FIELDS.SALARY_CLOSEST_TO_REVIEW_START,
        ),
        FIELDS.SALARY_CURRENCY: (
            FIELDS.SALARY_CLOSEST_TO_REVIEW_START,
        ),

        # reviews
        FIELDS.REVIEW_NAME: (
            D(FETCHERS.REVIEWS, db_fields='name'),
        ),
        FIELDS.REVIEW_KPI_LOADED: (
            D(FETCHERS.REVIEWS, db_fields='kpi_loaded')
        ),
        FIELDS.REVIEW_TYPE: (
            D(FETCHERS.REVIEWS, db_fields='type'),
        ),
        FIELDS.REVIEW_SCALE_ID: (
            D(FETCHERS.REVIEWS, db_fields='scale_id'),
        ),
        FIELDS.REVIEW_STATUS: (
            D(FETCHERS.REVIEWS, db_fields='status'),
        ),
        FIELDS.REVIEW_START_DATE: (
            D(FETCHERS.REVIEWS, db_fields='start_date'),
        ),
        FIELDS.REVIEW_FINISH_DATE: (
            D(FETCHERS.REVIEWS, db_fields='finish_date'),
        ),
        FIELDS.REVIEW_EVALUATION_FROM_DATE: (
            D(FETCHERS.REVIEWS, db_fields='evaluation_from_date'),
        ),
        FIELDS.REVIEW_EVALUATION_TO_DATE: (
            D(FETCHERS.REVIEWS, db_fields='evaluation_to_date'),
        ),
        FIELDS.REVIEW_FEEDBACK_FROM_DATE: (
            D(FETCHERS.REVIEWS, db_fields='feedback_from_date'),
        ),
        FIELDS.REVIEW_FEEDBACK_TO_DATE: (
            D(FETCHERS.REVIEWS, db_fields='feedback_to_date'),
        ),

        FIELDS.REVIEW_SALARY_DATE: (
            D(FETCHERS.REVIEWS, db_fields='salary_date'),
        ),

        FIELDS.REVIEW_GOALS_FROM_DATE: (
            FIELDS.REVIEW_EVALUATION_FROM_DATE,
            FIELDS.REVIEW_GOALS_TO_DATE,
        ),
        FIELDS.REVIEW_GOALS_TO_DATE: (
            FIELDS.REVIEW_START_DATE,
            FIELDS.REVIEW_EVALUATION_TO_DATE,
        ),
        FIELDS.REVIEW_OPTIONS_RSU_FORMAT: (
            D(FETCHERS.REVIEWS, db_fields='options_rsu_format'),
        ),
        FIELDS.REVIEW_OPTIONS_RSU_UNIT: (
            D(FETCHERS.REVIEWS, db_fields='options_rsu_unit'),
        ),
        FIELDS.REVIEW_MARKS_SCALE: (
            FIELDS.REVIEW_SCALE_ID,
            D(FETCHERS.MARKS_SCALES),
        ),


        # complex
        FIELDS.SALARY_CHANGE_IS_DIFFER_FROM_DEFAULT: (
            FIELDS.LEVEL,
            FIELDS.MARK,
            FIELDS.GOLDSTAR,
            FIELDS.SALARY_CHANGE,
            FIELDS.LEVEL_CHANGE,
            D(FETCHERS.GOODIES, db_fields='salary_change_history'),
        ),
        FIELDS.BONUS_IS_DIFFER_FROM_DEFAULT: (
            FIELDS.LEVEL,
            FIELDS.MARK,
            FIELDS.GOLDSTAR,
            FIELDS.BONUS,
            FIELDS.LEVEL_CHANGE,
            D(FETCHERS.GOODIES, db_fields='bonus_history'),
        ),
        FIELDS.OPTIONS_RSU_IS_DIFFER_FROM_DEFAULT: (
            FIELDS.LEVEL,
            FIELDS.MARK,
            FIELDS.GOLDSTAR,
            FIELDS.OPTIONS_RSU,
            FIELDS.LEVEL_CHANGE,
            D(FETCHERS.GOODIES, db_fields='options_rsu_history'),
        ),
        FIELDS.ACTION_AT: (
            FIELDS.PERMISSIONS_READ,
            D(FETCHERS.COLLECT, db_fields='status'),
            D(FETCHERS.COLLECT, db_fields='approve_level'),
            FIELDS.REVIEWERS,
        ),
        FIELDS.REVIEWERS: (
            FIELDS.PERMISSIONS_READ,
            FETCHERS.REVIEWERS,
        ),
        FIELDS.SHORT_HISTORY: (
            D(FETCHERS.PERSON_REVIEW_HISTORY, db_fields='type'),
            D(FETCHERS.REVIEWS, db_fields=('type', 'start_date')),
        ),
        FIELDS.MARK_LEVEL_HISTORY: (
            FIELDS.SHORT_HISTORY,
        ),

        # actions
        FIELDS.ACTION_MARK: (
            FIELDS.MODIFIERS,
            FIELDS.PERMISSIONS_WRITE,
        ),
        FIELDS.ACTION_GOLDSTAR: (
            FIELDS.MODIFIERS,
            FIELDS.PERMISSIONS_WRITE,
        ),
        FIELDS.ACTION_LEVEL_CHANGE: (
            FIELDS.MODIFIERS,
            FIELDS.PERMISSIONS_WRITE,
        ),
        FIELDS.ACTION_SALARY_CHANGE: (
            FIELDS.MODIFIERS,
            FIELDS.PERMISSIONS_WRITE,
        ),
        FIELDS.ACTION_SALARY_CHANGE_ABSOLUTE: (
            FIELDS.MODIFIERS,
            FIELDS.PERMISSIONS_WRITE,
        ),
        FIELDS.ACTION_BONUS: (
            FIELDS.MODIFIERS,
            FIELDS.PERMISSIONS_WRITE,
        ),
        FIELDS.ACTION_BONUS_ABSOLUTE: (
            FIELDS.MODIFIERS,
            FIELDS.PERMISSIONS_WRITE,
        ),
        FIELDS.ACTION_BONUS_RSU: (
            FIELDS.MODIFIERS,
            FIELDS.PERMISSIONS_WRITE,
        ),
        FIELDS.ACTION_DEFERRED_PAYMENT: (
            FIELDS.MODIFIERS,
            FIELDS.PERMISSIONS_WRITE,
        ),
        FIELDS.ACTION_OPTIONS_RSU: (
            FIELDS.MODIFIERS,
            FIELDS.PERMISSIONS_WRITE,
        ),
        FIELDS.ACTION_FLAGGED: (
            FIELDS.PERMISSIONS_WRITE,
        ),
        FIELDS.ACTION_FLAGGED_POSITIVE: (
            FIELDS.PERMISSIONS_WRITE,
        ),
        FIELDS.ACTION_APPROVE: (
            FIELDS.PERMISSIONS_WRITE,
            FIELDS.FLAGGED,
            FIELDS.FLAGGED_POSITIVE,
        ),
        FIELDS.ACTION_UNAPPROVE: (
            FIELDS.PERMISSIONS_WRITE,
            FIELDS.FLAGGED,
            FIELDS.FLAGGED_POSITIVE,
        ),
        FIELDS.ACTION_ALLOW_ANNOUNCE: (
            FIELDS.PERMISSIONS_WRITE,
            FIELDS.FLAGGED,
            FIELDS.FLAGGED_POSITIVE,
        ),
        FIELDS.ACTION_ANNOUNCE: (
            FIELDS.PERMISSIONS_WRITE,
            FIELDS.FLAGGED,
            FIELDS.FLAGGED_POSITIVE,
        ),
        FIELDS.ACTION_COMMENT: (
            FIELDS.PERMISSIONS_WRITE,
        ),
        FIELDS.ACTION_REVIEWERS: (
            FIELDS.PERMISSIONS_WRITE,
        ),
        FIELDS.ACTION_TAG_AVERAGE_MARK: (
            FIELDS.PERMISSIONS_WRITE,
        ),
        FIELDS.ACTION_UMBRELLA: (
            FIELDS.PERMISSIONS_WRITE,
        ),
        FIELDS.ACTION_MAIN_PRODUCT: (
            FIELDS.PERMISSIONS_WRITE,
        ),
        FIELDS.ACTION_TAKEN_IN_AVERAGE: (
            FIELDS.PERMISSIONS_WRITE,
        ),

        # related lists
        FIELDS.COMMENTS: (
            FIELDS.PERMISSIONS_READ,
            FIELDS.REVIEW_STATUS,
            D(FETCHERS.COMMENTS),
        ),
        FIELDS.CHANGES: (
            FIELDS.PERMISSIONS_READ,
            D(FETCHERS.CHANGES),
        ),

        # external
        FIELDS.GOALS_URL: (
            FIELDS.REVIEW_GOALS_FROM_DATE,
            FIELDS.REVIEW_GOALS_TO_DATE,
        ),
        FIELDS.ST_GOALS_URL: (
            FIELDS.PERSON_LOGIN,
            FIELDS.REVIEW_GOALS_FROM_DATE,
            FIELDS.REVIEW_GOALS_TO_DATE,
        ),
    }

    FILTERS_DEPS = {
        FILTERS.IDS: None,
        FILTERS.REVIEWS: None,
        FILTERS.REVIEW_ACTIVITY: None,
        FILTERS.PERSONS: None,
        FILTERS.REVIEW_STATUSES: None,
        FILTERS.SCALE: None,
        FILTERS.CALIBRATIONS: None,
        FILTERS.TAG_AVERAGE_MARK: None,
        FILTERS.TAKEN_IN_AVERAGE: FIELDS.TAKEN_IN_AVERAGE,
        FILTERS.MARK: FIELDS.MARK,
        FILTERS.GOLDSTAR: FIELDS.GOLDSTAR,
        FILTERS.LEVEL_CHANGE: FIELDS.LEVEL_CHANGE,
        FILTERS.LEVEL_CHANGED: FIELDS.LEVEL_CHANGE,
        FILTERS.SALARY_CHANGE: FIELDS.SALARY_CHANGE,
        FILTERS.BONUS: FIELDS.BONUS,
        FILTERS.OPTIONS_RSU: FIELDS.OPTIONS_RSU,
        FILTERS.DEPARTMENTS: FIELDS.PERSON_DEPARTMENT_PATH,
        FILTERS.PROFESSION: FIELDS.PROFESSION,
        FILTERS.LEVEL: FIELDS.LEVEL,
        FILTERS.SALARY_VALUE: FIELDS.SALARY_VALUE,
        FILTERS.SALARY_CURRENCY: FIELDS.SALARY_CURRENCY,
        FILTERS.PERSON_JOIN_AT: FIELDS.PERSON_JOIN_AT,
        FILTERS.STATUS: FIELDS.STATUS,
        FILTERS.FLAGGED: FIELDS.FLAG_FIELDS,
        FILTERS.ACTION_AT: FIELDS.ACTION_AT,
        FILTERS.IS_DIFFER_FROM_DEFAULT: frozenset([
            FIELDS.BONUS_IS_DIFFER_FROM_DEFAULT,
            FIELDS.SALARY_CHANGE_IS_DIFFER_FROM_DEFAULT,
            FIELDS.OPTIONS_RSU_IS_DIFFER_FROM_DEFAULT,
        ]),
        FILTERS.SUBORDINATION: FIELDS.SUBORDINATION,
        FILTERS.REVIEWER: FIELDS.REVIEWERS,
    }

    FETCHERS_ORDER = (
        FETCHERS.COLLECT,
        FETCHERS.REVIEWS,
        FETCHERS.MARKS_SCALES,
        FETCHERS.GOODIES,
        FETCHERS.SUBORDINATION,
        FETCHERS.CHIEF,
        FETCHERS.REVIEWERS,
        FETCHERS.PERSONS,
        FETCHERS.PERSON_REVIEW_HISTORY,
        FETCHERS.FINANCE_EVENTS,
        FETCHERS.DEPARTMENT_CHAIN,
        FETCHERS.COMMENTS,
        FETCHERS.CHANGES,
        FETCHERS.CURRENT_SALARIES,
        FETCHERS.UMBRELLA,
        FETCHERS.MAIN_PRODUCT,
        FETCHERS.PRODUCT_SCHEMA_LOADED,
    )


class State(object):

    def __init__(self, deps, fields=None, filters=None):
        fields = fields or {}
        filters = filters or {}

        unhandled_fields = set()
        self.filters_deps = {}
        self.fields_deps = {}
        self.deps = deps

        for filter in filters:
            filter_deps = deps.get_filter_deps(filter)
            self.filters_deps[filter] = filter_deps
            unhandled_fields |= {dep.id for dep in filter_deps}

        unhandled_fields |= set(fields)
        while unhandled_fields:
            current_field = unhandled_fields.pop()
            field_deps = deps.get_field_deps(current_field)
            self.fields_deps[current_field] = field_deps
            for dep in field_deps:
                if dep.is_fetcher:
                    continue
                if dep.id in self.filters_deps:
                    continue
                if dep.id in self.fields_deps:
                    continue
                unhandled_fields.add(dep.id)

    @property
    def has_unhandled(self):
        return bool(self.fields_deps or self.filters_deps)

    @property
    def has_applicable_filters(self):
        return bool(self.get_applicable_filters())

    def get_applicable_filters(self):
        result = []
        for filter, deps in self.filters_deps.items():
            if not deps:
                result.append(filter)
        # сортировка не важна, нужна только для предсказуемости в тестах
        return sorted(result)

    @property
    def has_settable_fields(self):
        return bool(self.get_settable_fields())

    def get_settable_fields(self):
        result = []
        for field, deps in self.fields_deps.items():
            if not deps:
                result.append(field)
        # сортировка не важна, нужна только для предсказуемости в тестах
        return sorted(result)

    def get_cheapest_fetcher(self):
        uncalled_fetchers = []
        for field, deps in self.fields_deps.items():
            for dep in deps:
                if not dep.is_fetcher:
                    continue
                uncalled_fetchers.append(dep)
        uncalled_fetchers.sort(
            key=lambda dep: self.deps.FETCHERS_ORDER.index(dep.id)
        )
        if not uncalled_fetchers:
            return
        return uncalled_fetchers[0]

    def mark_field_set(self, field):
        for some_field, deps in self.fields_deps.items():
            deps_for_remove = set()
            for dep in deps:
                if dep.id == field:
                    deps_for_remove.add(dep)
            deps -= deps_for_remove
        del self.fields_deps[field]

        for filter, deps in self.filters_deps.items():
            deps_for_remove = set()
            for dep in deps:
                if dep.id == field:
                    deps_for_remove.add(dep)
            deps -= deps_for_remove

    def mark_filter_applied(self, filter):
        del self.filters_deps[filter]

    def mark_fetcher_called(self, fetcher_dep):
        for field, deps in self.fields_deps.items():
            deps_for_remove = set()
            for dep in deps:
                if not dep.is_fetcher:
                    continue
                if dep == fetcher_dep:
                    deps_for_remove.add(dep)
            deps -= deps_for_remove

    def __repr__(self):
        return ', '.join([
            str(self.fields_deps),
            str(self.filters_deps),
        ])


class Steps(list):

    @property
    def collect_step_params(self):
        for step in self:
            if step.id == const.FETCHERS.COLLECT:
                return step.params
        return {}

    @property
    def collect_step_filters(self):
        return {
            step.id: step.value
            for step in self
            if step.id in const.FILTERS.FOR_COLLECT
        }

    @property
    def non_collect(self):
        return [step for step in self if not step.is_for_collect]

    def add_set_field_step(self, field):
        # не выставляем ранее выставленное поле
        for step in self:
            if not step.is_field:
                continue
            if step.id == field:
                return
        self.append(SetFieldStep(field))

    def add_apply_filter_step(self, filter, value):
        self.append(FilterStep(filter, value))

    def add_call_fetcher_step(self, fetcher, params):
        for step in self:
            if not step.is_fetcher:
                continue
            if step.id == fetcher:
                step.add_params(params)
                return
        self.append(CallFetcherStep(fetcher, params))

    def extend(self, iterable):
        for item in iterable:
            self.append(item)


class Step(object):

    def __init__(self, id, **kwargs):
        self.id = id

    @classmethod
    def typed(cls, id, **kwargs):
        if isinstance(id, D):
            id = id.id
        if isinstance(id, Step):
            return

        if id in const.FIELDS.ALL:
            return SetFieldStep(id, **kwargs)
        if id in const.FILTERS.ALL:
            return FilterStep(id, **kwargs)
        if id in const.FETCHERS.ALL:
            return CallFetcherStep(id, **kwargs)
        raise RuntimeError('Unknown id %s', id)

    @property
    def is_field(self):
        return self.id in DEPS_FIELD

    @property
    def is_filter(self):
        return self.id in const.FILTERS.ALL

    @property
    def is_fetcher(self):
        return self.id in const.FETCHERS.ALL

    @property
    def is_for_collect(self):
        return self.id in const.FILTERS.FOR_COLLECT | {const.FETCHERS.COLLECT}

    def get_params_str(self):
        raise NotImplementedError

    def __eq__(self, other):
        return self.id == other.id

    def __repr__(self):
        return '<Step %s%s>' % (self.id, self.get_params_str())


class SetFieldStep(Step):

    def get_params_str(self):
        return ''

    def __hash__(self):
        return hash(self.id)


class FilterStep(Step):

    def __init__(self, id, value):
        super(FilterStep, self).__init__(id)
        self.value = value

    def __eq__(self, other):
        return super(FilterStep, self).__eq__(other) and self.value == other.value

    def __hash__(self):
        return hash(self.id) ^ hash(self.value)

    def get_params_str(self):
        return '(value=%s)' % self.value


class CallFetcherStep(Step):

    def __init__(self, id, params=None):
        super(CallFetcherStep, self).__init__(id)
        self.params = params or {}

    def __hash__(self):
        return hash(self.id) ^ hash(tuple(self.params.items()))

    def __eq__(self, other):
        return super(CallFetcherStep, self).__eq__(other) and self.params == other.params

    def add_params(self, params):
        for key, value in params.items():
            current_value = set(self.params.get(key, ()))
            new_value = set(value)
            self.params[key] = tuple(sorted(current_value | new_value))

    def get_params_str(self):
        return '(%s)' % ','.join([
            '%s=%s' % (key, value) for key, value in
            list(self.params.items())
        ])
