# coding: utf-8
import operator

from django.db.models import F, Q

from review.lib import helpers
from review.lib import std
from review.lib import datetimes

from review.gradient import models as gradient_models
from review.oebs import logic as oebs_logic
from review.oebs import const as oebs_const
from review.oebs import models as oebs_models
from review.staff import models as staff_models
from review.core import models as core_models
from review.core import const
from review.staff import const as staff_const
from functools import reduce


FETCHERS = const.FETCHERS


def fetch(person_reviews, fetcher, fetched, **kwargs):
    fetcher_id = fetcher[len(const.FETCHERS.PREFIX):]
    globals()['fetch_' + fetcher_id](person_reviews, fetched, **kwargs)


@helpers.timeit_no_args_logging
def fetch_persons(person_reviews, fetched, subject, db_fields=None, **kwargs):
    renames = {
        field_name: F(field_name + '_' + subject.language)
        for field_name in staff_const.LOCALIZED_PERSON_FIELDS
    }

    default_fields = set(const.FIELDS.DB_PERSON_FIELDS_MAP.values())

    if db_fields:
        fields_to_fetch = set(db_fields) & default_fields
    else:
        fields_to_fetch = default_fields
    fields_to_fetch.add('id')

    queryset = staff_models.Person.objects.filter(
        id__in=[
            person_review.person_id
            for person_review in person_reviews
        ]
    ).annotate(**renames).values(*fields_to_fetch)

    for person_data in queryset:
        fetched.persons[person_data.pop('id')] = person_data


@helpers.timeit_no_args_logging
def fetch_finance_events(person_reviews, fetched, event_types=None, **kwargs):
    if not person_reviews or not event_types:
        return
    oebs_event_types = {
        oebs_const.FINANCE_EVENT_TYPES[et]
        for et in event_types
    }
    event_for_persons = []

    today = datetimes.today()

    def start_date_getter(pr):
        if not pr.review_salary_date or pr.review_salary_date == const.NOT_SET:
            return pr.review_start_date
        return min(today, pr.review_salary_date)

    grouped_by_start_date = std.group_by(person_reviews, 'not_used', lambda key: start_date_getter)
    for start_date, grouped_person_reviews in grouped_by_start_date.items():
        person_ids = {pr.person_id for pr in grouped_person_reviews}
        event_for_persons.append(Q(
            person_id__in=person_ids,
            date_from__lte=start_date,
            date_to__gte=start_date,
        ))

    event_for_persons_q = reduce(lambda x, y: x | y, event_for_persons)
    event_for_persons_q &= Q(type__in=oebs_event_types)
    queryset = oebs_models.FinanceEvents.objects.filter(
        event_for_persons_q,
    ).values('person_id', 'type', 'date_from', 'date_to', 'event')

    persons_finances = {}
    # {
    #   person_id: {
    #       salary_history: {
    #           (datefrom, dateto): {currency: '', hiddenInfo: '', salarySum: '', ...},
    #            ...
    #       },
    #       grade_history: {
    #            (datefrom, dateto): {currency: '', gradeName: '', gradeCity: '', min: '', max:'', mid: '', ...},
    #            ...
    #       }
    #   }
    # }
    for event in queryset:
        person_id = event.pop('person_id')
        type = event.pop('type')
        oebs_type = oebs_const.FINANCE_EVENT_BY_TYPE[type]
        person_events = persons_finances.setdefault(person_id, {})
        event_data = oebs_logic.decrypt_finance_event(event['event'])
        events_for_type = person_events.setdefault(oebs_type, {})
        range_key = event['date_from'], event['date_to']
        event_data['dateFrom'] = event['date_from']
        event_data['dateTo'] = event['date_to']
        events_for_type[range_key] = event_data

    for person_review in person_reviews:
        person_id = person_review.person_id
        start_date = start_date_getter(person_review)
        person_review_events = fetched.finance_events.setdefault(person_review.id, {})
        person_events = persons_finances.get(person_id)
        if person_events is None:
            continue
        for type, events in person_events.items():
            for range_key, event_data in events.items():
                date_from, date_to = range_key
                if date_from <= start_date <= date_to:
                    person_review_events[type] = event_data


@helpers.timeit_no_args_logging
def fetch_current_salaries(person_reviews, fetched, **kwargs):
    fetched.current_salaries = oebs_logic.get_finance_data(
        fields=[oebs_const.CURRENT_SALARY],
        person_id__in=[
            person_review.person_id
            for person_review in person_reviews
        ],
    )


@helpers.timeit_no_args_logging
def fetch_reviews(person_reviews, fetched, db_fields=None, **kwargs):
    default_fields = set(const.FIELDS.DB_REVIEW_FIELDS_MAP.values())
    default_fields |= set(const.REVIEW_MODE.FIELDS_TO_MODIFIERS.values())

    if db_fields:
        fields_to_fetch = set(db_fields) & default_fields
    else:
        fields_to_fetch = default_fields
    fields_to_fetch.add('id')

    queryset = core_models.Review.objects.filter(
        id__in=[
            person_review.review_id
            for person_review in person_reviews
        ]
    ).values(*fields_to_fetch)

    for review_data in queryset:
        fetched.reviews[review_data.pop('id')] = review_data


@helpers.timeit_no_args_logging
def fetch_marks_scales(person_reviews, fetched, **kwargs):
    scales_ids = set(it.review_scale_id for it in person_reviews)
    scales = core_models.MarksScale.objects.filter(
        id__in=scales_ids,
    )
    fetched.marks_scales.update(((it.id, it.scale) for it in scales))


@helpers.timeit_no_args_logging
def fetch_person_reviews_history(person_reviews, fetched, db_fields=None, **kwargs):
    default_fields = set()
    if db_fields:
        fields_to_fetch = set(db_fields) & default_fields
    else:
        fields_to_fetch = default_fields
    fields_to_fetch |= {'id', 'person_id', 'review_start_date', 'review_type', 'review_scale_id'}
    person_reviews_to_exclude = {pr.id for pr in person_reviews}
    person_ids = {pr.person_id for pr in person_reviews}
    related_person_reviews = core_models.PersonReview.objects.filter(
        person_id__in=person_ids,
    ).annotate(**{
        'review_start_date': F('review__start_date'),
        'review_type': F('review__type'),
        'review_scale_id': F('review__scale_id'),
    }).values(
        *fields_to_fetch
    )
    grouped_by_person_id = {pid: [] for pid in person_ids}
    for person_review in related_person_reviews:
        if person_review['id'] not in person_reviews_to_exclude:
            grouped_by_person_id[person_review.pop('person_id')].append(person_review)

    for person_id, person_review_data in grouped_by_person_id.items():
        fetched.person_review_history[person_id] = sorted(
            person_review_data,
            key=operator.itemgetter('review_start_date'),
        )


@helpers.timeit_no_args_logging
def fetch_goodies(person_reviews, fetched, db_fields=None, **kwargs):
    default_fields = {
        'review_id',
        'level',
        'mark',
        'goldstar',
        'level_change',
        'salary_change',
        'bonus',
        'options_rsu',
    }
    if db_fields:
        fields_to_fetch = set(db_fields) & default_fields
    else:
        fields_to_fetch = default_fields

    queryset = core_models.Goodie.objects.filter(
        review_id__in=[
            person_review.review_id for person_review in
            person_reviews
        ]
    ).values(*fields_to_fetch)

    for goodie in queryset:
        review_id = goodie.pop('review_id')
        level = goodie.pop('level')
        mark = goodie.pop('mark')
        goldstar = goodie.pop('goldstar')
        level_change = goodie.pop('level_change')
        fetched.goodies[(
            review_id,
            level,
            mark,
            goldstar,
            level_change,
        )] = goodie


@helpers.timeit_no_args_logging
def fetch_reviewers(person_reviews, fetched, subject, db_fields=None, **kwargs):
    default_fields = const.FIELDS.DEFAULT_REVIEWER_FIELDS
    if db_fields:
        fields_to_fetch = set(db_fields) & default_fields
    else:
        fields_to_fetch = default_fields
    queryset = core_models.PersonReviewRole.objects.filter(
        person_review_id__in=[pr.id for pr in person_reviews],
        type__in=[
            const.ROLE.PERSON_REVIEW.REVIEWER,
            const.ROLE.PERSON_REVIEW.TOP_REVIEWER,
        ]
    ).annotate(**{
        'login': F('person__login'),
        'is_dismissed': F('person__is_dismissed'),
        'first_name': F('person__first_name_' + subject.language),
        'last_name': F('person__last_name_' + subject.language),
        'gender': F('person__gender'),
    }).values(
        *fields_to_fetch
    ).order_by(
        'person_review_id',
        'position',
    )

    by_person_review_id = regroup_reviewers_helper(queryset)
    fetched.reviewers = by_person_review_id


def regroup_reviewers_helper(roles_flat, _format='dict'):
    by_person_review_id = {}
    previous_id_and_position = ()
    for role in roles_flat:

        # non-dict format только для переиспользования в нотификациях
        # если все слишком усложнится — можно переделать как-то иначе
        if _format == 'dict':
            person_review_id = role.pop('person_review_id')
            position = role.pop('position')
        else:
            person_review_id = role.person_review_id
            position = role.position

        chain = by_person_review_id.setdefault(person_review_id, [])
        if (person_review_id, position) == previous_id_and_position:
            if isinstance(chain[-1], list):
                chain[-1].append(role)
            else:
                chain[-1] = [chain[-1], role]
        else:
            chain.append(role)
        previous_id_and_position = person_review_id, position
    return by_person_review_id


@helpers.timeit_no_args_logging
def fetch_umbrella(person_reviews, fetched, subject, db_fields=None, **kwargs):
    default_fields = const.FIELDS.DB_UMBRELLA_FIELDS

    if db_fields:
        fields_to_fetch = set(db_fields) & default_fields
    else:
        fields_to_fetch = default_fields
    fields_to_fetch.add('id')

    umbrella_ids = [pr.umbrella_id for pr in person_reviews if pr.umbrella_id]

    if umbrella_ids:
        queryset = gradient_models.Umbrella.objects.filter(id__in=umbrella_ids).values(*fields_to_fetch)

        for umbrella in queryset:
            fetched.umbrellas[umbrella['id']] = umbrella


@helpers.timeit_no_args_logging
def fetch_product_schema_loaded(person_reviews, fetched, **kwargs):
    loaded_review_ids = (
        core_models.Review.objects
        .filter(
            product_schema_loaded__isnull=False,
            id__in={pr.review_id for pr in person_reviews}
        )
        .values_list('id', flat=True)
    )
    fetched.product_schema_loaded = set(loaded_review_ids)


@helpers.timeit_no_args_logging
def fetch_main_product(person_reviews, fetched, subject, db_fields=None, **kwargs):
    default_fields = const.FIELDS.DB_MAIN_PRODUCT_FIELDS

    if db_fields:
        fields_to_fetch = set(db_fields) & default_fields
    else:
        fields_to_fetch = default_fields

    main_product_ids = [pr.main_product_id for pr in person_reviews if pr.main_product_id]
    main_product_ids += [umbrella['main_product_id'] for umbrella in list(fetched.umbrellas.values())]

    queryset = gradient_models.MainProduct.objects.filter(id__in=main_product_ids).values(*fields_to_fetch)

    for main_product in queryset:
        fetched.main_products[main_product['id']] = main_product


@helpers.timeit_no_args_logging
def fetch_subordination(person_reviews, fetched, subject,
                        subordination_subject_login=None, **kwargs):
    subordination_subject_login = subordination_subject_login or subject.login
    data = dict(
        staff_models.Subordination.objects.filter(
            object_id__in=[pr.person_id for pr in person_reviews],
            subject__login=subordination_subject_login,
        ).values_list('object_id', 'type'),
    )
    fetched.subordination.update(data)


def fetch_chief(person_reviews, fetched, subject, **kwargs):
    person_ids = {pr.person_id for pr in person_reviews}
    fetched.person_chief = fetch_chief_for_persons(person_ids, subject)


@helpers.timeit_no_args_logging
def fetch_chief_for_persons(person_ids, subject):
    translate_fields = dict(
        login=F('subject__login'),
        is_dismissed=F('subject__is_dismissed'),
        first_name=F('subject__first_name_' + subject.language),
        last_name=F('subject__last_name_' + subject.language),
        gender=F('subject__gender'),
    )
    direct_chief_position = 0
    persons_chiefs = staff_models.Subordination.objects.filter(
        object__in=set(person_ids),
        type=staff_const.SUBORDINATION.DIRECT,
        position=direct_chief_position,
    ).annotate(**translate_fields).values('object_id', *translate_fields)
    return {
        it.pop('object_id'): it
        for it in persons_chiefs
    }


@helpers.timeit_no_args_logging
def fetch_department_chain(person_reviews, fetched, subject, db_fields=None, **kwargs):
    default_fields = const.FIELDS.DEFAULT_DEPARTMENT_CHAIN_FIELDS
    if db_fields:
        fields_to_fetch = set(db_fields) & default_fields
    else:
        fields_to_fetch = default_fields
    fields_to_fetch.add('path')
    paths = {
        person_review.person_department_path
        for person_review in person_reviews
        # there are persons in deleted departments
        if person_review.person_department_path is not None
    }
    all_paths = set()
    for path in paths:
        all_paths |= set(
            staff_models.Department.get_ancestor_paths(path, include_self=True)
        )

    departments = staff_models.Department.objects.filter(
        path__in=all_paths,
    ).annotate(**{
        'name': F('name_' + subject.language)
    }).values(*fields_to_fetch)

    by_path = {
        department.pop('path'): department for department in departments
    }

    for path in paths:
        chain = [
            by_path[path_from_chain]
            for path_from_chain in
            staff_models.Department.get_ancestor_paths(path, include_self=True)
        ]
        fetched.department_chains[path] = chain


@helpers.timeit_no_args_logging
def fetch_comments(person_reviews, fetched, subject, **kwargs):
    from review.core.logic import assemble

    comments = assemble.get_person_review_comments(
        subject=subject,
        filters={
            'person_review_ids': [pre.id for pre in person_reviews],
        },
        person_reviews=person_reviews,
    )

    for comment in list(comments.values()):
        person_review_id = comment.person_review_id
        person_review_comments = fetched.comments.setdefault(person_review_id, [])
        person_review_comments.append(comment)


@helpers.timeit_no_args_logging
def fetch_changes(person_reviews, fetched, subject, **kwargs):
    from review.core.logic import assemble

    changes = assemble.get_person_review_changes(
        subject=subject,
        filters={
            'person_review_ids': [pre.id for pre in person_reviews],
        },
        person_reviews=person_reviews,
    )

    for change in list(changes.values()):
        person_review_id = change.person_review_id
        person_review_comments = fetched.changes.setdefault(person_review_id, [])
        person_review_comments.append(change)
