from django.db.models import F

from review.core import models as core_models
from review.core import const


def fetch_review_roles(reviews, requested_fields, language=None):
    language = language or 'ru'
    requested_roles_fields = requested_fields and (set(requested_fields) & const.FIELDS.DB_REVIEW_ROLES_FIELDS)
    if not requested_roles_fields:
        return

    parameter_to_role = {
        'admins': const.ROLE.REVIEW.ADMIN,
        'super_reviewers': const.ROLE.REVIEW.SUPERREVIEWER,
        'accompanying_hrs': const.ROLE.REVIEW.ACCOMPANYING_HR,
    }
    role_types = [role for param, role in parameter_to_role.items() if param in requested_roles_fields]

    ids = {r.id for r in reviews}
    roles = core_models.ReviewRole.objects.filter(
        review_id__in=ids,
        type__in=role_types,
    ).annotate(**{
        'login': F('person__login'),
        'is_dismissed': F('person__is_dismissed'),
        'first_name': F('person__first_name_' + language),
        'last_name': F('person__last_name_' + language),
        'gender': F('person__gender'),
        'position': F('person__position_' + language),
        'department_name': F('person__department__name_' + language),
        'department_slug': F('person__department__slug'),
    }).values('review_id', 'type', *const.FIELDS.DEFAULT_REVIEW_ROLE_PERSON_FIELDS)

    parameter_to_role = {
        'admins': const.ROLE.REVIEW.ADMIN,
        'super_reviewers': const.ROLE.REVIEW.SUPERREVIEWER,
        'accompanying_hrs': const.ROLE.REVIEW.ACCOMPANYING_HR,
    }
    for param, role in parameter_to_role.items():
        if param in requested_roles_fields:
            _set_fetched_roles_to_reviews(
                reviews=reviews,
                fetched_roles=roles,
                review_parameter_name=param,
                role_type=role,
            )


def _set_fetched_roles_to_reviews(reviews, fetched_roles, review_parameter_name, role_type):
    grouped = {}
    for role in fetched_roles:
        if role['type'] == role_type:
            group = grouped.setdefault(role['review_id'], [])
            person = {
                key: value
                for key, value in role.items()
                if key in const.FIELDS.DEFAULT_REVIEW_ROLE_PERSON_FIELDS
            }
            group.append(person)
    for review in reviews:
        setattr(review, review_parameter_name, grouped.get(review.id, []))
