# coding: utf-8
from review.staff import const as staff_const
from review.core import const as core_const
from logging import getLogger

log = getLogger(__name__)


def apply(person_reviews, filters):
    FILTERS = core_const.FILTERS

    MATCHERS = {
        'mark': make_enumeration_matcher('mark'),
        'goldstar': make_enumeration_matcher('goldstar'),
        'level_change': make_range_matcher('level_change'),
        'level_changed': make_non_zero_matcher('level_change'),
        'salary_change': make_range_matcher('salary_change'),
        'bonus': make_boolean_decimal_matcher('bonus'),
        'options_rsu': make_boolean_decimal_matcher('options_rsu'),
        'departments': is_departments_matched,
        'profession': make_enumeration_matcher('profession'),
        'level': make_range_matcher('level'),
        'salary_value': make_range_matcher('salary_value'),
        'salary_currency': make_exact_matcher('salary_currency'),
        'person_join_at': make_range_matcher('person_join_at'),
        'status': make_enumeration_matcher('status'),
        'action_at': is_action_at_matched,
        'flagged': flag_matcher,
        # 'is_differ_from_default': is_differ_from_default_matched,
        'subordination': subordination_matched,
        'reviewer': has_reviewer,
        'taken_in_average': make_boolean_matcher('taken_in_average'),
    }

    for filter_id, filter_value in filters.items():
        filter_id = filter_id[len(FILTERS.PREFIX):]
        matcher = MATCHERS[filter_id]
        to_delete = set()
        for person_review in person_reviews:
            if not matcher(person_review, filter_value):
                to_delete.add(person_review)
        person_reviews -= to_delete


def make_enumeration_matcher(attr_name):
    def matcher(person_review, enumeration):
        if not enumeration:
            return True
        attr_value = getattr(person_review, attr_name)
        if isinstance(attr_value, core_const.SPECIAL_VALUE):
            return False
        return attr_value in enumeration
    return matcher


def make_non_zero_matcher(attr_name):
    def matcher(person_review, need_non_zero):
        value = getattr(person_review, attr_name)
        if isinstance(value, core_const.SPECIAL_VALUE):
            return False
        is_non_zero = bool(value)
        return need_non_zero == is_non_zero
    return matcher


def make_exact_matcher(attr_name):
    def matcher(person_review, value):
        if not value:
            return True
        attr_value = getattr(person_review, attr_name)
        if isinstance(attr_value, core_const.SPECIAL_VALUE):
            return False
        return attr_value == value
    return matcher


def make_range_matcher(attr_name):
    def matcher(person_review, range):
        if not range:
            return True
        to, from_ = range.get('to'), range.get('from')
        attr_value = getattr(person_review, attr_name)
        if isinstance(attr_value, core_const.SPECIAL_VALUE):
            return False
        if attr_value is None:
            log.warning('PersonReview id=%s has %s=None.', person_review.id, attr_name)
            return False
        if to is not None and from_ is not None:
            return from_ <= attr_value <= to
        elif to is not None:
            return attr_value <= to
        elif from_ is not None:
            return from_ <= attr_value
        else:
            return True
    return matcher


def make_boolean_matcher(attr_name):
    def matcher(person_review, value):
        if value is None:
            return True
        attr_value = getattr(person_review, attr_name)
        if isinstance(attr_value, core_const.SPECIAL_VALUE):
            return False
        return bool(attr_value) == value
    return matcher


def flag_matcher(person_review, value):
    if value is None:
        return True
    attr_vals = (
        getattr(person_review, flag_field)
        for flag_field in core_const.FIELDS.FLAG_FIELDS
    )
    is_flagged = any(
        bool(attr_val) for attr_val in attr_vals
        if not isinstance(attr_val, core_const.SPECIAL_VALUE)
    )
    return is_flagged == value


def make_boolean_decimal_matcher(attr_name):
    def matcher(person_review, value):
        if value is None:
            return True
        attr_value = getattr(person_review, attr_name)
        if isinstance(attr_value, core_const.SPECIAL_VALUE):
            return False
        return (attr_value > 0) == value
    return matcher


def is_departments_matched(person_review, paths):
    if person_review.person_department_path is None:
        # по крайней мере в тестинге такое может быть, не стоит падать
        return False
    for path in paths:
        if person_review.person_department_path.startswith(path):
            return True
    return False


def is_action_at_matched(person_review, persons):
    if person_review.action_at is core_const.NO_ACCESS:
        return False
    action_at_logins = set(p['login'] for p in person_review.action_at)
    return bool(action_at_logins & set(persons))


def has_reviewer(person_review, reviewer_login):

    def reviewers_flat():
        for obj in person_review.reviewers:
            if isinstance(obj, dict):
                yield obj
            else:
                for inner_obj in obj:
                    yield inner_obj

    return any(
        reviewer['login'] == reviewer_login
        for reviewer in reviewers_flat()
    )


# def is_differ_from_default_matched(person_review, is_differ):
#     person_review_is_differ = all([
#         person_review.level_change_is_differ_from_default,
#         person_review.salary_change_is_differ_from_default,
#         person_review.bonus_is_differ_from_default,
#         person_review.options_rsu_is_differ_from_default,
#     ])
#     return person_review_is_differ == is_differ


def subordination_matched(person_review, subordination_type):
    subordination = person_review.subordination
    if subordination_type == staff_const.SUBORDINATION.ANY:
        return subordination in staff_const.SUBORDINATION.ANY_CHOICES
    else:
        return subordination == subordination_type
