# coding: utf-8
"""
TODO: Need to redo rights system to be similar to calibration.
TODO: Now the difference is that permissions are used to describe user can read
TODO: something right now. Need to move those permission into actions
"""
import collections
import logging

from review.lib import functools_utils
from review.core import const
from review.core.const import ROLE, PERSON_REVIEW_STATUS

log = logging.getLogger(__name__)


def is_announced_or_wait_announce(fetched_person_review):
    return fetched_person_review['status'] in (
        PERSON_REVIEW_STATUS.ANNOUNCED,
        PERSON_REVIEW_STATUS.WAIT_ANNOUNCE,
    )


def review_is_active(fetched_review, **kwargs):
    return fetched_review['status'] not in const.REVIEW_STATUS.INACTIVE


def review_product_schema_loaded(person_review, **kwargs):
    return person_review.product_schema_loaded


def review_is_visible(fetched_review):
    return fetched_review['status'] not in const.REVIEW_STATUS.VISIBLE


def check_person_review_transition(person_review, fetched_person_review, transition, **kwargs):
    return _check_person_review_transition(
        person_review=person_review,
        fetched_person_review=fetched_person_review,
        transition=transition
    )


def _check_person_review_transition(person_review, fetched_person_review, transition):
    status = fetched_person_review['status']
    REVIEW_MODE = const.REVIEW_MODE
    is_marks_enabled = person_review.modifiers[REVIEW_MODE.MARK_MODE] == REVIEW_MODE.MODE_MANUAL
    if is_marks_enabled:
        transitions = const.PERSON_REVIEW_ACTIONS.TRANSITIONS_MARKS_ENABLED
    else:
        transitions = const.PERSON_REVIEW_ACTIONS.TRANSITIONS_MARKS_DISABLED
    available_transitions = transitions.get(status)
    return bool(available_transitions) and transition in available_transitions


def person_review_as_self_read(fetched_person_review, **kwargs):
    return is_announced_or_wait_announce(fetched_person_review)


def subject_is_current_or_future_reviewer(fetched_person_review, fetched_reviewers, subject):
    approve_level = fetched_person_review['approve_level']
    reviewers = fetched_reviewers[approve_level:]
    for reviewer in reviewers:
        if isinstance(reviewer, list):
            if subject.login in [r['login'] for r in reviewer]:
                return True
        else:
            if subject.login == reviewer['login']:
                return True
    return False


def person_review_previous_action_at_subject(fetched_person_review, fetched_reviewers, subject):
    approve_level = fetched_person_review['approve_level']

    if approve_level == 0:
        return False

    def check_reviewer(reviewer):
        return reviewer['login'] == subject.login

    if approve_level > len(fetched_reviewers):
        # TODO del this logging after resolving CIA-1528
        try:
            pr_id = fetched_person_review['id']
        except:
            pr_id = 'fail to get person_review.id'
        log.warning('Trying to get reviewer at pos {} for reviewers list {} at pr {}'.format(
            approve_level - 1,
            fetched_reviewers,
            pr_id
        ))

    reviewer = fetched_reviewers[approve_level - 1]
    if isinstance(reviewer, (list, tuple)):
        return any(check_reviewer(r) for r in reviewer)
    else:
        return check_reviewer(reviewer)


def person_review_perm_for_reviewer_write(fetched_review, fetched_person_review, fetched_reviewers, subject, **kwargs):
    return (
        review_is_active(fetched_review) and
        subject_is_current_or_future_reviewer(fetched_person_review, fetched_reviewers, subject) and
        not is_announced_or_wait_announce(fetched_person_review)
    )


def reviewer_can_approve(person_review, fetched_review, fetched_person_review, fetched_reviewers, subject, **kwargs):
    return (
        review_is_active(fetched_review) and
        subject_is_current_or_future_reviewer(fetched_person_review, fetched_reviewers, subject) and
        _check_person_review_transition(person_review, fetched_person_review, const.PERSON_REVIEW_ACTIONS.APPROVE)
    )


def reviewer_can_unapprove(subject, person_review, fetched_person_review, fetched_review, fetched_reviewers, **kwargs):
    return (
        review_is_active(fetched_review) and
        _check_person_review_transition(person_review, fetched_person_review, const.PERSON_REVIEW_ACTIONS.UNAPPROVE) and
        person_review_previous_action_at_subject(fetched_person_review, fetched_reviewers, subject) and
        fetched_person_review['status'] == const.PERSON_REVIEW_STATUS.APPROVAL
    )


def top_reviewer_can_unapprove(subject, person_review, fetched_person_review, fetched_review, fetched_reviewers, **kwargs):
    return (
        review_is_active(fetched_review) and
        _check_person_review_transition(person_review, fetched_person_review, const.PERSON_REVIEW_ACTIONS.UNAPPROVE) and
        fetched_person_review['status'] == const.PERSON_REVIEW_STATUS.APPROVED
    )


class PermissionSet(object):
    def __init__(self, const_perms=None, dynamic_perms=None):
        self.const_perms = const_perms or set()
        self.dynamic_perms = dynamic_perms or {}

    def get_perms(self, perms, **kwargs):
        perms = set(perms)
        result = {
            perm: True
            for perm in self.const_perms & perms
        }
        dynamic_perms = self.dynamic_perms.keys() & perms
        result.update({
            perm: perm_resolver(**kwargs)
            for perm, perm_resolver in self.dynamic_perms.items()
            if perm in dynamic_perms
        })
        result.update({
            perm: False
            for perm in perms - (self.const_perms | self.dynamic_perms.keys())
        })
        return result


PERMS = const.PERSON_REVIEW_PERMS


ROLE_PERMS = {
    ROLE.GLOBAL.ROBOT: PermissionSet(
        const_perms={
            PERMS.MARK_READ,
            PERMS.STATUS_READ,
            PERMS.REVIEWERS_READ,
            PERMS.APPROVE_LEVEL_READ,
            PERMS.MARK_WRITE,
            PERMS.BONUS_READ,
            PERMS.DEFERRED_PAYMENT_READ,
            PERMS.SALARY_CHANGE_READ,
            PERMS.SALARY_CHANGE_WRITE,  # for migrating
            PERMS.FLAGGED_READ,
            PERMS.UMBRELLA_WRITE,  # for freezing umbrellas
            PERMS.MAIN_PRODUCT_WRITE,  # for freezing main_products
        },
        dynamic_perms={
            PERMS.APPROVE: functools_utils.partial(
                check_person_review_transition,
                transition=const.PERSON_REVIEW_ACTIONS.APPROVE
            ),
            PERMS.UNAPPROVE: functools_utils.partial(
                check_person_review_transition,
                transition=const.PERSON_REVIEW_ACTIONS.UNAPPROVE
            ),
            PERMS.ALLOW_ANNOUNCE: functools_utils.partial(
                check_person_review_transition,
                transition=const.PERSON_REVIEW_ACTIONS.ALLOW_ANNOUNCE
            ),
            PERMS.ANNOUNCE: functools_utils.partial(
                check_person_review_transition,
                transition=const.PERSON_REVIEW_ACTIONS.ANNOUNCE
            ),
        },
    ),
    ROLE.GLOBAL.EXPORTER: PermissionSet({
        PERMS.MARK_READ,
        PERMS.LEVEL_CHANGE_READ,
        PERMS.SALARY_CHANGE_READ,
        PERMS.SALARY_READ,
        PERMS.DEFERRED_PAYMENT_READ,
        PERMS.BONUS_READ,
        PERMS.OPTIONS_RSU_READ,
        PERMS.GOLDSTAR_READ,
        PERMS.STATUS_READ,
        PERMS.TAG_AVERAGE_MARK_READ,
        PERMS.UPDATED_AT_READ,
        PERMS.TAKEN_IN_AVERAGE_READ,
    }),
    ROLE.REVIEW.ADMIN: PermissionSet({
        PERMS.STATUS_READ,
        PERMS.APPROVE_LEVEL_READ,
        PERMS.REVIEWERS_READ,
        PERMS.REVIEWERS_WRITE,
        PERMS.TAG_AVERAGE_MARK_READ,
        PERMS.UPDATED_AT_READ,
        PERMS.TAG_AVERAGE_MARK_WRITE,
        PERMS.TAKEN_IN_AVERAGE_READ,
    }),
    ROLE.REVIEW.ACCOMPANYING_HR: PermissionSet(
        const_perms={
            PERMS.STATUS_READ,
            PERMS.APPROVE_LEVEL_READ,
            PERMS.REVIEWERS_READ,
            PERMS.REVIEWERS_WRITE,
            PERMS.UPDATED_AT_READ,
            PERMS.TAG_AVERAGE_MARK_READ,
            PERMS.TAG_AVERAGE_MARK_WRITE,
            PERMS.TAKEN_IN_AVERAGE_READ,
            PERMS.TAKEN_IN_AVERAGE_WRITE,
        },
        dynamic_perms={
            PERMS.UMBRELLA_WRITE: review_is_active,
            PERMS.MAIN_PRODUCT_WRITE: review_is_active,
        },
    ),
    ROLE.REVIEW.SUPERREVIEWER: PermissionSet(
        const_perms={
            PERMS.MARK_READ,
            PERMS.GOLDSTAR_READ,
            PERMS.LEVEL_CHANGE_READ,
            PERMS.SALARY_CHANGE_READ,
            PERMS.BONUS_READ,
            PERMS.DEFERRED_PAYMENT_READ,
            PERMS.OPTIONS_RSU_READ,
            PERMS.TAG_AVERAGE_MARK_READ,
            PERMS.UPDATED_AT_READ,
            PERMS.TAG_AVERAGE_MARK_WRITE,
            PERMS.TAKEN_IN_AVERAGE_READ,
            PERMS.TAKEN_IN_AVERAGE_WRITE,

            PERMS.FLAGGED_READ,
            PERMS.FLAGGED_WRITE,

            PERMS.COMMENTS_READ,
            PERMS.COMMENTS_WRITE,

            PERMS.CHANGES_READ,

            PERMS.SALARY_READ,
            PERMS.LEVEL_READ,
            PERMS.FTE_READ,
            PERMS.MARK_LEVEL_HISTORY_READ,
            PERMS.STATUS_READ,
            PERMS.ACTION_AT_READ,
            PERMS.APPROVE_LEVEL_READ,

            PERMS.REVIEWERS_READ,
            PERMS.REVIEWERS_WRITE,
        },
        dynamic_perms={
            PERMS.MARK_WRITE: review_is_active,
            PERMS.GOLDSTAR_WRITE: review_is_active,
            PERMS.GOLDSTAR_WRITE_CHOSEN: review_is_active,
            PERMS.LEVEL_CHANGE_WRITE: review_is_active,
            PERMS.SALARY_CHANGE_WRITE: review_is_active,
            PERMS.BONUS_WRITE: review_is_active,
            PERMS.DEFERRED_PAYMENT_WRITE: review_is_active,
            PERMS.OPTIONS_RSU_WRITE: review_is_active,
            PERMS.SALARY_CHANGE_AUTO_WRITE: review_is_active,
            PERMS.BONUS_AUTO_WRITE: review_is_active,
            PERMS.DEFERRED_PAYMENT_AUTO_WRITE: review_is_active,
            PERMS.OPTIONS_RSU_AUTO_WRITE: review_is_active,
            PERMS.UMBRELLA_WRITE: review_is_active,
            PERMS.MAIN_PRODUCT_WRITE: review_is_active,

            PERMS.APPROVE: functools_utils.partial(
                check_person_review_transition,
                transition=const.PERSON_REVIEW_ACTIONS.APPROVE
            ),
            PERMS.UNAPPROVE: functools_utils.partial(
                check_person_review_transition,
                transition=const.PERSON_REVIEW_ACTIONS.UNAPPROVE
            ),
            PERMS.ALLOW_ANNOUNCE: functools_utils.partial(
                check_person_review_transition,
                transition=const.PERSON_REVIEW_ACTIONS.ALLOW_ANNOUNCE
            ),
            PERMS.ANNOUNCE: functools_utils.partial(
                check_person_review_transition,
                transition=const.PERSON_REVIEW_ACTIONS.ANNOUNCE
            ),
        }
    ),
    ROLE.PERSON_REVIEW.SUPERREVIEWER_INHERITED: PermissionSet({
        PERMS.MARK_READ,
        PERMS.GOLDSTAR_READ,
        PERMS.LEVEL_CHANGE_READ,
        PERMS.SALARY_CHANGE_READ,
        PERMS.BONUS_READ,
        PERMS.DEFERRED_PAYMENT_READ,
        PERMS.OPTIONS_RSU_READ,
        PERMS.SALARY_READ,
        PERMS.LEVEL_READ,
        PERMS.FTE_READ,
        PERMS.MARK_LEVEL_HISTORY_READ,
        PERMS.STATUS_READ,
        PERMS.ACTION_AT_READ,
        PERMS.COMMENTS_READ,
        PERMS.CHANGES_READ,
        PERMS.UPDATED_AT_READ,
        PERMS.TAG_AVERAGE_MARK_READ,
        PERMS.TAKEN_IN_AVERAGE_READ,
    }),
    ROLE.CALIBRATION.ADMIN: PermissionSet(
        const_perms={
            PERMS.MARK_READ,
            PERMS.GOLDSTAR_READ,
            PERMS.LEVEL_CHANGE_READ,
            PERMS.STATUS_READ,
            PERMS.ACTION_AT_READ,
            PERMS.APPROVE_LEVEL_READ,
            PERMS.REVIEWERS_READ,
            PERMS.MARK_LEVEL_HISTORY_READ,
            PERMS.COMMENTS_WRITE,
            PERMS.COMMENTS_READ,
            PERMS.LEVEL_READ,
            PERMS.CHANGES_READ,
            PERMS.FLAGGED_READ,
            PERMS.FLAGGED_WRITE,
            PERMS.UPDATED_AT_READ,
            PERMS.TAG_AVERAGE_MARK_READ,
            PERMS.TAKEN_IN_AVERAGE_READ,
        },
        dynamic_perms={
            PERMS.LEVEL_CHANGE_WRITE: review_is_active,
            PERMS.MARK_WRITE: review_is_active,
            PERMS.GOLDSTAR_WRITE: review_is_active,
            PERMS.GOLDSTAR_WRITE_CHOSEN: review_is_active,
        },
    ),
    ROLE.PERSON_REVIEW.CALIBRATION_ADMIN_DENORMALIZED: PermissionSet(
        const_perms={
            PERMS.MARK_READ,
            PERMS.GOLDSTAR_READ,
            PERMS.LEVEL_CHANGE_READ,
            PERMS.STATUS_READ,
            PERMS.ACTION_AT_READ,
            PERMS.APPROVE_LEVEL_READ,
            PERMS.REVIEWERS_READ,
            PERMS.MARK_LEVEL_HISTORY_READ,
            PERMS.COMMENTS_WRITE,
            PERMS.COMMENTS_READ,
            PERMS.LEVEL_READ,
            PERMS.CHANGES_READ,
            PERMS.FLAGGED_READ,
            PERMS.FLAGGED_WRITE,
            PERMS.UPDATED_AT_READ,
            PERMS.TAG_AVERAGE_MARK_READ,
            PERMS.TAKEN_IN_AVERAGE_READ,
        },
        dynamic_perms={
            PERMS.LEVEL_CHANGE_WRITE: review_is_active,
            PERMS.MARK_WRITE: review_is_active,
            PERMS.GOLDSTAR_WRITE: review_is_active,
            PERMS.GOLDSTAR_WRITE_CHOSEN: review_is_active,
        },
    ),
    ROLE.PERSON_REVIEW.CALIBRATION_ADMIN_INHERITED: PermissionSet({
        PERMS.MARK_READ,
        PERMS.GOLDSTAR_READ,
        PERMS.LEVEL_CHANGE_READ,
        PERMS.STATUS_READ,
        PERMS.ACTION_AT_READ,
        PERMS.MARK_LEVEL_HISTORY_READ,
        PERMS.COMMENTS_READ,
        PERMS.LEVEL_READ,
        PERMS.CHANGES_READ,
        PERMS.FLAGGED_READ,
        PERMS.UPDATED_AT_READ,
        PERMS.TAG_AVERAGE_MARK_READ,
        PERMS.TAKEN_IN_AVERAGE_READ,
    }),
    ROLE.CALIBRATION.CALIBRATOR: PermissionSet({
        PERMS.MARK_READ,
        PERMS.GOLDSTAR_READ,
        PERMS.LEVEL_CHANGE_READ,
        PERMS.STATUS_READ,
        PERMS.ACTION_AT_READ,
        PERMS.APPROVE_LEVEL_READ,
        PERMS.REVIEWERS_READ,
        PERMS.MARK_LEVEL_HISTORY_READ,
        PERMS.COMMENTS_READ,
        PERMS.LEVEL_READ,
        PERMS.CHANGES_READ,
        PERMS.FLAGGED_READ,
        PERMS.UPDATED_AT_READ,
        PERMS.TAG_AVERAGE_MARK_READ,
        PERMS.TAKEN_IN_AVERAGE_READ,
        PERMS.FLAGGED_WRITE,
    }),
    ROLE.PERSON_REVIEW.CALIBRATOR_DENORMALIZED: PermissionSet({
        PERMS.MARK_READ,
        PERMS.GOLDSTAR_READ,
        PERMS.LEVEL_CHANGE_READ,
        PERMS.STATUS_READ,
        PERMS.ACTION_AT_READ,
        PERMS.MARK_LEVEL_HISTORY_READ,
        PERMS.COMMENTS_READ,
        PERMS.LEVEL_READ,
        PERMS.CHANGES_READ,
        PERMS.FLAGGED_READ,
        PERMS.UPDATED_AT_READ,
        PERMS.TAG_AVERAGE_MARK_READ,
        PERMS.TAKEN_IN_AVERAGE_READ,
        PERMS.FLAGGED_WRITE,
    }),
    ROLE.PERSON_REVIEW.CALIBRATOR_INHERITED: PermissionSet({
        PERMS.MARK_READ,
        PERMS.GOLDSTAR_READ,
        PERMS.LEVEL_CHANGE_READ,
        PERMS.STATUS_READ,
        PERMS.ACTION_AT_READ,
        PERMS.MARK_LEVEL_HISTORY_READ,
        PERMS.COMMENTS_READ,
        PERMS.LEVEL_READ,
        PERMS.CHANGES_READ,
        PERMS.FLAGGED_READ,
        PERMS.UPDATED_AT_READ,
        PERMS.TAG_AVERAGE_MARK_READ,
        PERMS.TAKEN_IN_AVERAGE_READ,
    }),
    ROLE.PERSON_REVIEW.CALIBRATOR_ARCHIVED: PermissionSet(),
    ROLE.DEPARTMENT.HEAD: PermissionSet(
        const_perms={
            PERMS.MARK_READ,
            PERMS.GOLDSTAR_READ,
            PERMS.LEVEL_CHANGE_READ,
            PERMS.SALARY_CHANGE_READ,
            PERMS.BONUS_READ,
            PERMS.DEFERRED_PAYMENT_READ,
            PERMS.OPTIONS_RSU_READ,
            PERMS.SALARY_READ,
            PERMS.LEVEL_READ,
            PERMS.FTE_READ,
            PERMS.MARK_LEVEL_HISTORY_READ,
            PERMS.STATUS_READ,
            PERMS.ACTION_AT_READ,
            PERMS.APPROVE_LEVEL_READ,
            PERMS.COMMENTS_READ,
            PERMS.CHANGES_READ,
            PERMS.UPDATED_AT_READ,
            PERMS.TAG_AVERAGE_MARK_READ,
            PERMS.TAKEN_IN_AVERAGE_READ,
        },
        dynamic_perms={
            PERMS.ANNOUNCE: functools_utils.partial(
                check_person_review_transition,
                transition=const.PERSON_REVIEW_ACTIONS.ANNOUNCE
            )
        }
    ),
    ROLE.DEPARTMENT.HR_PARTNER: PermissionSet(
        const_perms={
            PERMS.MARK_READ,
            PERMS.GOLDSTAR_READ,
            PERMS.STATUS_READ,
            PERMS.UPDATED_AT_READ,
            PERMS.TAG_AVERAGE_MARK_READ,
            PERMS.TAG_AVERAGE_MARK_WRITE,
            PERMS.TAKEN_IN_AVERAGE_READ,
        },
    ),
    ROLE.DEPARTMENT.HR_ANALYST: PermissionSet(
        const_perms={
            PERMS.MARK_READ,
            PERMS.GOLDSTAR_READ,
            PERMS.LEVEL_CHANGE_READ,
            PERMS.SALARY_CHANGE_READ,
            PERMS.BONUS_READ,
            PERMS.DEFERRED_PAYMENT_READ,
            PERMS.OPTIONS_RSU_READ,
            PERMS.FLAGGED_READ,
            PERMS.CHANGES_READ,
            PERMS.UPDATED_AT_READ,
            PERMS.TAG_AVERAGE_MARK_READ,
            PERMS.TAG_AVERAGE_MARK_WRITE,
            PERMS.TAKEN_IN_AVERAGE_READ,

            PERMS.COMMENTS_READ,
            PERMS.COMMENTS_WRITE,

            PERMS.SALARY_READ,
            PERMS.LEVEL_READ,
            PERMS.FTE_READ,
            PERMS.MARK_LEVEL_HISTORY_READ,
            PERMS.STATUS_READ,
            PERMS.ACTION_AT_READ,
            PERMS.APPROVE_LEVEL_READ,
            PERMS.REVIEWERS_READ,
        },
        dynamic_perms={
            PERMS.MARK_WRITE: review_is_active,
            PERMS.GOLDSTAR_WRITE: review_is_active,
            PERMS.GOLDSTAR_WRITE_CHOSEN: review_is_active,
            PERMS.LEVEL_CHANGE_WRITE: review_is_active,
            PERMS.SALARY_CHANGE_WRITE: review_is_active,
            PERMS.BONUS_WRITE: review_is_active,
            PERMS.DEFERRED_PAYMENT_WRITE: review_is_active,
            PERMS.OPTIONS_RSU_WRITE: review_is_active,
            PERMS.SALARY_CHANGE_AUTO_WRITE: review_is_active,
            PERMS.BONUS_AUTO_WRITE: review_is_active,
            PERMS.DEFERRED_PAYMENT_AUTO_WRITE: review_is_active,
            PERMS.OPTIONS_RSU_AUTO_WRITE: review_is_active,

            PERMS.APPROVE: functools_utils.partial(
                check_person_review_transition,
                transition=const.PERSON_REVIEW_ACTIONS.APPROVE
            ),
            PERMS.UNAPPROVE: functools_utils.partial(
                check_person_review_transition,
                transition=const.PERSON_REVIEW_ACTIONS.UNAPPROVE
            ),
            PERMS.ALLOW_ANNOUNCE: functools_utils.partial(
                check_person_review_transition,
                transition=const.PERSON_REVIEW_ACTIONS.ALLOW_ANNOUNCE
            ),
            PERMS.ANNOUNCE: functools_utils.partial(
                check_person_review_transition,
                transition=const.PERSON_REVIEW_ACTIONS.ANNOUNCE
            ),
        },
    ),
    ROLE.PERSON.SELF: PermissionSet(
        const_perms={
            PERMS.SALARY_READ,
            PERMS.LEVEL_READ,
            PERMS.FTE_READ,
            PERMS.MARK_LEVEL_HISTORY_READ,
            PERMS.TAG_AVERAGE_MARK_READ,
        },
        dynamic_perms={
            PERMS.MARK_READ: person_review_as_self_read,
            PERMS.GOLDSTAR_READ: person_review_as_self_read,
            PERMS.LEVEL_CHANGE_READ: person_review_as_self_read,
            PERMS.SALARY_CHANGE_READ: person_review_as_self_read,
            PERMS.BONUS_READ: person_review_as_self_read,
            PERMS.DEFERRED_PAYMENT_READ: person_review_as_self_read,
            PERMS.OPTIONS_RSU_READ: person_review_as_self_read,
            PERMS.STATUS_READ: person_review_as_self_read,
        }
    ),
    ROLE.PERSON_REVIEW.REVIEWER: PermissionSet(
        const_perms={
            PERMS.MARK_READ,
            PERMS.GOLDSTAR_READ,
            PERMS.LEVEL_CHANGE_READ,
            PERMS.SALARY_CHANGE_READ,
            PERMS.BONUS_READ,
            PERMS.DEFERRED_PAYMENT_READ,
            PERMS.OPTIONS_RSU_READ,
            PERMS.FLAGGED_READ,
            PERMS.COMMENTS_READ,
            PERMS.LEVEL_READ,
            PERMS.FTE_READ,
            PERMS.MARK_LEVEL_HISTORY_READ,
            PERMS.STATUS_READ,
            PERMS.ACTION_AT_READ,
            PERMS.APPROVE_LEVEL_READ,
            PERMS.COMMENTS_WRITE,
            PERMS.CHANGES_READ,
            PERMS.REVIEWERS_READ,
            PERMS.UPDATED_AT_READ,
            PERMS.TAG_AVERAGE_MARK_READ,
            PERMS.TAKEN_IN_AVERAGE_READ,
        },
        dynamic_perms={
            PERMS.MARK_WRITE: person_review_perm_for_reviewer_write,
            PERMS.GOLDSTAR_WRITE: person_review_perm_for_reviewer_write,
            PERMS.LEVEL_CHANGE_WRITE: person_review_perm_for_reviewer_write,
            PERMS.SALARY_CHANGE_WRITE: person_review_perm_for_reviewer_write,
            PERMS.BONUS_WRITE: person_review_perm_for_reviewer_write,
            PERMS.DEFERRED_PAYMENT_WRITE: review_is_active,
            PERMS.OPTIONS_RSU_WRITE: person_review_perm_for_reviewer_write,
            PERMS.FLAGGED_WRITE: person_review_perm_for_reviewer_write,

            PERMS.APPROVE: reviewer_can_approve,
            PERMS.UNAPPROVE: reviewer_can_unapprove,
            PERMS.ANNOUNCE: functools_utils.partial(
                check_person_review_transition,
                transition=const.PERSON_REVIEW_ACTIONS.ANNOUNCE
            ),
        }
    ),
    ROLE.PERSON_REVIEW.REVIEWER_INHERITED: PermissionSet({
        PERMS.MARK_READ,
        PERMS.GOLDSTAR_READ,
        PERMS.LEVEL_CHANGE_READ,
        PERMS.LEVEL_READ,
        PERMS.FTE_READ,
        PERMS.MARK_LEVEL_HISTORY_READ,
        PERMS.STATUS_READ,
        PERMS.ACTION_AT_READ,
        PERMS.COMMENTS_READ,
        PERMS.CHANGES_READ,
        PERMS.UPDATED_AT_READ,
        PERMS.TAG_AVERAGE_MARK_READ,
        PERMS.TAKEN_IN_AVERAGE_READ,
    }),
    ROLE.PERSON_REVIEW.TOP_REVIEWER: PermissionSet(
        const_perms={
            PERMS.MARK_READ,
            PERMS.GOLDSTAR_READ,
            PERMS.LEVEL_CHANGE_READ,
            PERMS.SALARY_CHANGE_READ,
            PERMS.BONUS_READ,
            PERMS.DEFERRED_PAYMENT_READ,
            PERMS.OPTIONS_RSU_READ,
            PERMS.FLAGGED_READ,
            PERMS.COMMENTS_READ,
            PERMS.SALARY_READ,
            PERMS.LEVEL_READ,
            PERMS.FTE_READ,
            PERMS.MARK_LEVEL_HISTORY_READ,
            PERMS.STATUS_READ,
            PERMS.ACTION_AT_READ,
            PERMS.APPROVE_LEVEL_READ,
            PERMS.COMMENTS_WRITE,
            PERMS.CHANGES_READ,
            PERMS.REVIEWERS_READ,
            PERMS.UPDATED_AT_READ,
            PERMS.TAG_AVERAGE_MARK_READ,
            PERMS.TAG_AVERAGE_MARK_WRITE,
            PERMS.TAKEN_IN_AVERAGE_READ,
        },
        dynamic_perms={
            PERMS.MARK_WRITE: person_review_perm_for_reviewer_write,
            PERMS.GOLDSTAR_WRITE: person_review_perm_for_reviewer_write,
            PERMS.GOLDSTAR_WRITE_CHOSEN: person_review_perm_for_reviewer_write,
            PERMS.LEVEL_CHANGE_WRITE: review_is_active,
            PERMS.SALARY_CHANGE_WRITE: review_is_active,
            PERMS.BONUS_WRITE: review_is_active,
            PERMS.DEFERRED_PAYMENT_WRITE: review_is_active,
            PERMS.OPTIONS_RSU_WRITE: review_is_active,
            PERMS.SALARY_CHANGE_AUTO_WRITE: review_is_active,
            PERMS.BONUS_AUTO_WRITE: review_is_active,
            PERMS.DEFERRED_PAYMENT_AUTO_WRITE: review_is_active,
            PERMS.OPTIONS_RSU_AUTO_WRITE: review_is_active,
            PERMS.FLAGGED_WRITE: person_review_perm_for_reviewer_write,

            PERMS.APPROVE: reviewer_can_approve,
            PERMS.UNAPPROVE: top_reviewer_can_unapprove,
            PERMS.ALLOW_ANNOUNCE: functools_utils.partial(
                check_person_review_transition,
                transition=const.PERSON_REVIEW_ACTIONS.ALLOW_ANNOUNCE
            ),
            PERMS.ANNOUNCE: functools_utils.partial(
                check_person_review_transition,
                transition=const.PERSON_REVIEW_ACTIONS.ANNOUNCE
            ),
        }
    ),
    ROLE.PERSON_REVIEW.TOP_REVIEWER_INHERITED: PermissionSet({
        PERMS.MARK_READ,
        PERMS.GOLDSTAR_READ,
        PERMS.LEVEL_CHANGE_READ,
        PERMS.SALARY_CHANGE_READ,
        PERMS.BONUS_READ,
        PERMS.DEFERRED_PAYMENT_READ,
        PERMS.OPTIONS_RSU_READ,
        PERMS.LEVEL_READ,
        PERMS.FTE_READ,
        PERMS.MARK_LEVEL_HISTORY_READ,
        PERMS.STATUS_READ,
        PERMS.ACTION_AT_READ,
        PERMS.COMMENTS_READ,
        PERMS.SALARY_READ,
        PERMS.CHANGES_READ,
        PERMS.UPDATED_AT_READ,
        PERMS.TAG_AVERAGE_MARK_READ,
        PERMS.TAKEN_IN_AVERAGE_READ,
    }),
    ROLE.PERSON_REVIEW.READER: PermissionSet({
        PERMS.MARK_READ,
        PERMS.GOLDSTAR_READ,
        PERMS.LEVEL_CHANGE_READ,
        PERMS.MARK_LEVEL_HISTORY_READ,
        PERMS.STATUS_READ,
        PERMS.ACTION_AT_READ,
        PERMS.APPROVE_LEVEL_READ,
        PERMS.COMMENTS_READ,
        PERMS.CHANGES_READ,
        PERMS.UPDATED_AT_READ,
        PERMS.TAG_AVERAGE_MARK_READ,
        PERMS.TAKEN_IN_AVERAGE_READ,
    }),
    ROLE.PERSON_REVIEW.READER_INHERITED: PermissionSet({
        PERMS.MARK_READ,
        PERMS.GOLDSTAR_READ,
        PERMS.LEVEL_CHANGE_READ,
        PERMS.MARK_LEVEL_HISTORY_READ,
        PERMS.STATUS_READ,
        PERMS.ACTION_AT_READ,
        PERMS.COMMENTS_READ,
        PERMS.CHANGES_READ,
        PERMS.UPDATED_AT_READ,
        PERMS.TAG_AVERAGE_MARK_READ,
        PERMS.TAKEN_IN_AVERAGE_READ,
    }),
    ROLE.PERSON_REVIEW.SUPERREADER: PermissionSet({
        PERMS.MARK_READ,
        PERMS.GOLDSTAR_READ,
        PERMS.LEVEL_CHANGE_READ,
        PERMS.SALARY_CHANGE_READ,
        PERMS.BONUS_READ,
        PERMS.DEFERRED_PAYMENT_READ,
        PERMS.OPTIONS_RSU_READ,
        PERMS.SALARY_READ,
        PERMS.FTE_READ,
        PERMS.LEVEL_READ,
        PERMS.MARK_LEVEL_HISTORY_READ,
        PERMS.STATUS_READ,
        PERMS.ACTION_AT_READ,
        PERMS.APPROVE_LEVEL_READ,
        PERMS.COMMENTS_READ,
        PERMS.CHANGES_READ,
        PERMS.UPDATED_AT_READ,
        PERMS.TAG_AVERAGE_MARK_READ,
        PERMS.TAKEN_IN_AVERAGE_READ,
    }),
    ROLE.PERSON_REVIEW.SUPERREADER_INHERITED: PermissionSet({
        PERMS.MARK_READ,
        PERMS.GOLDSTAR_READ,
        PERMS.LEVEL_CHANGE_READ,
        PERMS.SALARY_CHANGE_READ,
        PERMS.BONUS_READ,
        PERMS.DEFERRED_PAYMENT_READ,
        PERMS.OPTIONS_RSU_READ,
        PERMS.SALARY_READ,
        PERMS.FTE_READ,
        PERMS.LEVEL_READ,
        PERMS.MARK_LEVEL_HISTORY_READ,
        PERMS.STATUS_READ,
        PERMS.ACTION_AT_READ,
        PERMS.COMMENTS_READ,
        PERMS.CHANGES_READ,
        PERMS.UPDATED_AT_READ,
        PERMS.TAG_AVERAGE_MARK_READ,
        PERMS.TAKEN_IN_AVERAGE_READ,
    })
}


def get_permissions(person_review, fetched, subject, perms=None, **kwargs):
    if perms is None:
        perms = PERMS.ALL
    perms_for_roles = [
        get_permissions_for_role(
            role=role,
            perms=perms,
            person_review=person_review,
            fetched_person_review=fetched.person_reviews[person_review.id],
            fetched_review=fetched.reviews[person_review.review_id],
            fetched_reviewers=fetched.reviewers.get(person_review.id, []),
            subject=subject,
        )
        for role in person_review.roles
    ]

    perms_data = collections.defaultdict(list)
    for perms_for_role in perms_for_roles:
        for perm, value in perms_for_role.items():
            perms_data[perm].append(value)

    for perm, values in perms_data.items():
        perms_data[perm] = max(values)

    return perms_data


def get_permissions_for_role(role, perms, **kwargs):
    return ROLE_PERMS[role].get_perms(perms, **kwargs)
