# coding: utf-8
"""
Rights system is similar to calibration
Described in review.core.logic.calibration_rights
"""
from itertools import chain

from review.lib import errors
from review.core import const


def set_permissions(review):
    review.permissions = get_permissions(review)


def set_actions(review):
    review.actions = get_actions(review)


def clear_unavailable_fields(review):
    if const.REVIEW_PERMS.LIST_SUPERREVIEWERS not in review.permissions:
        review.super_reviewers = []
    if const.REVIEW_PERMS.LIST_ADMINS not in review.permissions:
        review.admins = []
    if const.REVIEW_PERMS.LIST_ACCOMPANYING_HRS not in review.permissions:
        review.accompanying_hrs = []


def get_permissions(review):
    return set(chain.from_iterable(
        const.REVIEW_PERMS.ROLE_PERMS[role] for role in review.roles
    ))


def get_actions(review):
    assert review.permissions is not None, 'Set permissions before getting actions'
    assert review.status is not None, 'Status is needed to set transition actions'
    actions = {
        action: _get_action_status(action, review.permissions)
        for action in const.REVIEW_ACTIONS.ALL
    }
    allowed_transitions = const.REVIEW_ACTIONS.STATUS_TO_TRANSITIONS.get(review.status, set())
    actions.update({
        act: const.NO_ACCESS
        for act in const.REVIEW_ACTIONS.STATUS
        if act not in allowed_transitions
    })
    return actions


def _get_action_status(action, current_permissions):
    perms, actions = const.REVIEW_PERMS, const.REVIEW_ACTIONS
    action_to_perm = {
        actions.ADD_PERSONS: perms.ADD_PERSONS,
        actions.DELETE_PERSONS: perms.DELETE_PERSONS,
        actions.ADD_ANY_PERSONS: perms.ADD_ANY_PERSONS,
        actions.DELETE_ANY_PERSONS: perms.DELETE_ANY_PERSONS,
        actions.EDIT_ADMINS: perms.EDIT_ADMINS,
        actions.EDIT_SUPERREVIEWERS: perms.EDIT_SUPERREVIEWERS,
        actions.EDIT_ACCOMPANYING_HRS: perms.EDIT_ACCOMPANYING_HRS,
        actions.EDIT: perms.EDIT,
        actions.LIST_GOODIES: perms.LIST_GOODIES,
        actions.STATUS_ARCHIVE: perms.STATUS_ARCHIVE,
        actions.STATUS_UNARCHIVE: perms.STATUS_UNARCHIVE,
        actions.STATUS_IN_DRAFT: perms.STATUS_IN_DRAFT,
        actions.STATUS_PUBLISH: perms.STATUS_PUBLISH,
        actions.LOAD_PRODUCT_SCHEME: perms.LOAD_PRODUCT_SCHEME,
        actions.VIEW_PARTICIPANTS: perms.VIEW_PARTICIPANTS,
        actions.LOAD_SECRET_TEMPLATE: perms.LOAD_SECRET_TEMPLATE,
        actions.LOAD_KPI: perms.LOAD_KPI,
    }
    if action_to_perm[action] in current_permissions:
        return const.OK
    else:
        return const.NO_ACCESS


def ensure_has_action(review, actions):
    if review is None:
        raise errors.PermissionDenied(type='review')

    for action in actions:
        if review.actions[action] == const.OK:
            return True
    raise errors.PermissionDenied(
        type='review',
        perm=action,
        id=review.id
    )
