# coding: utf-8
import functools
import itertools
import operator
import logging

import yenv

from review.core import const
from review.core.logic.assemble import permissions_person_review
from review.lib import datetimes
from review.lib import iterables
from review.oebs import const as oebs_const
from review.core.logic import cia

log = logging.getLogger(__name__)


FIELDS = const.FIELDS
PERMS = const.PERSON_REVIEW_PERMS


def set_field(field, person_reviews, fetched, **kwargs):
    bulk_setter = globals().get('bulk_set_' + field)
    if bulk_setter is not None:
        bulk_setter(person_reviews, fetched, **kwargs)
        return

    getter = globals().get('get_' + field)
    if getter is None:
        return
    for person_review in person_reviews:
        try:
            value = getter(person_review, fetched, **kwargs)
        except Exception:
            if yenv.type == 'development':
                raise
            log.exception('set field %s failed', field)
            value = const.ERROR
        setattr(person_review, field, value)


# subject fields
def bulk_set_roles(person_reviews, fetched, **kwargs):
    pr_to_roles = {}
    for role_type, person_reviews_group in fetched.grouped_by_roles.items():
        for person_review in person_reviews_group:
            roles = pr_to_roles.setdefault(person_review, set())
            roles.add(role_type)

    for person_review in person_reviews:
        person_review.roles = list(set(pr_to_roles[person_review]))


def get_modifiers(person_review, fetched, **kwargs):
    MODE = const.REVIEW_MODE
    review_data = fetched.reviews[person_review.review_id]
    result = {}
    for field, modifier_name in MODE.FIELDS_TO_MODIFIERS.items():
        result[modifier_name] = review_data[modifier_name]
    return result


def _get_permissions(perms):
    def getter(person_review, fetched, subject, **kwargs):
        return permissions_person_review.get_permissions(person_review, fetched, subject, perms, **kwargs)
    return getter


get_permissions_read = _get_permissions(PERMS.READ)
get_permissions_write = _get_permissions(PERMS.WRITE)


def get_subordination(person_review, fetched, **kwargs):
    return fetched.subordination.get(person_review.person_id)


# person_review
def _get_person_review_fields_modifiable_getter(field):
    def getter(person_review, fetched, **kwargs):
        modifier_for_field = const.REVIEW_MODE.FIELDS_TO_MODIFIERS[field]
        if person_review.modifiers[modifier_for_field] == const.REVIEW_MODE.MODE_DISABLED:
            return const.DISABLED
        if not person_review.permissions_read[PERMS.FIELD_TO_PERMISSION[field]]:
            return const.NO_ACCESS
        return fetched.person_reviews[person_review.id][field]
    return getter


def _get_bonus_fields(field):
    value_getter = _get_person_review_fields_modifiable_getter(field)

    def getter(person_review, fetched, **kwargs):
        value = value_getter(person_review, fetched, **kwargs)
        is_special = isinstance(value, const.SPECIAL_VALUE)
        has_salary_perm = person_review.permissions_read[PERMS.SALARY_READ]
        has_write_perm = any(
            person_review.permissions_write[p]
            for p in (PERMS.BONUS_WRITE, PERMS.BONUS_AUTO_WRITE)
        )
        if is_special or has_salary_perm or has_write_perm:
            return value
        available_to_see = {
            const.SALARY_DEPENDENCY_TYPE.PERCENTAGE: const.FIELDS.BONUS,
            const.SALARY_DEPENDENCY_TYPE.ABSOLUTE: const.FIELDS.BONUS_ABSOLUTE,
        }
        if available_to_see[person_review.bonus_type] != field:
            return const.NO_ACCESS
        return value
    return getter


def _get_salary_fields(field):
    value_getter = _get_person_review_fields_modifiable_getter(field)

    def getter(person_review, fetched, **kwargs):
        value = value_getter(person_review, fetched, **kwargs)
        is_special = isinstance(value, const.SPECIAL_VALUE)
        has_salary_perm = person_review.permissions_read[PERMS.SALARY_READ]
        has_write_perm = any(
            person_review.permissions_write[p]
            for p in (PERMS.SALARY_CHANGE_WRITE, PERMS.SALARY_CHANGE_AUTO_WRITE)
        )
        if is_special or has_salary_perm or has_write_perm:
            return value
        available_to_see = {
            const.SALARY_DEPENDENCY_TYPE.PERCENTAGE: const.FIELDS.SALARY_CHANGE,
            const.SALARY_DEPENDENCY_TYPE.ABSOLUTE: const.FIELDS.SALARY_CHANGE_ABSOLUTE,
        }
        if available_to_see[person_review.salary_change_type] != field:
            return const.NO_ACCESS
        return value
    return getter


def _get_person_review_fields_simple_getter(field):
    def getter(person_review, fetched, **kwargs):
        if not person_review.permissions_read[PERMS.FIELD_TO_PERMISSION[field]]:
            return const.NO_ACCESS
        return fetched.person_reviews[person_review.id][field]
    return getter


get_status = _get_person_review_fields_simple_getter(FIELDS.STATUS)
get_approve_level = _get_person_review_fields_simple_getter(FIELDS.APPROVE_LEVEL)
get_flagged = _get_person_review_fields_simple_getter(FIELDS.FLAGGED)
get_bonus_type = _get_person_review_fields_simple_getter(FIELDS.BONUS_TYPE)
get_salary_change_type = _get_person_review_fields_simple_getter(FIELDS.SALARY_CHANGE_TYPE)
get_flagged_positive = _get_person_review_fields_simple_getter(FIELDS.FLAGGED_POSITIVE)
get_tag_average_mark = _get_person_review_fields_simple_getter(FIELDS.TAG_AVERAGE_MARK)
get_taken_in_average = _get_person_review_fields_simple_getter(FIELDS.TAKEN_IN_AVERAGE)
get_updated_at = _get_person_review_fields_simple_getter(FIELDS.UPDATED_AT)
get_mark = _get_person_review_fields_modifiable_getter(FIELDS.MARK)
get_goldstar = _get_person_review_fields_modifiable_getter(FIELDS.GOLDSTAR)
get_level_change = _get_person_review_fields_modifiable_getter(FIELDS.LEVEL_CHANGE)
get_salary_change = _get_salary_fields(FIELDS.SALARY_CHANGE)
get_salary_change_absolute = _get_salary_fields(FIELDS.SALARY_CHANGE_ABSOLUTE)
get_bonus = _get_bonus_fields(FIELDS.BONUS)
get_bonus_absolute = _get_bonus_fields(FIELDS.BONUS_ABSOLUTE)
get_bonus_rsu = _get_person_review_fields_modifiable_getter(FIELDS.BONUS_RSU)
get_deferred_payment = _get_person_review_fields_modifiable_getter(FIELDS.DEFERRED_PAYMENT)
get_options_rsu = _get_person_review_fields_modifiable_getter(FIELDS.OPTIONS_RSU)
get_options_rsu_legacy = _get_person_review_fields_modifiable_getter(FIELDS.OPTIONS_RSU_LEGACY)


# persons
def _person_data_getter(field):
    def getter(person_review, fetched, **kwargs):
        person_data = fetched.persons[person_review.person_id]
        return person_data[FIELDS.DB_PERSON_FIELDS_MAP[field]]
    return getter


get_person_login = _person_data_getter(FIELDS.PERSON_LOGIN)
get_person_first_name = _person_data_getter(FIELDS.PERSON_FIRST_NAME)
get_person_last_name = _person_data_getter(FIELDS.PERSON_LAST_NAME)
get_person_is_dismissed = _person_data_getter(FIELDS.PERSON_IS_DISMISSED)
get_person_join_at = _person_data_getter(FIELDS.PERSON_JOIN_AT)
get_person_position = _person_data_getter(FIELDS.PERSON_POSITION)
get_person_city_name = _person_data_getter(FIELDS.PERSON_CITY_NAME)
get_person_department_id = _person_data_getter(FIELDS.PERSON_DEPARTMENT_ID)
get_person_department_slug = _person_data_getter(FIELDS.PERSON_DEPARTMENT_SLUG)
get_person_department_name = _person_data_getter(FIELDS.PERSON_DEPARTMENT_NAME)
get_person_department_path = _person_data_getter(FIELDS.PERSON_DEPARTMENT_PATH)
get_person_gender = _person_data_getter(FIELDS.PERSON_GENDER)


def get_umbrella(person_review, fetched, **kwargs):
    if not person_review.umbrella_id:
        return None
    umbrella = fetched.umbrellas[person_review.umbrella_id]

    umbrella_data = {
        'id': umbrella['id'],
        'name': umbrella['name'],
    }
    if umbrella['main_product_id']:
        main_product = fetched.main_products[umbrella['main_product_id']]
        umbrella_data['main_product'] = {
            'id': main_product['id'],
            'name': main_product['name'],
            'abc_service_id': main_product['abc_service_id'],
        }
    else:
        umbrella_data['main_product'] = None

    return umbrella_data


def get_main_product(person_review, fetched, **kwargs):
    if not person_review.main_product_id:
        return None

    main_product = fetched.main_products[person_review.main_product_id]

    return {
        'id': main_product['id'],
        'name': main_product['name'],
        'abc_service_id': main_product['abc_service_id'],
    }


def get_product_schema_loaded(person_review, fetched, **kwargs):
    return person_review.review_id in fetched.product_schema_loaded


# finance
def get_salary_closest_to_review_start(person_review, fetched, **kwargs):
    event = _get_salary_event(person_review, fetched, **kwargs)
    if not event:
        return const.EMPTY
    return event


def get_salary_value(person_review, fetched, **kwargs):
    if not person_review.permissions_read[PERMS.SALARY_READ]:
        return const.NO_ACCESS
    closest_salary = person_review.salary_closest_to_review_start
    if closest_salary == const.EMPTY:
        return const.EMPTY
    return closest_salary['value']


def get_salary_currency(person_review, fetched, **kwargs):
    if not person_review.permissions_read[PERMS.SALARY_READ]:
        return const.NO_ACCESS
    closest_salary = person_review.salary_closest_to_review_start
    if closest_salary == const.EMPTY:
        return const.EMPTY
    return closest_salary['currency']


def get_salary_after_review(person_review, fetched, **kwargs):
    if not person_review.permissions_read[PERMS.SALARY_READ]:
        return const.NO_ACCESS
    to_sum = (person_review.salary_value, person_review.salary_change_absolute)
    empty_val = next(
        (it for it in to_sum if isinstance(it, const.SPECIAL_VALUE)),
        None
    )
    if empty_val is not None:
        return empty_val
    return sum([float(val) for val in to_sum])


def _get_salary_event(person_review, fetched, **kwargs):
    person_review_events = fetched.finance_events[person_review.id]
    event = person_review_events.get(oebs_const.SALARY_HISTORY)
    return event and {
        'value': event['salarySum'],
        'currency': event['currency'],
        'date': event['dateFrom'],
    }


def get_grade_closest_to_review_start(person_review, fetched, **kwargs):
    grade_event = _get_grade_event(person_review, fetched)
    if grade_event is None:
        return const.EMPTY
    return grade_event


def get_level(person_review, fetched, **kwargs):
    if not person_review.permissions_read[PERMS.LEVEL_READ]:
        return const.NO_ACCESS
    closest_grade = person_review.grade_closest_to_review_start
    if closest_grade == const.EMPTY:
        return const.EMPTY
    return closest_grade[FIELDS.LEVEL]


def get_profession(person_review, fetched, **kwargs):
    closest_grade = person_review.grade_closest_to_review_start
    if closest_grade == const.EMPTY:
        return const.EMPTY
    return closest_grade['profession']


def _get_grade_event(person_review, fetched, **kwargs):
    person_review_events = fetched.finance_events[person_review.id]
    event = person_review_events.get(oebs_const.GRADE_HISTORY)
    if not event:
        return
    if event['gradeName']:
        profession, level, scale = event['gradeName'].split('.')
    else:
        profession, level, scale = '', None, None
    try:
        # может быть строка «Без грейда»
        level = int(level)
    except Exception:
        level = None
    return {
        'profession': profession,
        'level': level,
        'scale': scale,
        'date': event['dateFrom'],
    }


def get_fte(person_review, fetched, **kwargs):
    if not person_review.permissions_read[PERMS.FTE_READ]:
        return const.NO_ACCESS
    if person_review.person_id not in fetched.current_salaries:
        return const.EMPTY
    current_salary_data = fetched.current_salaries[person_review.person_id]
    return current_salary_data['current_salary'].get('factFTE', const.EMPTY)


# reviews
def _review_data_getter(field):
    def getter(person_review, fetched, **kwargs):
        review_data = fetched.reviews[person_review.review_id]
        return review_data[FIELDS.DB_REVIEW_FIELDS_MAP[field]]
    return getter


get_review_name = _review_data_getter(FIELDS.REVIEW_NAME)
get_review_type = _review_data_getter(FIELDS.REVIEW_TYPE)
get_review_scale_id = _review_data_getter(FIELDS.REVIEW_SCALE_ID)
get_review_salary_date = _review_data_getter(FIELDS.REVIEW_SALARY_DATE)
get_review_status = _review_data_getter(FIELDS.REVIEW_STATUS)
get_review_start_date = _review_data_getter(FIELDS.REVIEW_START_DATE)
get_review_finish_date = _review_data_getter(FIELDS.REVIEW_FINISH_DATE)
get_review_evaluation_from_date = _review_data_getter(FIELDS.REVIEW_EVALUATION_FROM_DATE)
get_review_evaluation_to_date = _review_data_getter(FIELDS.REVIEW_EVALUATION_TO_DATE)
get_review_feedback_from_date = _review_data_getter(FIELDS.REVIEW_FEEDBACK_FROM_DATE)
get_review_feedback_to_date = _review_data_getter(FIELDS.REVIEW_FEEDBACK_TO_DATE)
get_review_options_rsu_format = _review_data_getter(FIELDS.REVIEW_OPTIONS_RSU_FORMAT)
get_review_options_rsu_unit = _review_data_getter(FIELDS.REVIEW_OPTIONS_RSU_UNIT)
get_review_kpi_loaded = _review_data_getter(FIELDS.REVIEW_KPI_LOADED)


def get_review_marks_scale(person_review, fetched, **kwargs):
    return fetched.marks_scales.get(person_review.review_scale_id, {})


# complex
def bulk_set_short_history(person_reviews, fetched, subject, role_types, **kwargs):
    short_histories = {}
    for person_review in person_reviews:
        review = fetched.reviews[person_review.review_id]
        review_start_date = review['start_date']
        full_history = iterables.PersonReviewHistory(
            fetched.person_review_history[person_review.person_id]
        )
        history_for_year_before = full_history.get_range(
            from_date=datetimes.shifted(review_start_date, years=-2),
            to_date=review_start_date,
        )
        short_histories[person_review.id] = [
            pr['id'] for pr in history_for_year_before if
            pr['review_type'] == const.REVIEW_TYPE.NORMAL
        ][-const.MARK_LEVEL_HISTORY_RANGE:]

    history_person_review_ids = set(itertools.chain.from_iterable(
        list(short_histories.values()),
    ))
    if not history_person_review_ids:
        for pr in person_reviews:
            pr.short_history = []
        return

    from review.core.logic.assemble import get_person_reviews
    if not role_types:
        role_types = const.ROLE.PERSON_REVIEW_LIST_RELATED
    inherited_roles = {
        const.ROLE.INHERITANCE[r] for r in role_types
        if r in const.ROLE.INHERITANCE
    }
    allowed_person_reviews = get_person_reviews(
        subject=subject,
        filters_chosen={const.FILTERS.IDS: history_person_review_ids},
        fields_requested=(
            FIELDS.MARK,
            FIELDS.BONUS,
            FIELDS.LEVEL,
            FIELDS.GOLDSTAR,
            FIELDS.LEVEL_CHANGE,
            FIELDS.REVIEW_SCALE_ID,
            FIELDS.REVIEW_MARKS_SCALE,
        ),
        role_types=set(role_types) | inherited_roles
    )
    id_to_person_review = {pr.id: pr for pr in allowed_person_reviews}

    for person_review in person_reviews:
        history_person_reviews = (
            id_to_person_review[id_]
            for id_ in short_histories[person_review.id]
            if id_ in id_to_person_review
        )
        person_review.short_history = [
            dict(
                mark=pr.mark,
                level=pr.level,
                level_change=pr.level_change,
                bonus=pr.bonus,
                goldstar=pr.goldstar,
                scale_id=pr.review_scale_id,
                scale=pr.review_marks_scale,
            )
            for pr in history_person_reviews
        ]


def get_mark_level_history(person_review, fetched, **kwargs):
    return [
        {
            'mark': item['mark'],
            'level': item['level'],
            'level_change': item['level_change'],
            'scale_id': item['scale_id'],
        }
        for item in person_review.short_history
    ]


def _is_differ_from_default_getter(field):
    def getter(person_review, fetched, **kwargs):
        key = (
            person_review.review_id,
            person_review.level,
            person_review.mark,
            person_review.goldstar,
            person_review.level_change,
        )

        default_goodies = fetched.goodies.get(key, {})
        if not default_goodies:
            return False

        target_field = field[:-len('_is_differ_from_default')]
        target_value = getattr(person_review, target_field)
        if target_value is const.SPECIAL_VALUE:
            return const.EMPTY

        default_value = default_goodies[target_field]
        return default_value != target_value
    return getter


get_salary_change_is_differ_from_default = _is_differ_from_default_getter(
    FIELDS.SALARY_CHANGE_IS_DIFFER_FROM_DEFAULT)
get_bonus_is_differ_from_default = _is_differ_from_default_getter(
    FIELDS.BONUS_IS_DIFFER_FROM_DEFAULT)
get_options_rsu_is_differ_from_default = _is_differ_from_default_getter(
    FIELDS.OPTIONS_RSU_IS_DIFFER_FROM_DEFAULT)


def get_reviewers(person_review, fetched, **kwargs):
    if not person_review.permissions_read[PERMS.REVIEWERS_READ]:
        return []
    return fetched.reviewers.get(person_review.id, [])


def get_action_at(person_review, fetched, **kwargs):
    if not person_review.permissions_read[PERMS.ACTION_AT_READ]:
        return []
    approve_level = fetched.person_reviews[person_review.id]['approve_level']
    if approve_level == const.NO_ACCESS:
        return const.NO_ACCESS

    status = fetched.person_reviews[person_review.id]['status']
    reviewers = person_review.reviewers
    return action_at_helper(
        reviewers=reviewers,
        status=status,
        approve_level=approve_level,
    )


def action_at_helper(reviewers, status, approve_level):
    """
    Для переиспользования в нотификации.
    """
    if status in (
        const.PERSON_REVIEW_STATUS.WAIT_EVALUATION,
        const.PERSON_REVIEW_STATUS.EVALUATION,
        const.PERSON_REVIEW_STATUS.APPROVAL,
    ):
        action_at = reviewers and reviewers[approve_level]
    elif status == const.PERSON_REVIEW_STATUS.APPROVED:
        action_at = reviewers and reviewers[-1]
    else:
        action_at = []

    if not isinstance(action_at, list):
        return [action_at]
    else:
        return action_at


# actions
def _edit_action_modifiable_getter(field):
    def getter(person_review, fetched, subject, **kwargs):
        action = const.PERSON_REVIEW_ACTIONS.TO_ACTION_FIELDS_REVERSED[field]
        affected_field = const.PERSON_REVIEW_ACTIONS.AFFECTED_FIELDS.get(action)
        modifier_for_field = const.REVIEW_MODE.FIELDS_TO_MODIFIERS[affected_field]
        modifier = person_review.modifiers[modifier_for_field]

        if modifier and modifier == const.REVIEW_MODE.MODE_DISABLED:
            return const.DISABLED

        # 5 строчек после коммента нужны, чтобы
        # не упала миграция core.0048
        # можно удалять их после того, как выполнится core.0049
        if action == const.PERSON_REVIEW_ACTIONS.LEVEL_CHANGE:
            if person_review.permissions_write[PERMS.LEVEL_CHANGE_WRITE]:
                return const.OK
            else:
                return const.NO_ACCESS

        if modifier and modifier == const.REVIEW_MODE.MODE_AUTO:
            action_auto_permission = {
                const.PERSON_REVIEW_ACTIONS.SALARY_CHANGE: PERMS.SALARY_CHANGE_AUTO_WRITE,
                const.PERSON_REVIEW_ACTIONS.SALARY_CHANGE_ABSOLUTE: PERMS.SALARY_CHANGE_AUTO_WRITE,
                const.PERSON_REVIEW_ACTIONS.BONUS: PERMS.BONUS_AUTO_WRITE,
                const.PERSON_REVIEW_ACTIONS.BONUS_ABSOLUTE: PERMS.BONUS_AUTO_WRITE,
                const.PERSON_REVIEW_ACTIONS.DEFERRED_PAYMENT: PERMS.DEFERRED_PAYMENT_AUTO_WRITE,
                const.PERSON_REVIEW_ACTIONS.BONUS_RSU: PERMS.BONUS_AUTO_WRITE,
                const.PERSON_REVIEW_ACTIONS.OPTIONS_RSU: PERMS.OPTIONS_RSU_AUTO_WRITE,
            }[action]
            if person_review.permissions_write[action_auto_permission]:
                return const.OK
            else:
                return const.NO_ACCESS
        if not person_review.permissions_write[PERMS.FIELD_TO_PERMISSION[field]]:
            return const.NO_ACCESS
        return const.OK
    return getter


def get_action_goldstar(person_review, fetched, subject, **kwargs):
    goldstar_mode = person_review.modifiers[const.REVIEW_MODE.GOLDSTAR_MODE]
    if goldstar_mode == const.REVIEW_MODE.MODE_DISABLED:
        return const.NO_ACCESS
    elif goldstar_mode == const.REVIEW_MODE.MODE_MANUAL:
        if person_review.permissions_write[PERMS.GOLDSTAR_WRITE]:
            return const.OK
        else:
            return const.NO_ACCESS
    elif goldstar_mode == const.REVIEW_MODE.MODE_MANUAL_BY_CHOSEN:
        if person_review.permissions_write[PERMS.GOLDSTAR_WRITE_CHOSEN]:
            return const.OK
        else:
            return const.NO_ACCESS


def _edit_action_simple_getter(field):
    def getter(person_review, fetched, subject, **kwargs):
        if not person_review.permissions_write[PERMS.FIELD_TO_PERMISSION[field]]:
            return const.NO_ACCESS
        return const.OK
    return getter


def _workflow_action_getter(field):
    def getter(person_review, fetched, subject, **kwargs):
        permission = PERMS.FIELD_TO_PERMISSION[field]
        has_permission = person_review.permissions_write[permission]
        if not has_permission or person_review.flagged or person_review.flagged_positive:
            return const.NO_ACCESS
        return const.OK
    return getter


get_action_umbrella = _edit_action_simple_getter(const.FIELDS.ACTION_UMBRELLA)
get_action_main_product = _edit_action_simple_getter(const.FIELDS.ACTION_MAIN_PRODUCT)
get_action_mark = _edit_action_modifiable_getter(const.FIELDS.ACTION_MARK)
get_action_level_change = _edit_action_modifiable_getter(const.FIELDS.ACTION_LEVEL_CHANGE)
get_action_salary_change = _edit_action_modifiable_getter(const.FIELDS.ACTION_SALARY_CHANGE)
get_action_salary_change_absolute = _edit_action_modifiable_getter(const.FIELDS.ACTION_SALARY_CHANGE_ABSOLUTE)
get_action_bonus = _edit_action_modifiable_getter(const.FIELDS.ACTION_BONUS)
get_action_bonus_absolute = _edit_action_modifiable_getter(const.FIELDS.ACTION_BONUS_ABSOLUTE)
get_action_bonus_rsu = _edit_action_modifiable_getter(const.FIELDS.ACTION_BONUS_RSU)
get_action_options_rsu = _edit_action_modifiable_getter(const.FIELDS.ACTION_OPTIONS_RSU)
get_action_deferred_payment = _edit_action_modifiable_getter(const.FIELDS.ACTION_DEFERRED_PAYMENT)

get_action_flagged = _edit_action_simple_getter(const.FIELDS.ACTION_FLAGGED)
get_action_flagged_positive = _edit_action_simple_getter(const.FIELDS.ACTION_FLAGGED_POSITIVE)
get_action_tag_average_mark = _edit_action_simple_getter(const.FIELDS.ACTION_TAG_AVERAGE_MARK)
get_action_taken_in_average = _edit_action_simple_getter(const.FIELDS.ACTION_TAKEN_IN_AVERAGE)
get_action_comment = _edit_action_simple_getter(const.FIELDS.ACTION_COMMENT)
get_action_reviewers = _edit_action_simple_getter(const.FIELDS.ACTION_REVIEWERS)

get_action_unapprove = _workflow_action_getter(const.FIELDS.ACTION_UNAPPROVE)
get_action_approve = _workflow_action_getter(const.FIELDS.ACTION_APPROVE)
get_action_allow_announce = _workflow_action_getter(const.FIELDS.ACTION_ALLOW_ANNOUNCE)
get_action_announce = _workflow_action_getter(const.FIELDS.ACTION_ANNOUNCE)


def get_person_department_chain(person_review, fetched, **kwargs):
    return fetched.department_chains.get(person_review.person_department_path, [])


def _convert_person_department_chain_to_flat_list(field_to_flat, *args, **kwargs):
    chain = get_person_department_chain(*args, **kwargs)
    return [obj[field_to_flat] for obj in chain]


get_person_department_chain_slugs = functools.partial(
    _convert_person_department_chain_to_flat_list, 'slug')
get_person_department_chain_names = functools.partial(
    _convert_person_department_chain_to_flat_list, 'name')


def get_person_chief(person_review, fetched, **kwargs):
    return fetched.person_chief.get(person_review.person_id, {})


def get_comments(person_review, fetched, **kwargs):
    comments = fetched.comments.get(person_review.id, [])
    comments = sorted(
        comments,
        key=operator.attrgetter('created_at'),
        reverse=True,
    )
    return comments


def get_changes(person_review, fetched, **kwargs):
    # this check should be in changes while they are assembled
    if not person_review.permissions_read[PERMS.CHANGES_READ]:
        return []
    changes = fetched.changes.get(person_review.id, [])
    changes = sorted(
        changes,
        key=operator.attrgetter('created_at'),
        reverse=True,
    )
    return changes


def get_goals_url(person_review, fetched, subject, **kwargs):
    return cia.get_goals_url(subject, person_review)


def get_st_goals_url(person_review, fetched, subject, **kwargs):
    return cia.get_st_goals_url(subject, person_review)


def get_review_goals_to_date(person_review, fetched, **kwargs):
    start_date = person_review.review_start_date
    evaluation_to_date = person_review.review_evaluation_to_date
    if evaluation_to_date is not None:
        return evaluation_to_date
    else:
        return start_date


def get_review_goals_from_date(person_review, fetched, **kwargs):
    to_date = person_review.review_goals_to_date
    evaluation_from_date = person_review.review_evaluation_from_date
    if evaluation_from_date is not None:
        return evaluation_from_date
    else:
        return datetimes.shifted(to_date, months=-6)
