# coding: utf-8
from functools import partial
from django.db import transaction
from logging import getLogger
import waffle

from review.lib import helpers as lib_helpers
from review.staff import models as staff_models
from review.xiva.update_calibration import post_to_xiva

from review.core.logic.bulk import diffs, save, helpers
from review.core import const
from review.core.logic import (
    assemble,
    domain_objs,
)

log = getLogger(__name__)


@lib_helpers.timeit_no_args_logging
def bulk_same_action_set(
    subject,
    ids,
    params,
    subject_type=const.PERSON_REVIEW_CHANGE_TYPE.PERSON,
):
    log.info('bulk_same_action_set started. IDs count: %d, actions: %s', len(ids), ','.join(params))
    roles = (
        const.ROLE.PERSON_REVIEW_LIST_RELATED |
        const.ROLE.CALIBRATION.ALL -
        # микрооптимизация — себя никогда нельзя редактировать
        {const.ROLE.PERSON.SELF}
    )
    person_reviews = assemble.get_person_reviews(
        subject=subject,
        filters_chosen={
            const.FILTERS.IDS: ids,
            const.FILTERS.REVIEW_STATUSES: const.REVIEW_STATUS.ALL,
        },
        fields_requested=const.FIELDS.FOR_DOING_ACTION,
        role_types=roles,
    )
    person_review_ids = set(pr.id for pr in person_reviews)
    no_access_ids = set(ids) - person_review_ids
    res = {}
    for id in no_access_ids:
        person_review = domain_objs.PersonReview(id=id)
        no_access_all_actions = dict.fromkeys(params, const.NO_ACCESS)
        res[person_review] = {
            'failed': no_access_all_actions,
            'changes': {},
        }
    res.update(_bulk_same_action_set(
        subject,
        person_reviews,
        params,
        subject_type,
    ))

    if waffle.switch_is_active("calibration_auto_update"):
        post_to_xiva(subject.login, list(person_review_ids))

    return res


def _bulk_same_action_set(subject, person_reviews, params, subject_type):
    res = {}
    if not person_reviews:
        return res
    for person_review in person_reviews:
        res[person_review] = diffs.make_diff_for_actions(
            subject=subject,
            person_review=person_review,
            params=params,
        )
    diffs.update_auto_goodies_for_all(result={
        person_review: diff
        for person_review, diff in res.items()
        if diff['changes']
    })

    save.save_for_same_action_set(
        subject=subject,
        diff_bulk={
            person_review: actions_result['changes']
            for person_review, actions_result in res.items()
        },
        subject_type=subject_type,
    )
    return res


def bulk_different_action_set(
    subject,
    data,
    subject_type=const.PERSON_REVIEW_CHANGE_TYPE.PERSON,
):
    update_ids = data.keys()
    fields = {
        const.FIELDS.REVIEW_ID,
        const.FIELDS.REVIEWERS,
        const.FIELDS.PERSON_LOGIN,
        const.FIELDS.STATUS,
        const.FIELDS.APPROVE_LEVEL,
        const.FIELDS.REVIEW_MARKS_SCALE,
        const.FIELDS.SALARY_VALUE,
        const.FIELDS.TAG_AVERAGE_MARK,
    } | set(const.FIELDS.PERSON_REVIEW_IMPORT_FIELDS) | const.FIELDS.EDIT_ACTION_FIELDS
    person_reviews = assemble.get_person_reviews(
        subject=subject,
        fields_requested=fields,
        filters_chosen={const.FILTERS.IDS: update_ids}
    )
    person_reviews_to_update_actions = {
        person_review: {
            key: value
            for key, value in data[person_review.id].items()
            if key != const.FIELDS.REVIEWERS
        }
        for person_review in person_reviews
    }
    person_reviews_to_update_reviewers = {
        person_review: {
            key: value
            for key, value in data[person_review.id].items()
            if key == const.FIELDS.REVIEWERS
        }
        for person_review in person_reviews
        if const.FIELDS.REVIEWERS in data[person_review.id]
    }
    diff_for_actions = diffs.make_diff_for_actions_different(
        subject=subject,
        data=person_reviews_to_update_actions,
    )
    diff_for_reviewers = diffs.make_diff_for_reviewers_different(
        subject=subject,
        data=person_reviews_to_update_reviewers,
    )
    save.save_for_different_action_edit(
        diff={
            person_review: actions_result['changes']
            for person_review, actions_result in diff_for_actions.items()
            if not actions_result['failed'] and actions_result['changes']
        },
    )

    if diff_for_reviewers:
        reviewers_changes = {
            person_review: actions_result['changes'][const.FIELDS.REVIEWERS]
            for person_review, actions_result in diff_for_reviewers.items()
            if not actions_result['failed'] and actions_result['changes']
        }
        person_fetch_ids = [
            r[0] for reviewers in list(reviewers_changes.values())
            for r in helpers.flat_reviewers(reviewers['new'])
        ]
        person_to_login = dict(staff_models.Person.objects.filter(
            id__in=set(person_fetch_ids)
        ).values_list(
            'id', 'login'
        ))

        for _, actions_result in reviewers_changes.items():
            new_diff = []
            for reviewer in actions_result['new']:
                if not isinstance(reviewer, list):
                    new_diff.append(person_to_login[reviewer])
                else:
                    new_list = [person_to_login[r] for r in reviewer]
                    new_diff.append(new_list)
            actions_result['new'] = new_diff

        save.save_for_different_action_reviewers(
            diff_bulk=reviewers_changes,
            person_to_login=person_to_login
        )

    result = dict(diff_for_actions)

    for person_review, diff in diff_for_reviewers.items():
        change = result.setdefault(
            person_review,
            {
                'failed': {},
                'changes': {},
            },
        )
        change['changes'].update(diff['changes'])
        change['failed'].update(diff['failed'])

    changes = save.save_for_different_action_change_models(
        subject=subject,
        diff_bulk={
            person_review: actions_result['changes']
            for person_review, actions_result in result.items()
        },
        subject_type=subject_type,
    )
    for person_review, changes_object in changes.items():
        result[person_review]['changes'] = changes_object

    no_access_ids = set(update_ids) - set(pr.id for pr in person_reviews)
    for id in no_access_ids:
        person_review_extended = domain_objs.PersonReview(id=id)
        result[person_review_extended] = const.NO_ACCESS
    return result


STATUS = const.PERSON_REVIEW_STATUS


@transaction.atomic
def publish_person_reviews(subject, review_id):
    bulk_function = partial(
        _bulk_same_action_set,
        subject=subject,
        subject_type=const.PERSON_REVIEW_CHANGE_TYPE.ROBOT,
    )
    person_reviews = assemble.get_person_reviews(
        subject=subject,
        fields_requested=const.FIELDS.FOR_ROBOT,
        filters_chosen={const.FILTERS.REVIEWS: [review_id]}
    )
    review = assemble.get_review(
        subject=subject,
        filters={'ids': [review_id]},
        requested_fields={
            'id',
            'mark_mode',
        },
    )
    marks_enabled = review.mark_mode == const.REVIEW_MODE.MODE_MANUAL
    if marks_enabled:
        _set_mark(person_reviews, bulk_function)
    _approve(person_reviews, bulk_function, marks_enabled)
    _allow_announce(person_reviews, bulk_function)
    _announce(person_reviews, bulk_function)
    return len(person_reviews)


def _set_mark(person_reviews, bulk_function):
    wait_eval = [
        pr for pr in person_reviews
        if pr.status == STATUS.WAIT_EVALUATION
    ]
    bulk_function(
        person_reviews=wait_eval,
        params=dict(mark=const.MARK.NO_MARK),
    )
    for pr in wait_eval:
        pr.status = STATUS.EVALUATION
        pr.action_approve = const.OK


def _approve(person_reviews, bulk_function, marks_enabled):
    to_approve = {
        pr: len(pr.reviewers) - pr.approve_level - 1
        for pr in person_reviews
        if (
            pr.status in (STATUS.EVALUATION, STATUS.APPROVAL) or
            (pr.status == STATUS.WAIT_EVALUATION and not marks_enabled)
        )
    }

    def get_need_approve():
        return [pr for pr, left in to_approve.items() if left > 0]

    need_approve = get_need_approve()
    while need_approve:
        bulk_function(
            person_reviews=need_approve,
            params=dict(approve=True),
        )
        for pr in need_approve:
            to_approve[pr] -= 1
            if not to_approve[pr]:
                pr.status = STATUS.APPROVED
                pr.action_allow_announce = const.OK
        need_approve = get_need_approve()


def _allow_announce(person_reviews, bulk_function):
    approved = [
        pr for pr in person_reviews
        if pr.status == STATUS.APPROVED
    ]
    bulk_function(
        person_reviews=approved,
        params=dict(allow_announce=True),
    )
    for pr in approved:
        pr.status = STATUS.WAIT_ANNOUNCE
        pr.action_announce = const.OK


def _announce(person_reviews, bulk_function):
    wait_announce = [
        pr for pr in person_reviews
        if pr.status == STATUS.WAIT_ANNOUNCE
    ]
    bulk_function(
        person_reviews=wait_announce,
        params=dict(announce=True),
    )
    for pr in wait_announce:
        pr.status = STATUS.ANNOUNCED


def bulk_same_action_set_for_calibration_person_review(subject, ids, actions):
    calibration_person_reviews = assemble.get_calibration_person_reviews(
        subject=subject,
        filters_chosen={'ids': ids},
        requested_person_review_fields={
            const.FIELDS.ID,
            const.FIELDS.PERSON_LOGIN,
        }
    )
    actions = {
        field: new for field, new in actions.items()
        if field in const.CALIBRATION_PERSON_REVIEW_ACTIONS.ALL
    }
    result = {
        cpr.id: diffs.make_diff_for_calibration_person_review_actions(
            subject=subject,
            calibration_person_review=cpr,
            params=actions,
        )
        for cpr in calibration_person_reviews
    }

    save.update_calibration_person_review_models(diff={
        id: actions_result['changes']
        for id, actions_result in result.items()
        if actions_result['changes']
    })
    no_access_ids = set(ids) - set(pc.id for pc in calibration_person_reviews)
    result.update({
        id: dict(failed=dict.fromkeys(actions, const.NO_ACCESS))
        for id in no_access_ids
    })

    if waffle.switch_is_active("calibration_auto_update"):
        person_review_ids = set(cpr.person_review.id for cpr in calibration_person_reviews)
        post_to_xiva(subject.login, person_review_ids)

    return result
