# coding: utf-8

import decimal
import json
import logging
from itertools import chain
from typing import Optional

from review.gradient import models as gradient_models
from review.core import const
from review.core import models
from review.lib import helpers as lib_helpers

from . import helpers

log = logging.getLogger(__name__)


def make_diff_for_actions_different(subject, data):
    result_diff = {}
    for person_review, update_params in data.items():
        result_diff[person_review] = make_diff_for_actions(
            subject=subject,
            person_review=person_review,
            params=update_params,
        )
    return result_diff


def make_diff_for_reviewers_different(subject, data):
    result = {}
    for person_review, update_data in data.items():
        new = update_data.get(const.FIELDS.REVIEWERS)
        if new is None:
            continue
        action_access = getattr(person_review, const.FIELDS.ACTION_REVIEWERS)
        res = result.setdefault(person_review, {'failed': {}, 'changes': {}})
        if action_access != const.OK:
            res['failed'][const.FIELDS.REVIEWERS] = action_access
            continue
        new_chain = helpers.normalize_reviewers_chain(new, person_review)
        if not new_chain:
            res['failed'][const.FIELDS.REVIEWERS] = const.ERROR_CODES.DELETING_LAST_REVIEWER
            continue
        res['changes'][const.FIELDS.REVIEWERS] = {
            'old': helpers.extract_login_from_reviewers(person_review.reviewers),
            'new': new_chain
        }

    return result


def _weight_of_action(args):
    act = args[0]
    if act in const.PERSON_REVIEW_ACTIONS.EXECUTE_ORDER:
        return const.PERSON_REVIEW_ACTIONS.EXECUTE_ORDER.index(act)
    else:
        return float('inf')


def make_diff_for_actions(subject, person_review, params):
    result = {
        'failed': {},
        'changes': {},
    }
    sorted_actions = sorted(list(params.items()), key=_weight_of_action)
    for action, new in sorted_actions:
        action_access = getattr(person_review, 'action_' + action)

        # TODO: remove duplicating check
        field = const.PERSON_REVIEW_ACTIONS.AFFECTED_FIELDS.get(action)
        editing_flags = field in const.FIELDS.FLAG_FIELDS
        if field and not editing_flags and getattr(person_review, field) == new:
            continue

        if action_access != const.OK:
            result['failed'][action] = action_access
            continue
        try:
            changes = make_diff_for_action(subject, person_review, action, new, result['changes'])
        except Exception:
            msg = f'Fail to make diff for action {action} person_review {person_review.id}.'
            if action == const.PERSON_REVIEW_ACTIONS.REVIEWERS:
                msg += f' New reviewers are {new}.'
            log.exception(msg, exc_info=True)
            raise
        result['changes'].update(changes)
    _evaluate_if_needed(subject, person_review, params, result['changes'])
    validate_result(person_review, result)
    return result


def make_diff_for_fk(subject, person_review, changes):
    result = {}
    for field, new in changes.items():
        pr_field = f'{field}_id'
        old_id = getattr(person_review, pr_field)
        new_id = None if new == -1 else new.id
        if new_id == old_id:
            return result
        result[field] = {
            'old': old_id,
            'new': new_id,
        }
        setattr(person_review, pr_field, new_id)

    return result


def make_diff_for_action(subject, person_review, action, new, already_changed):
    if action == const.PERSON_REVIEW_ACTIONS.MARK:
        return _make_diff_if_changed(subject, person_review, {const.FIELDS.MARK: new})
    if action == const.PERSON_REVIEW_ACTIONS.UMBRELLA:
        return make_diff_for_fk(subject, person_review, {const.FIELDS.UMBRELLA: new})
    if action == const.PERSON_REVIEW_ACTIONS.MAIN_PRODUCT:
        return make_diff_for_fk(subject, person_review, {const.FIELDS.MAIN_PRODUCT: new})
    if action in const.PERSON_REVIEW_ACTIONS.GROUPS[const.PERSON_REVIEW_ACTIONS.EDIT]:
        field = const.PERSON_REVIEW_ACTIONS.AFFECTED_FIELDS[action]
        return _make_diff_if_changed(subject, person_review, {field: new})
    if action in const.PERSON_REVIEW_ACTIONS.GROUPS[const.PERSON_REVIEW_ACTIONS.BONUS_EDIT]:
        field = const.PERSON_REVIEW_ACTIONS.AFFECTED_FIELDS[action]
        bonus_fields = const.PERSON_REVIEW_ACTIONS.GROUPS[const.PERSON_REVIEW_ACTIONS.BONUS_EDIT]
        if any(f in already_changed for f in bonus_fields):
            return {}
        changes = _make_diff_with_post_effect(
            subject,
            person_review,
            field,
            new,
            _make_change_for_another_bonus,
        )
        return changes
    if action in const.PERSON_REVIEW_ACTIONS.GROUPS[const.PERSON_REVIEW_ACTIONS.SALARY_EDIT]:
        field = const.PERSON_REVIEW_ACTIONS.AFFECTED_FIELDS[action]
        salary_fields = const.PERSON_REVIEW_ACTIONS.GROUPS[const.PERSON_REVIEW_ACTIONS.SALARY_EDIT]
        if any(f in already_changed for f in salary_fields):
            return {}
        return _make_diff_with_post_effect(subject, person_review, field, new, _make_change_for_another_salary)
    if action in const.PERSON_REVIEW_ACTIONS.GROUPS[const.PERSON_REVIEW_ACTIONS.FLAGS]:
        field = const.PERSON_REVIEW_ACTIONS.AFFECTED_FIELDS[action]
        return make_diff_for_flag(subject, person_review, field, new)
    if action == const.PERSON_REVIEW_ACTIONS.APPROVE:
        return make_diff_for_approve(subject, person_review)
    if action == const.PERSON_REVIEW_ACTIONS.UNAPPROVE:
        return make_diff_for_unapprove(subject, person_review)
    if action == const.PERSON_REVIEW_ACTIONS.ANNOUNCE:
        return make_diff_for_announce(subject, person_review)
    if action == const.PERSON_REVIEW_ACTIONS.ALLOW_ANNOUNCE:
        return make_diff_for_allow_announce(subject, person_review)
    if action == const.PERSON_REVIEW_ACTIONS.COMMENT:
        return make_diff_for_comment(subject, person_review, action, new)
    if action == const.PERSON_REVIEW_ACTIONS.REVIEWERS:
        return make_diff_for_reviewers(subject, person_review, action, new)


def _evaluate_if_needed(subject, person_review, actions, changes):
    if person_review.status != const.PERSON_REVIEW_STATUS.WAIT_EVALUATION:
        return
    mark_set = const.PERSON_REVIEW_ACTIONS.MARK in actions
    mark_mode = person_review.modifiers[const.REVIEW_MODE.MARK_MODE]
    marks_disabled = mark_mode == const.REVIEW_MODE.MODE_DISABLED
    bonus_actions = const.PERSON_REVIEW_ACTIONS.GROUPS[const.PERSON_REVIEW_ACTIONS.BONUS_EDIT]
    bonus_set = set(actions) & bonus_actions
    if mark_set or (marks_disabled and bonus_set):
        changes.update(
            _make_diff_if_changed(
                subject,
                person_review,
                {const.FIELDS.STATUS: const.PERSON_REVIEW_STATUS.EVALUATION},
            )
        )


def _make_diff_if_changed(subject, person_review, changes):
    result = {}
    for field, new in changes.items():
        old = getattr(person_review, field)
        if old == new:
            continue
        setattr(person_review, field, new)
        result[field] = {
            'old': old,
            'new': new,
        }
    return result


def _make_diff_with_post_effect(subject, person_review, field, new, post_effect):
    changed = _make_diff_if_changed(subject, person_review, {field: new})
    if not changed:
        return changed
    changed.update(post_effect(subject, person_review, field, new))
    return changed


def _get_closest_salary(person_review) -> Optional[decimal.Decimal]:
    if isinstance(person_review.salary_closest_to_review_start, dict):
        str_salary = person_review.salary_closest_to_review_start['value']
        if str_salary is not None:
            return decimal.Decimal('{:.2f}'.format(str_salary))
    log.warning(
        'salary_closest_to_review_start is %s for person_review %s',
        type(person_review.salary_closest_to_review_start),
        person_review.id,
    )


def _make_change_for_another_bonus(subject, person_review, first_bonus_field, first_bonus_val):
    salary = _get_closest_salary(person_review)

    if first_bonus_field == const.FIELDS.BONUS:
        update_by_bonus_type = {
            const.FIELDS.BONUS_ABSOLUTE: percent_of_salary_to_absolute(salary, first_bonus_val)
            if salary else decimal.Decimal(0.00),
            const.FIELDS.BONUS_TYPE: const.SALARY_DEPENDENCY_TYPE.PERCENTAGE,
        }
    else:
        try:
            update_by_bonus_type = {
                const.FIELDS.BONUS: absolute_of_salary_to_percent(salary, first_bonus_val)
                if salary else decimal.Decimal(0.00),
                const.FIELDS.BONUS_TYPE: const.SALARY_DEPENDENCY_TYPE.ABSOLUTE,
            }
        except ZeroDivisionError:
            from review.lib.encryption import encrypt
            log.warning('zerodivisionconvert: ' + encrypt(json.dumps(dict(
                salary=salary,
                first_bonus_val=first_bonus_val,
                pr_id=person_review.id,
            ))))
    return _make_diff_if_changed(subject, person_review, update_by_bonus_type)


def _make_change_for_another_salary(subject, person_review, first_salary_field, first_salary_val):
    salary = _get_closest_salary(person_review)

    if first_salary_field == const.FIELDS.SALARY_CHANGE:
        update_by_salary_type = {
            const.FIELDS.SALARY_CHANGE_ABSOLUTE: percent_of_salary_to_absolute(salary, first_salary_val)
            if salary else decimal.Decimal(0.00),
            const.FIELDS.SALARY_CHANGE_TYPE: const.SALARY_DEPENDENCY_TYPE.PERCENTAGE,
        }
    else:
        try:
            update_by_salary_type = {
                const.FIELDS.SALARY_CHANGE: absolute_of_salary_to_percent(salary, first_salary_val)
                if salary else decimal.Decimal(0.00),
                const.FIELDS.SALARY_CHANGE_TYPE: const.SALARY_DEPENDENCY_TYPE.ABSOLUTE,
            }
        except ZeroDivisionError:
            from review.lib.encryption import encrypt
            log.warning('zerodivisionconvert: ' + encrypt(json.dumps(dict(
                salary=salary,
                first_bonus_val=first_salary_val,
                pr_id=person_review.id,
            ))))
    return _make_diff_if_changed(subject, person_review, update_by_salary_type)


def percent_of_salary_to_absolute(salary, percent):
    return int(salary * percent / 100)


def absolute_of_salary_to_percent(salary, absolute):
    return salary and absolute / (salary / 100)


def make_diff_for_flag(subject, person_review, field, new):
    if new:
        return _make_diff_if_changed(subject, person_review, {field: new})
    else:
        return _make_diff_if_changed(
            subject,
            person_review,

            # if we set flag to false - all flags have to be set to false
            {flag_field: new for flag_field in const.FIELDS.FLAG_FIELDS},
        )


def make_diff_for_approve(subject, person_review):
    changes = {}
    max_approve_level = helpers.get_max_approve_level(
        subject=subject,
        reviewers=person_review.reviewers,
    )
    can_approve_incrementally = any(
        r in person_review.roles for r in (
            const.ROLE.REVIEW.SUPERREVIEWER,
            const.ROLE.GLOBAL.ROBOT,
            const.ROLE.DEPARTMENT.HR_PARTNER,
            const.ROLE.DEPARTMENT.HR_ANALYST,
        )
    )
    if can_approve_incrementally:
        effective_approve_level = person_review.approve_level
    else:
        effective_approve_level = max_approve_level

    approve_levels_count = len(person_review.reviewers) - 1
    if effective_approve_level >= approve_levels_count:
        changes[const.FIELDS.APPROVE_LEVEL] = approve_levels_count
        changes[const.FIELDS.STATUS] = const.PERSON_REVIEW_STATUS.APPROVED
    else:
        changes[const.FIELDS.APPROVE_LEVEL] = effective_approve_level + 1
        changes[const.FIELDS.STATUS] = const.PERSON_REVIEW_STATUS.APPROVAL

    return _make_diff_if_changed(subject, person_review, changes)


def make_diff_for_unapprove(subject, person_review):
    changes = {}
    current_status = person_review.status
    current_approve_level = person_review.approve_level
    is_marks_enabled = person_review.modifiers[const.REVIEW_MODE.MARK_MODE] == const.REVIEW_MODE.MODE_MANUAL

    if current_status == const.PERSON_REVIEW_STATUS.APPROVAL:
        if current_approve_level == 1:
            if is_marks_enabled:
                previous_status = const.PERSON_REVIEW_STATUS.EVALUATION
            else:
                previous_status = const.PERSON_REVIEW_STATUS.WAIT_EVALUATION
            changes[const.FIELDS.STATUS] = previous_status
        changes[const.FIELDS.APPROVE_LEVEL] = current_approve_level - 1
    elif current_status == const.PERSON_REVIEW_STATUS.APPROVED:
        changes[const.FIELDS.STATUS] = const.PERSON_REVIEW_STATUS.APPROVAL

    return _make_diff_if_changed(subject, person_review, changes)


def make_diff_for_allow_announce(subject, person_review):
    return _make_diff_if_changed(
        subject=subject,
        person_review=person_review,
        changes={
            const.FIELDS.STATUS: const.PERSON_REVIEW_STATUS.WAIT_ANNOUNCE,
        },
    )


def make_diff_for_announce(subject, person_review):
    return _make_diff_if_changed(
        subject=subject,
        person_review=person_review,
        changes={
            const.FIELDS.STATUS: const.PERSON_REVIEW_STATUS.ANNOUNCED,
        },
    )


def make_diff_for_workflow(subject, person_review, action):
    # https://wiki.yandex-team.ru/staff/pool/CIA/Revju-2.0/flow/#ozhidaetocenki
    if action == const.PERSON_REVIEW_ACTIONS.APPROVE:
        approve_levels_count = len(person_review.reviewers)
        if person_review.approve_level == approve_levels_count - 1:
            field = const.FIELDS.STATUS
            new = const.PERSON_REVIEW_STATUS.APPROVED
        else:
            field = const.FIELDS.APPROVE_LEVEL
            new = person_review.approve_level + 1
    elif action == const.PERSON_REVIEW_ACTIONS.UNAPPROVE:
        field = const.FIELDS.APPROVE_LEVEL
        new = person_review.approve_level - 1
    else:
        field = const.FIELDS.STATUS
        new = {
            const.PERSON_REVIEW_ACTIONS.ALLOW_ANNOUNCE: const.PERSON_REVIEW_STATUS.WAIT_ANNOUNCE,
            const.PERSON_REVIEW_ACTIONS.ANNOUNCE: const.PERSON_REVIEW_STATUS.ANNOUNCED,
        }[action]

    return _make_diff_if_changed(person_review, field, new)


def make_diff_for_comment(subject, person_review, action, new):
    return {'comment': {'new': new}}


def make_diff_for_reviewers(subject, person_review, action, new):
    old_reviewers = helpers.extract_login_from_reviewers(person_review.reviewers)
    if old_reviewers:
        next_approver = helpers.get_next_approver_login(
            old_reviewers,
            person_review.approve_level,
        )
    else:
        next_approver = None

    if new['type'] == const.PERSON_REVIEW_ACTIONS.REVIEW_CHAIN_DELETE:
        new_chain = edit_chain_delete(
            reviewers=old_reviewers,
            persons=new['persons']
        )
    elif new['type'] == const.PERSON_REVIEW_ACTIONS.REVIEW_CHAIN_REPLACE:
        new_chain = edit_chain_replace(
            reviewers=old_reviewers,
            person=new['person'],
            person_to=new['person_to'],
        )
    elif new['type'] == const.PERSON_REVIEW_ACTIONS.REVIEW_CHAIN_REPLACE_ALL:
        new_chain = [new['person_to'].login]

    elif new['position'] in (
        const.PERSON_REVIEW_ACTIONS.ADD_POSITION_END,
        const.PERSON_REVIEW_ACTIONS.ADD_POSITION_START
    ):
        new_chain = edit_chain_add(
            reviewers=old_reviewers,
            position=new['position'],
            person=new['person'],
        )
    else:
        new_chain = edit_chain_insert(
            reviewers=old_reviewers,
            position=new['position'],
            person=new['person'],
            position_person=new['position_person'],
        )
    new_reviewers = helpers.normalize_reviewers_chain(new_chain, person_review)
    if old_reviewers == new_reviewers:
        return {}

    diff = dict(reviewers=dict(
        old=old_reviewers,
        new=new_reviewers,
    ))

    new_approve_level = (
        (
            next_approver and
            helpers.get_approve_level_of_approver(new_reviewers, next_approver)
        ) or
        person_review.approve_level
    )

    new_max_approve_lvl = len(new_reviewers) - 1

    logins = []

    if 'persons' in new:
        logins.extend([p.login for p in new['persons']])

    if 'person' in new and new['person']:
        logins.extend([new['person'].login])

    if new['type'] == const.PERSON_REVIEW_ACTIONS.REVIEW_CHAIN_REPLACE_ALL:
        new_approve_level = 0
    elif new_reviewers[-1] in logins and person_review.status == const.PERSON_REVIEW_STATUS.APPROVED:
        new_approve_level = new_max_approve_lvl
    elif new_approve_level > 0 and new_reviewers[new_approve_level - 1] in logins:
        new_approve_level -= 1

    diff.update(
        _make_diff_if_changed(
            subject,
            person_review,
            {const.FIELDS.APPROVE_LEVEL: new_approve_level},
        )
    )

    if person_review.approve_level > new_max_approve_lvl:
        approve_change = {
            const.FIELDS.APPROVE_LEVEL: new_max_approve_lvl,
            const.FIELDS.STATUS: const.PERSON_REVIEW_STATUS.APPROVED
        }
        diff.update(_make_diff_if_changed(subject, person_review, approve_change))
    return diff


def validate_result(person_review, result):
    # validate_mark_is_set_if_should(person_review, result) TODO: return after completing CIA-1783
    validate_mark_is_in_scale(person_review, result)
    validate_chain_is_not_empty(person_review, result)


def validate_mark_is_set_if_should(person_review, result):
    is_marks_enabled = (
        person_review.modifiers[const.REVIEW_MODE.MARK_MODE] ==
        const.REVIEW_MODE.MODE_MANUAL
    )
    mark_not_set_but_should = (
        is_marks_enabled and person_review.mark == const.MARK.NOT_SET and
        const.FIELDS.MARK not in result['changes']
    )
    should_not_before_mark = (
        const.FIELDS.SALARY_CHANGE,
        const.FIELDS.BONUS,
        const.FIELDS.LEVEL_CHANGE,
        const.FIELDS.OPTIONS_RSU,
    )

    if not mark_not_set_but_should:
        return

    for changed_field, change in result['changes'].items():
        if changed_field in should_not_before_mark:
            _mark_change_as_error(
                result,
                changed_field,
                const.ERROR_CODES.SHOULD_NOT_BEFORE_MARK,
            )
        if changed_field == const.FIELDS.STATUS:
            new_status = change['new']
            if new_status != const.PERSON_REVIEW_STATUS.EVALUATION:
                _mark_change_as_error(
                    result,
                    changed_field,
                    const.ERROR_CODES.SHOULD_NOT_BEFORE_MARK,
                )


def validate_chain_is_not_empty(person_review, result):
    if const.FIELDS.REVIEWERS not in result['changes']:
        return

    new = result['changes'][const.FIELDS.REVIEWERS]['new']
    if not new:
        _mark_change_as_error(
            result,
            const.FIELDS.REVIEWERS,
            const.ERROR_CODES.DELETING_LAST_REVIEWER,
        )


def validate_mark_is_in_scale(person_review, result):
    if const.FIELDS.MARK not in result['changes']:
        return

    new = result['changes'][const.FIELDS.MARK]['new']
    if new not in chain(person_review.review_marks_scale, const.MARK.SPECIAL_VALUES):
        _mark_change_as_error(
            result,
            const.FIELDS.MARK,
            const.ERROR_CODES.MARK_NOT_FROM_SCALE,
        )


def _mark_change_as_error(result, field, error='ERROR'):
    del result['changes'][field]
    result['failed'][field] = error


def update_auto_goodies_for_all(result):
    review_ids = set(person_review.review_id for person_review in result)
    related_goodies = _get_goodies(review_ids)
    for person_review, diff in result.items():
        update_auto_goodies_for_one(
            person_review=person_review,
            diff=diff,
            data={
                'goodies': related_goodies,
            }
        )


@lib_helpers.timeit_no_args_logging
def _get_goodies(review_ids):
    goodies = models.Goodie.objects.filter(review_id__in=review_ids)
    grouped = {}
    for goodie in goodies:
        key = (
            goodie.review_id,
            goodie.level,
            goodie.mark,
            goodie.goldstar,
            goodie.level_change
        )
        grouped[key] = {
            const.FIELDS.SALARY_CHANGE: goodie.salary_change,
            const.FIELDS.BONUS: goodie.bonus,
            const.FIELDS.OPTIONS_RSU: goodie.options_rsu,
        }
    return grouped


def update_auto_goodies_for_one(person_review, diff, data):
    raw_changes = diff['changes']
    if (
        const.FIELDS.MARK not in raw_changes and
        const.FIELDS.GOLDSTAR not in raw_changes and
        const.FIELDS.LEVEL_CHANGE not in raw_changes
    ):
        return

    mark_diff = raw_changes.get(const.FIELDS.MARK, {})
    goldstar_diff = raw_changes.get(const.FIELDS.GOLDSTAR, {})
    level_change_diff = raw_changes.get(const.FIELDS.LEVEL_CHANGE, {})
    is_mark_changed = bool(mark_diff)
    is_goldstar_changed = bool(goldstar_diff)
    is_level_change_changed = bool(level_change_diff)

    if not any([is_mark_changed, is_goldstar_changed, is_level_change_changed]):
        return

    new_mark = mark_diff.get('new')
    new_goldstar = goldstar_diff.get('new')
    new_level_change = level_change_diff.get('new')

    goodie_key = (
        person_review.review_id,
        person_review.level,
        new_mark or _get_value_or_default(person_review, const.FIELDS.MARK),
        new_goldstar or _get_value_or_default(person_review, const.FIELDS.GOLDSTAR),
        new_level_change or _get_value_or_default(person_review, const.FIELDS.LEVEL_CHANGE),
    )

    auto_goodies = data['goodies'].get(goodie_key)
    for modifier_name, modifier_value in person_review.modifiers.items():
        field = const.REVIEW_MODE.MODIFIERS_FOR_AUTO_GOODIES[modifier_name]
        action = const.PERSON_REVIEW_ACTIONS.TO_ACTION_FIELDS[field]

        if modifier_name not in const.REVIEW_MODE.MODIFIERS_OUTPUT:
            continue
        if getattr(person_review, action) == const.OK:
            # некоторые роли редактируют автополя вручную и не хотят АВТО
            continue
        if modifier_value != const.REVIEW_MODE.MODE_AUTO:
            # если не редактируют вручную, то проверяем режим
            continue

        old_value = getattr(person_review, field)
        new_value = auto_goodies and auto_goodies[field] or 0
        if old_value == new_value:
            continue
        raw_changes[field] = {
            'old': old_value,
            'new': new_value,
        }
        post_effect = None
        if action in const.FIELDS.BONUS_ACTIONS:
            post_effect = _make_change_for_another_bonus
        elif action in const.FIELDS.SALARY_ACTIONS:
            post_effect = _make_change_for_another_salary
        if post_effect:
            raw_changes.update(post_effect(None, person_review, field, new_value))


def _get_value_or_default(person_review, field):
    value = getattr(person_review, field)
    if not isinstance(value, const.SPECIAL_VALUE):
        return value
    return const.REVIEW_MODE.DEFAULTS[field]


def edit_chain_insert(reviewers, position, person, position_person):
    new_chain = None

    if position == const.PERSON_REVIEW_ACTIONS.ADD_POSITION_SAME:
        new_chain = edit_chain_insert_same_level(reviewers, person, position_person)
    elif position == const.PERSON_REVIEW_ACTIONS.ADD_POSITION_AFTER:
        new_chain = edit_chain_insert_after(reviewers, person, position_person)
    elif position == const.PERSON_REVIEW_ACTIONS.ADD_POSITION_BEFORE:
        reverse_reviewers = reversed(reviewers)
        reversed_new_chain = edit_chain_insert_after(reverse_reviewers, person, position_person)
        new_chain = list(reversed(reversed_new_chain))

    return new_chain


def edit_chain_insert_same_level(reviewers, person, position_person):
    new_chain = []
    for reviewer in reviewers:
        if isinstance(reviewer, list):
            new_chain.append(
                reviewer if position_person.login not in reviewer
                else [person.login] + reviewer
            )
        else:
            new_chain.append(
                reviewer if reviewer != position_person.login
                else [reviewer, person.login]
            )

    return new_chain


def edit_chain_insert_after(reviewers, person, position_person):
    def is_insert_position(reviewer):
        return any((
            isinstance(reviewer, list) and position_person.login in reviewer,
            isinstance(reviewer, str) and position_person.login == reviewer,
        ))

    new_chain = []
    for reviewer in reviewers:
        new_chain.append(reviewer)
        if is_insert_position(reviewer):
            new_chain.append(person.login)
    return new_chain


def edit_chain_add(reviewers, position, person):
    if position == const.PERSON_REVIEW_ACTIONS.ADD_POSITION_END:
        return reviewers + [person.login]

    elif position == const.PERSON_REVIEW_ACTIONS.ADD_POSITION_START:
        return [person.login] + reviewers


def edit_chain_replace(reviewers, person, person_to):
    new_chain = []
    for reviewer in reviewers:
        if not isinstance(reviewer, list):
            new_chain.append(
                reviewer if reviewer != person.login
                else person_to.login
            )
        else:
            new_list = [
                r if r != person.login
                else person_to.login
                for r in reviewer
            ]
            new_chain.append(new_list)
    return new_chain


def edit_chain_delete(reviewers, persons):
    new_chain = []
    logins = [person.login for person in persons]
    for reviewer in reviewers:
        if not isinstance(reviewer, list):
            if reviewer not in logins:
                new_chain.append(reviewer)
        else:
            new_list = [r for r in reviewer if r not in logins]
            if new_list:
                new_chain.append(new_list)

    return new_chain


def make_diff_for_calibration_person_review_actions(subject, calibration_person_review, params):
    result = {
        'failed': {},
        'changes': {},
    }
    for action, new in params.items():
        action_access = calibration_person_review.actions.get(action)
        if action_access != const.OK:
            result['failed'][action] = action_access
            continue
        result['changes'].update(
            make_diff_for_calibration_person_review_action(
                subject=subject,
                calibration_person_review=calibration_person_review,
                action=action,
                new=new,
            )
        )
    return result


def make_diff_for_calibration_person_review_action(subject, calibration_person_review, action, new):
    if action == 'discuss':
        field = 'discuss'
        old = calibration_person_review.discuss
        if old == new:
            return {}
        calibration_person_review.discuss = new
        return {field: {'old': old, 'new': new}}
