# coding: utf-8
import json

from django.utils import timezone
from jsonfield import encoder

from review.lib import std as lib_std
from review.core import models


def extract_login_from_reviewers(reviewers):
    """
    >>> extract_login_from_reviewers([
    ...     {'login': 'user1'},
    ...     [
    ...         {'login': 'user3'},
    ...         {'login': 'user2'},
    ...     ],
    ... ])
    ["user1", ["user2", "user3"]]
    >>>
    """
    res = []
    for it in reviewers:
        if isinstance(it, (tuple, list)):
            res.append(sorted([r['login'] for r in it]))
        else:
            res.append(it['login'])
    return res


def normalize_reviewers_chain(reviewers_chain, person_review):
    new_chain = []
    for elem in reviewers_chain:
        # don't let person to be reviewer for himself
        if isinstance(elem, (str, int)):
            # single reviewer
            if elem != person_review.person_login:
                new_chain.append(elem)
        else:
            # several reviewers at same approve level
            reviewers = sorted({r for r in elem if r != person_review.person_login})
            if len(reviewers) == 1:
                new_chain.append(reviewers[0])
            elif reviewers:
                new_chain.append(reviewers)

    lib_std.remove_adjacent_doubles(new_chain)
    return new_chain


def flat_reviewers(reviewers):
    flatten = []
    for index, reviewer in enumerate(reviewers):
        if isinstance(reviewer, (list, tuple)):
            flatten += [(r, index) for r in reviewer]
        else:
            flatten.append((reviewer, index))
    return flatten


def get_max_approve_level(subject, reviewers):
    max_approve_level = None
    for reviewer, index in flat_reviewers(reviewers):
        if subject.login == reviewer['login']:
            max_approve_level = index
    return max_approve_level


def build_change_model(subject, person_review, changes, subject_type):
    changes = {
        key: value for key, value in changes.items() if key != 'comment'
    }
    if not changes:
        return
    # чтобы в пресохраненной модели все было как будет при доставании из базы
    # (обработался Decimal и прочие)
    changes = json.loads(
        json.dumps(
            changes,
            cls=encoder.JSONEncoder,
        )
    )
    return models.PersonReviewChange(
        subject=subject,
        person_review_id=person_review.id,
        diff=changes,
        created_at=timezone.now(),
        subject_type=subject_type,
    )


def build_comment_model(subject, person_review, comment_text):
    if not comment_text:
        return
    now = timezone.now()
    return models.PersonReviewComment(
        subject=subject,
        person_review_id=person_review.id,
        text_wiki=comment_text,
        created_at=now,
        updated_at=now,
    )


def get_next_approver_login(approver_logins, approve_level):
    approver = approver_logins[approve_level]
    if isinstance(approver, list):
        approver = approver[0]
    return approver


def get_approve_level_of_approver(approver_logins, login):
    for level, approver in enumerate(approver_logins):
        if isinstance(approver, list) and login in approver:
            return level
        elif approver == login:
            return level
