# coding: utf-8

from collections import defaultdict

import yenv
from django.db import transaction
from django.db.models import Q
from django.db.models import fields
from django.utils import timezone
from django_bulk_update.helper import bulk_update

from review.core import const, models
from review.core.logic import roles
from review.gradient import models as gradient_models
from review.lib import helpers
from review.staff import models as staff_models

from review.core.logic.bulk import helpers as bulk_helpers
from functools import reduce


@helpers.timeit_no_args_logging
def save_for_same_action_set(subject, diff_bulk, subject_type):
    result = {}
    grouped_by_same_update = {}
    change_models = []
    comment_models = []
    reviewers_update = []
    for person_review, diff in diff_bulk.items():
        change_model = bulk_helpers.build_change_model(
            subject=subject,
            person_review=person_review,
            changes=diff,
            subject_type=subject_type,
        )
        if change_model:
            change_models.append(change_model)

        for field, field_diff in diff.items():
            new = field_diff['new']
            if field == "reviewers":
                reviewers_update.append((person_review, field_diff))
            elif field == 'comment':
                comment_model = bulk_helpers.build_comment_model(
                    subject=subject,
                    person_review=person_review,
                    comment_text=new,
                )
                if comment_model:
                    comment_models.append(comment_model)
            else:
                group = grouped_by_same_update.setdefault((field, new), [])
                group.append(person_review)

    prepared_reviewers_update = prepare_reviewers_update(reviewers_update)
    with transaction.atomic():
        save_person_review_objects(data=grouped_by_same_update)
        save_person_review_roles(
            data=prepared_reviewers_update,
            reviewers_update=reviewers_update,
        )
        if change_models:
            save_person_review_changes(prepared_models=change_models)
        if comment_models:
            save_person_review_comments(prepared_models=comment_models)
    return result


@helpers.timeit_no_args_logging
def save_for_different_action_reviewers(diff_bulk, person_to_login):
    login_to_person = {value: key for key, value in person_to_login.items()}
    prepared_reviewers_update = prepare_reviewers_update(
        reviewers_update=list(diff_bulk.items()),
        login_to_person=login_to_person,
    )
    save_person_review_roles(
        data=prepared_reviewers_update,
        reviewers_update=list(diff_bulk.items())
    )


@helpers.timeit_no_args_logging
def save_for_different_action_edit(diff):
    group_by_updated_fields = defaultdict(list)
    fk_changable_fields = {const.FIELDS.UMBRELLA, const.FIELDS.MAIN_PRODUCT}
    for person_review, update_data in diff.items():
        model_fields = {
            it.name for it in models.PersonReview._meta.get_fields()
            if not isinstance(it, fields.reverse_related.ForeignObjectRel)
        } | fk_changable_fields
        update_fields = set(update_data) & model_fields
        model_kwargs = {}
        for field_name in update_fields | {const.FIELDS.ID}:
            if getattr(person_review, field_name) != const.DISABLED:
                if field_name in fk_changable_fields:
                    # HACK
                    # for related object we have to set ids instead
                    # person review contains related objects as dicts,
                    # so we need to get 'id' from those dicts
                    # /HACK
                    update_fields.discard(field_name)
                    field_name += '_id'
                    update_fields.add(field_name)
                model_kwargs[field_name] = getattr(person_review, field_name)
        update_models = group_by_updated_fields[frozenset(update_fields)]
        update_models.append(models.PersonReview(**model_kwargs))

    with transaction.atomic():
        for update_fields, update_models in group_by_updated_fields.items():
            bulk_update(
                objs=update_models,
                update_fields=update_fields,
            )


@helpers.timeit_no_args_logging
def save_for_different_action_change_models(subject, diff_bulk, subject_type):
    result = {}
    change_models = []
    comment_models = []
    for person_review, diff in diff_bulk.items():
        diff = diff.copy()
        change_model = bulk_helpers.build_change_model(
            subject=subject,
            person_review=person_review,
            changes=diff,
            subject_type=subject_type,
        )
        comment_model = bulk_helpers.build_comment_model(
            subject=subject,
            person_review=person_review,
            comment_text=diff.get('comment', {}).get('new'),
        )
        if change_model:
            change_models.append(change_model)
        if comment_model:
            comment_models.append(comment_model)
    if change_models:
        save_person_review_changes(change_models)
    if comment_models:
        save_person_review_comments(comment_models, subject_type)
    return result


@helpers.timeit_no_args_logging
def save_person_review_objects(data):
    for update, person_reviews in data.items():
        field, value = update
        models.PersonReview.objects.filter(
            id__in=[pr.id for pr in person_reviews]
        ).update(
            **{
                field: value,
                'updated_at': timezone.now(),
            }
        )


@helpers.timeit_no_args_logging
def save_person_review_roles(data, reviewers_update):
    if any(data.values()):
        if data['delete']:
            models.PersonReviewRole.objects.filter(data['delete']).delete()
        for reviewer_update_value, reviewer_update_filter in data['update']:
            models.PersonReviewRole.objects.filter(reviewer_update_filter).update(**reviewer_update_value)
        if data['create']:
            models.PersonReviewRole.objects.bulk_create(data['create'])

    if reviewers_update:
        person_review_ids = [person_review.id for person_review, _ in reviewers_update]

        ids_len = len(person_review_ids)
        chunk_offset = 0
        chunk_size = 100
        while chunk_offset < ids_len:
            pr_ids = person_review_ids[chunk_offset:chunk_offset + chunk_size]
            roles.async_denormalize_person_review_roles(person_review_ids=pr_ids)
            chunk_offset += chunk_size


@helpers.timeit_no_args_logging
def save_person_review_changes(prepared_models):
    models.PersonReviewChange.objects.bulk_create(prepared_models)


@helpers.timeit_no_args_logging
def save_person_review_comments(prepared_models, subject_type=const.PERSON_REVIEW_CHANGE_TYPE.PERSON):
    if subject_type == const.PERSON_REVIEW_CHANGE_TYPE.FILE:
        robot = staff_models.Person.objects.get(login='robot-review')
        for model in prepared_models:
            model.subject = robot
    models.PersonReviewComment.objects.bulk_create(prepared_models)


def prepare_reviewers_update(reviewers_update, login_to_person=None):
    review_role_update = []
    review_role_create = []
    review_role_delete = []

    for person_review, reviewer_update in reviewers_update:
        update, create, delete = build_reviewers_update(person_review, reviewer_update)
        review_role_update += update
        review_role_create += create
        review_role_delete += delete

    review_role_update, review_role_create, review_role_delete = build_reviewers_update_queries(
        review_role_update,
        review_role_create,
        review_role_delete,
        login_to_person,
    )

    return {
        'update': review_role_update,
        'create': review_role_create,
        'delete': review_role_delete,
    }


# todo: optimize
def build_reviewers_update(person_review, diff):
    old = bulk_helpers.flat_reviewers(diff['old'])
    new = bulk_helpers.flat_reviewers(diff['new'])
    old_size = len(diff['old'])
    new_size = len(diff['new'])
    delete = [reviewer for reviewer in old if reviewer not in new]
    create = [reviewer for reviewer in new if reviewer not in old]
    undeleted = [reviewer for reviewer in old if reviewer not in delete]
    create = [
        {
            'person_review_id': person_review.id,
            'person__login': reviewer,
            'position': position,
            'type': const.ROLE.PERSON_REVIEW.REVIEWER if position != new_size - 1
            else const.ROLE.PERSON_REVIEW.TOP_REVIEWER
        }
        for (reviewer, position) in create
    ]
    if old_size < new_size:
        update = [
            {
                'filter': {
                    'person_review_id': person_review.id,
                    'person__login': reviewer,
                    'type': const.ROLE.PERSON_REVIEW.TOP_REVIEWER,
                    'position': position
                },
                'update': {
                    'type': const.ROLE.PERSON_REVIEW.REVIEWER
                }
            }
            for reviewer, position in undeleted if position == old_size - 1
        ]
    elif old_size > new_size:
        update = [
            {
                'filter': {
                    'person_review_id': person_review.id,
                    'person__login': reviewer,
                    'type': const.ROLE.PERSON_REVIEW.REVIEWER,
                    'position': position
                },
                'update': {
                    'type': const.ROLE.PERSON_REVIEW.TOP_REVIEWER
                }
            }
            for reviewer, position in undeleted if position == new_size - 1
        ]
    else:
        update = []

    delete = [
        {
            'person_review_id': person_review.id,
            'person__login': reviewer,
            'position': position
        }
        for reviewer, position in delete
    ]

    return update, create, delete


def build_reviewers_update_queries(update, create, delete, login_to_person=None):
    unknown_persons = [reviewer['person__login'] for reviewer in create]
    if login_to_person is None:
        login_to_person = {
            value['login']: value['id']
            for value in staff_models.Person.objects.filter(
                login__in=unknown_persons
            ).values('login', 'id')
        }

    for reviewer in create:
        login = reviewer.pop('person__login')
        reviewer['person_id'] = login_to_person[login]

    create = [models.PersonReviewRole(**reviewer) for reviewer in create]
    if delete:
        delete = reduce(lambda x, y: x | y, [Q(**reviewer) for reviewer in delete])
        delete &= Q(type__in=(const.ROLE.PERSON_REVIEW.REVIEWER, const.ROLE.PERSON_REVIEW.TOP_REVIEWER))

    grouped_by_same_update = {}
    for reviewer_update in update:
        for field, value in reviewer_update['update'].items():
            group = grouped_by_same_update.setdefault((field, value), [])
            group.append(Q(**reviewer_update['filter']))

    for reviewer_update_value, reviewer_update_filter in grouped_by_same_update.items():
        grouped_by_same_update[reviewer_update_value] = reduce(lambda x, y: x | y, reviewer_update_filter)
    update = [({field: update_value}, _filter) for (field, update_value), _filter in grouped_by_same_update.items()]
    return update, create, delete


def update_calibration_person_review_models(diff):
    update_vals_to_ids = defaultdict(list)
    for id, diff in diff.items():
        for field, update_info in diff.items():
            new_val = update_info['new']
            update_vals_to_ids[(field, new_val)].append(id)

    with transaction.atomic():
        for (field, value), ids in update_vals_to_ids.items():
            models.CalibrationPersonReview.objects.filter(
                id__in=ids
            ).update(**{
                field: value,
                'updated_at': timezone.now(),
            })
