# coding=utf-8
from collections import defaultdict

import arrow
from django.db import (
    models,
    transaction,
)
from django.db.models import Q

from review.bi.logic import get_actual_grades
from review.core import models as core_models
from review.core import const as core_const
from review.core.logic import (
    roles,
    calibration_rights as rights,
    assemble,
)
from review.core.logic.assemble.fetch_person_reviews import fetch_chief_for_persons
from review.core.logic.domain_objs import Calibration
from review.staff import logic as staff_logic
from review.staff import models as staff_models


@transaction.atomic
def create(person, person_reviews, name='', start_date=None, finish_date=None, admins=None):
    calibration = core_models.Calibration.objects.create(
        status=core_const.CALIBRATION_STATUS.DRAFT,
        author=person,
        start_date=start_date,
        finish_date=finish_date,
        name=name,
    )
    now = arrow.now().datetime
    person_calibrations = [
        core_models.CalibrationPersonReview(
            calibration=calibration,
            person_review_id=person_review.id,
            updated_at=now,
        )
        for person_review in person_reviews
    ]
    core_models.CalibrationPersonReview.objects.bulk_create(person_calibrations)
    core_models.CalibrationRole.objects.create(
        person=person,
        calibration=calibration,
        type=core_const.ROLE.CALIBRATION.ADMIN,
    )
    person_review_ids = (pr.id for pr in person_reviews)
    if admins:
        add_admins(
            calibration=calibration,
            persons=admins,
            person_review_ids=person_review_ids,
        )
    else:
        # only for else coz add_admins contains denormalizing roles
        roles.async_denormalyze_calibration_roles(person_review_ids=list(person_review_ids))
    return calibration


@transaction.atomic
def edit_parameters(calibration, data):
    calibration_params = data.keys() & {
        'name',
        'finish_date',
        'start_date',
    }

    to_update = {
        key: value
        for key, value in data.items()
        if key in calibration_params and value != getattr(calibration, key)
    }
    core_models.Calibration.objects.filter(
        id=calibration.id
    ).update(**to_update)
    if 'admins' in data:
        update_admins(
            calibration=calibration,
            persons=data['admins'],
        )
    return calibration


@transaction.atomic
def update_admins(calibration, persons):
    core_models.CalibrationRole.objects.filter(
        calibration_id=calibration.id,
        type=core_const.ROLE.CALIBRATION.ADMIN,
    ).exclude(
        person__in=persons,
    ).delete()
    add_admins(calibration=calibration, persons=persons)


@transaction.atomic
def add_admins(calibration, persons, person_review_ids=None):
    new_admin_ids = {p.id for p in persons}
    existing_admins = set(core_models.CalibrationRole.objects.filter(
        calibration_id=calibration.id,
        type=core_const.ROLE.CALIBRATION.ADMIN,
    ).values_list('person_id', flat=True))
    to_create = new_admin_ids - existing_admins
    new_models = [
        core_models.CalibrationRole(
            person_id=pid,
            calibration_id=calibration.id,
            type=core_const.ROLE.CALIBRATION.ADMIN,
        )
        for pid in to_create
    ]
    core_models.CalibrationRole.objects.bulk_create(new_models)
    if person_review_ids is None:
        person_review_ids = set(core_models.CalibrationPersonReview.objects.filter(
            calibration_id=calibration.id
        ).values_list('person_review_id', flat=True))
    roles.async_denormalyze_calibration_roles(person_review_ids=list(person_review_ids))


def get_calibrators(calibration):
    calibrators = staff_models.Person.objects.filter(
        calibration_role__type=core_const.ROLE.CALIBRATION.CALIBRATOR,
        calibration_role__calibration__id=calibration.id,
    ).prefetch_related('department').order_by('-calibration_role__created_at_auto', 'id')
    if not calibrators:
        return []

    total_subordinates, in_calibration_subordinates = get_subordination_stats(calibrators, calibration)

    return [{
        'person': calibrator,
        'subordinates_total': total_subordinates.get(calibrator.id, 0),
        'subordinates_in_calibration': in_calibration_subordinates.get(calibrator.id, 0),
    } for calibrator in calibrators]


def get_admins(calibration):
    return staff_models.Person.objects.filter(
        calibration_role__type=core_const.ROLE.CALIBRATION.ADMIN,
        calibration_role__calibration_id=calibration.id,
    ).prefetch_related('department')


def get_review_ids(calibration):
    return list(
        core_models.CalibrationPersonReview.objects
        .filter(calibration_id=calibration.id)
        .distinct('person_review__review_id')
        .values_list('person_review__review_id', flat=True)
    )


def get_recommended_calibrators(calibration):
    persons_with_head = core_models.CalibrationPersonReview.objects.filter(
        calibration_id=calibration.id
    ).values_list(
        'person_review__person__superior__subject__id',
        'person_review__person__id',
    )
    calibrators = set(core_models.CalibrationRole.objects.filter(
        calibration_id=calibration.id,
        type=core_const.ROLE.CALIBRATION.CALIBRATOR,
    ).values_list('person_id', flat=True))
    persons = {person_id for _, person_id in persons_with_head}
    recommended_calibrator_ids = {head_id for head_id, _ in persons_with_head}
    recommended_calibrator_ids -= persons
    recommended_calibrator_ids -= calibrators
    if not recommended_calibrator_ids:
        return []
    recommended_calibrators = list(staff_models.Person.objects.filter(
        id__in=recommended_calibrator_ids
    ))
    total_subordinates, in_calibration_subordinates = get_subordination_stats(
        calibrators=recommended_calibrators,
        calibration=calibration,
    )
    result = [{
        'person': person,
        'subordinates_total': total_subordinates.get(person.id, 0),
        'subordinates_in_calibration': in_calibration_subordinates.get(person.id, 0),
    } for person in recommended_calibrators]
    result.sort(key=lambda x: x['subordinates_in_calibration'], reverse=True)
    return result


def prepopulate_calibrators_data(calibrators, subject):
    chiefs = fetch_chief_for_persons((c.id for c in calibrators), subject)

    for person in calibrators:
        person.chief = chiefs.get(person.id)


def get_subordination_stats(calibrators, calibration):
    if not calibrators:
        return {}, {}
    total_subordinates = defaultdict(int)
    empl_to_chiefs = defaultdict(set)
    active_review_ids = (
        core_models.Review.objects
        .filter(status=core_const.REVIEW_STATUS.IN_PROGRESS)
        .values_list('id')
    )
    empl_to_chiefs_q = (
        staff_logic.get_chief_to_empl_id(it.id for it in calibrators)
        .filter(Q(object__is_dismissed=False) | Q(object__person_review__review_id__in=active_review_ids))
        .distinct()
    )
    for chief_id, empl_id in empl_to_chiefs_q:
        total_subordinates[chief_id] += 1
        empl_to_chiefs[empl_id].add(chief_id)

    empl_in_calibration = (
        core_models.CalibrationPersonReview.objects
        .filter(calibration_id=calibration.id, person_review__person_id__in=empl_to_chiefs)
        .values_list('person_review__person_id', flat=True)
    )
    in_calibration_subordinates = defaultdict(int)
    for empl_id in empl_in_calibration:
        for chief_id in empl_to_chiefs[empl_id]:
            in_calibration_subordinates[chief_id] += 1
    return total_subordinates, in_calibration_subordinates


@transaction.atomic
def delete_roles(calibration, persons, role_type):
    core_models.CalibrationRole.objects.filter(
        calibration_id=calibration.id,
        person__in=persons,
        type=role_type
    ).delete()


@transaction.atomic
def add_calibrators(calibration, persons):
    existing_calibrators_ids = set(core_models.CalibrationRole.objects.filter(
        calibration_id=calibration.id,
        person__in=persons,
        type=core_const.ROLE.CALIBRATION.CALIBRATOR
    ).values_list('person_id', flat=True))
    persons_ids_in_calibration = set(core_models.CalibrationPersonReview.objects.filter(
        calibration_id=calibration.id,
    ).values_list('person_review__person__id', flat=True))
    existing_calibrators = []
    persons_in_calibration = []
    to_create = []
    for person in persons:
        if person.id in existing_calibrators_ids:
            existing_calibrators.append(person)
        elif person.id in persons_ids_in_calibration:
            persons_in_calibration.append(person)
        else:
            to_create.append(person)
    new_models = [
        core_models.CalibrationRole(
            person=person,
            type=core_const.ROLE.CALIBRATION.CALIBRATOR,
            calibration_id=calibration.id,
        )
        for person in to_create
    ]
    core_models.CalibrationRole.objects.bulk_create(new_models)
    person_review_ids = set(core_models.CalibrationPersonReview.objects.filter(
        calibration_id=calibration.id
    ).values_list('person_review_id', flat=True))
    roles.async_denormalyze_calibration_roles(person_review_ids=list(person_review_ids))

    def get_logins(li):
        return list([it.login for it in li])

    return dict(
        existing_calibrators=get_logins(existing_calibrators),
        persons_in_calibration=get_logins(persons_in_calibration),
        created=get_logins(to_create),
    )


@transaction.atomic
def follow_workflow(calibration, action):
    action_to_status = {
        rights.ACTIONS.STATUS_PUBLISH: core_const.CALIBRATION_STATUS.IN_PROGRESS,
        rights.ACTIONS.STATUS_ARCHIVE: core_const.CALIBRATION_STATUS.ARCHIVE,
    }
    new_status = action_to_status[action]
    core_models.Calibration.objects.filter(
        id=calibration.id
    ).update(status=new_status)
    roles.async_denormalyze_calibration_roles(calibration_id=calibration.id)


@transaction.atomic
def add_person_reviews(subject, calibration, person_reviews_ids):
    person_reviews = (
        core_models.PersonReview.objects
        .filter(id__in=person_reviews_ids)
        .values('id', 'person_id', 'person__login')
    )

    now = arrow.now().datetime
    id_to_person_review = {pr['id']: pr for pr in person_reviews}
    person_review_id_to_person_id = {pr['id']: pr['person_id'] for pr in person_reviews}
    existing_person_reviews = set(core_models.CalibrationPersonReview.objects.filter(
        calibration_id=calibration.id,
        person_review_id__in=id_to_person_review,
    ).values_list('person_review_id', flat=True))
    to_create = set(person_review_id_to_person_id) - existing_person_reviews

    # person with roles in calibration can't be added as calibrated
    calibration_roles = core_models.CalibrationRole.objects.filter(
        calibration_id=calibration.id,
        person_id__in=(id_to_person_review[id_]['person_id'] for id_ in to_create),
    ).values_list('person_id', 'type')
    with_roles = defaultdict(list)
    if calibration_roles:
        person_id_to_person_review = {pr['person_id']: pr for pr in person_reviews}
        for person_id, role in calibration_roles:
            person_review = person_id_to_person_review[person_id]
            with_roles[role].append(person_review['person__login'])
            to_create.discard(person_review['id'])

    person_calibrations = [
        core_models.CalibrationPersonReview(
            calibration_id=calibration.id,
            person_review_id=person_review_id,
            updated_at=now,
        )
        for person_review_id in to_create
    ]
    core_models.CalibrationPersonReview.objects.bulk_create(person_calibrations)
    roles.async_denormalyze_calibration_roles(person_review_ids=list(to_create))
    created, existed = [], []
    for pr in person_reviews:
        if pr['id'] in to_create:
            created.append(pr['person__login'])
        elif pr['id'] in existing_person_reviews:
            existed.append(pr['person__login'])
    return created, existed, dict(with_roles), person_reviews


def copy(person, calibration):
    # type: (staff_models.Person, Calibration) -> int
    cprs = assemble.get_calibration_person_reviews(person, {'calibration_id': calibration.id}, {core_const.FIELDS.ID})
    calibration = create(
        person=person,
        person_reviews=[it.person_review for it in cprs],
        name='Копия калибровки: {}'.format(calibration.name),
        start_date=calibration.start_date,
        finish_date=calibration.finish_date,
        admins=[person],
    )
    return calibration.id
