# coding: utf-8
"""
Rights for calibrations
There are three layers of rights:
1. Roles
2. Permissions
3. Actions

Roles are used just to group several permission into one attribute.
Permission describes that user can do something in general.
Action describes that user can do something right now.
Action can depend on permission and something more specific - status of the object for example.

When we do something with object, that is not available for every user at every moment -
we have to check if user has Action to do it.
"""

from collections import defaultdict
from itertools import chain

from review.lib import errors
from review.core import const


class ACTIONS(object):
    const_creator = const.ConstCreator()
    cc = const_creator.create_const

    EDIT_CALIBRATION_PERSON_REVIEWS = cc('edit_calibration_person_reviews')
    EDIT_CALIBRATORS = cc('edit_calibrators')
    EDIT_PARAMETERS = cc('edit_parameters')
    COPY_CALIBRATION = cc('copy_calibration')
    LIST_CALIBRATORS = cc('list_calibrators')
    LIST_PERSON_REVIEWS = cc('list_person_reviews')
    STATUS_PUBLISH = cc('publish')
    STATUS_ARCHIVE = cc('archive')
    DELETE = cc('delete')
    READ_FEEDBACK = cc('read_feedback')

    ALL = const_creator.get_all()
    del cc, const_creator

    STATUS = frozenset({
        STATUS_ARCHIVE,
        STATUS_PUBLISH,
    })


class PERMS(object):
    const_creator = const.ConstCreator()
    cc = const_creator.create_const

    EDIT_CALIBRATION_PERSON_REVIEWS = cc('edit_person_reviews')
    EDIT_CALIBRATORS = cc('edit_calibrators')
    EDIT_PARAMETERS = cc('edit_parameters')
    COPY_CALIBRATION = cc('copy_calibration')
    LIST_CALIBRATORS = cc('list_calibrators')
    LIST_PERSON_REVIEWS = cc('list_person_reviews')
    STATUS = cc('status')
    DELETE = cc('delete')
    READ_FEEDBACK = cc('read_feedback')

    ALL = const_creator.get_all()
    del cc, const_creator

    ROLE_PERMS = defaultdict(frozenset)
    ROLE_PERMS.update({
        const.ROLE.GLOBAL.CALIBRATION_CREATOR: frozenset({
            EDIT_PARAMETERS,
        }),
        const.ROLE.CALIBRATION.ADMIN: frozenset({
            EDIT_CALIBRATORS,
            COPY_CALIBRATION,
            LIST_CALIBRATORS,
            EDIT_CALIBRATION_PERSON_REVIEWS,
            EDIT_PARAMETERS,
            DELETE,
            LIST_PERSON_REVIEWS,
            STATUS,
            READ_FEEDBACK,
        }),
        const.ROLE.CALIBRATION.CALIBRATOR: frozenset({
            LIST_PERSON_REVIEWS,
            READ_FEEDBACK,
        }),
    })


ACTION_TO_PERM = {
    ACTIONS.COPY_CALIBRATION: PERMS.COPY_CALIBRATION,
    ACTIONS.EDIT_CALIBRATION_PERSON_REVIEWS: PERMS.EDIT_CALIBRATION_PERSON_REVIEWS,
    ACTIONS.EDIT_CALIBRATORS: PERMS.EDIT_CALIBRATORS,
    ACTIONS.EDIT_PARAMETERS: PERMS.EDIT_PARAMETERS,
    ACTIONS.LIST_CALIBRATORS: PERMS.LIST_CALIBRATORS,
    ACTIONS.LIST_PERSON_REVIEWS: PERMS.LIST_PERSON_REVIEWS,
    ACTIONS.STATUS_PUBLISH: PERMS.STATUS,
    ACTIONS.STATUS_ARCHIVE: PERMS.STATUS,
    ACTIONS.READ_FEEDBACK: PERMS.READ_FEEDBACK,
    ACTIONS.DELETE: PERMS.DELETE,
}


def get_permissions(calibration):
    assert calibration.roles is not None, 'Roles are not set'

    return set(chain.from_iterable(
        PERMS.ROLE_PERMS[role] for role in calibration.roles
    ))


def get_actions_state(calibration):
    assert calibration.permissions is not None, 'Permissions are not set'
    assert calibration.roles is not None, 'Roles are not set'
    assert calibration.status is not None, 'Status is not set'

    cur_perms = calibration.permissions
    actions_state = {
        act: const.OK if ACTION_TO_PERM[act] in cur_perms else const.NO_ACCESS
        for act in ACTIONS.ALL
    }

    # process to status is possible only from specific current status
    if calibration.status != const.CALIBRATION_STATUS.DRAFT:
        actions_state[ACTIONS.STATUS_PUBLISH] = const.NO_ACCESS
        actions_state[ACTIONS.DELETE] = const.NO_ACCESS
    if calibration.status != const.CALIBRATION_STATUS.IN_PROGRESS:
        actions_state[ACTIONS.STATUS_ARCHIVE] = const.NO_ACCESS

    # some actions are not available while calibration is archived
    if calibration.status == const.CALIBRATION_STATUS.ARCHIVE:
        actions_state[ACTIONS.EDIT_CALIBRATION_PERSON_REVIEWS] = const.NO_ACCESS
        actions_state[ACTIONS.EDIT_PARAMETERS] = const.NO_ACCESS

    is_admin = const.ROLE.CALIBRATION.ADMIN in calibration.roles
    is_draft = calibration.status == const.CALIBRATION_STATUS.DRAFT
    if not is_admin and is_draft:
        actions_state[ACTIONS.LIST_PERSON_REVIEWS] = const.NO_ACCESS

    # feedback available only if calibration is in progress
    if calibration.status != const.CALIBRATION_STATUS.IN_PROGRESS:
        actions_state[ACTIONS.READ_FEEDBACK] = const.NO_ACCESS

    if actions_state[ACTIONS.LIST_PERSON_REVIEWS] == const.NO_ACCESS:
        # user have to see calibration_person_reviews to copy them
        actions_state[ACTIONS.COPY_CALIBRATION] = const.NO_ACCESS

    return actions_state


def ensure_has_action(calibration, action):
    assert calibration.actions is not None, 'Actions are not set'

    if calibration.actions.get(action, const.NO_ACCESS) != const.OK:
        raise errors.PermissionDenied(
            action=action,
            type='calibration',
            id=calibration.id,
        )
