# coding: utf-8
import urllib.error
import urllib.parse
import urllib.request
from datetime import date

import yenv
from django.conf import settings
from django_ids.auth import Auth
from django_ids.connector.http import HttpConnector
from django_ids.services.goals import connector as goals_connector

from review.core import const
from review.core.logic import calibration_rights


def get_connector_config():
    return yenv.choose_key_by_type(
        dict_={
            'testing': {
                'protocol': 'https',
                'host': 'fb.test.yandex-team.ru',
            },
            'production': {
                'protocol': 'https',
                'host': 'fb.yandex-team.ru',
            },
        },
        fallback=True,
    )


class FeedbackPersonConnector(HttpConnector):

    default_connect_timeout = 60

    url_patterns = {
        'subordinates': '/api/feedback/subordinates/{login}/',
    }

    @property
    def config(self):
        return get_connector_config()

    def handle_bad_response(self, response):
        status_code = response.status_code
        if status_code == 403:
            return
        return super(FeedbackPersonConnector, self).handle_bad_response(response)


class FeedbackRobotConnector(HttpConnector):

    default_connect_timeout = 60

    url_patterns = {
        'subordinates': '/api/feedback/subordinates/{login}/for-review/',
    }

    def __init__(self, **kwargs):
        assert 'auth' not in kwargs, 'Auth have to be detected automatically'
        auth = Auth(oauth_token=settings.ROBOT_OAUTH_TOKEN)
        super(FeedbackRobotConnector, self).__init__(auth=auth, **kwargs)

    @property
    def config(self):
        return get_connector_config()


# TODO: delete after frontend switched to urls
# https://st.yandex-team.ru/CIA-1062
def get_goals(auth, person_review_id):
    from review.core.logic import assemble
    person_review = assemble.get_person_review(
        subject=auth.user,
        id=person_review_id,
        fields_requested={
            const.FIELDS.PERSON_ID,
            const.FIELDS.REVIEW_START_DATE,
            const.FIELDS.REVIEW_GOALS_FROM_DATE,
            const.FIELDS.REVIEW_GOALS_TO_DATE,
        }
    )

    if person_review is None:
        return []

    result_set = goals_connector.get_result_set(
        auth=auth,
        resource='goals',
        connector_params={},
        params={
            'person_id': person_review.person_id,
            # потому что в голс deadline почему-то совпадает с началом квартала
            'deadline__min': get_quarter_first_day(
                person_review.review_goals_from_date
            ).isoformat(),
            'deadline__max': get_quarter_first_day(
                person_review.review_goals_to_date
            ).isoformat(),
        },
        headers={
            'DEBUG-LOGIN': auth.login,
        },
    )
    return list(result_set)


def get_feedback_for_person_review(auth, id):
    from review.core.logic import assemble
    person_review = assemble.get_person_review(
        subject=auth.user,
        id=id,
        fields_requested={
            const.FIELDS.PERSON_ID,
            const.FIELDS.PERSON_LOGIN,
            const.FIELDS.REVIEW_START_DATE,
            const.FIELDS.REVIEW_FEEDBACK_FROM_DATE,
            const.FIELDS.REVIEW_FEEDBACK_TO_DATE,
        }
    )

    if person_review is None:
        return {}

    connector = FeedbackPersonConnector(auth=auth)
    response = connector.get(
        resource='subordinates',
        url_vars={'login': person_review.person_login},
        params={
            'from_date': person_review.review_feedback_from_date.isoformat(),
            'to_date': person_review.review_feedback_to_date.isoformat(),
        },
        headers={
            'DEBUG-LOGIN': auth.login,
        },
    )
    _skip_revoked_feedbacks(feedback_data=response)
    return response


def get_feedback_for_calibration_person_review(auth, id):
    from review.core.logic import assemble
    calibration_person_review = assemble.get_calibration_person_review(
        subject=auth.user,
        id=id,
        requested_person_review_fields={
            const.FIELDS.PERSON_ID,
            const.FIELDS.PERSON_LOGIN,
            const.FIELDS.REVIEW_FEEDBACK_FROM_DATE,
            const.FIELDS.REVIEW_FEEDBACK_TO_DATE,
        },
    )
    if not calibration_person_review:
        return {}

    calibration = assemble.get_calibration(
        subject=auth.user,
        id=calibration_person_review.calibration_id,
    )
    calibration_rights.ensure_has_action(
        calibration,
        calibration_rights.ACTIONS.READ_FEEDBACK,
    )

    person_review = calibration_person_review.person_review

    response = FeedbackRobotConnector().get(
        resource='subordinates',
        url_vars={'login': person_review.person_login},
        params=dict(
            from_date=person_review.review_feedback_from_date.isoformat(),
            to_date=person_review.review_feedback_to_date.isoformat(),
        ),
        headers={'DEBUG-LOGIN': 'robot-review'},
    )
    _skip_revoked_feedbacks(feedback_data=response)
    clean_dict = dict.fromkeys((
        'reporter',
        'reporter_staff',
        'reporter_type',
        'persons',
    ))
    to_clean = (
        fb for fb in response['feedbacks']
        if fb['persons'] and len(fb['persons']) > 1
    )
    for fb in to_clean:
        fb.update(clean_dict)
    return response


def _skip_revoked_feedbacks(feedback_data):
    cur_feedback = feedback_data.setdefault('feedbacks', [])
    feedback_data['feedbacks'] = [
        feedback for feedback in cur_feedback
        if not feedback['is_revoked']
    ]
    return feedback_data


def get_goals_url(subject, person_review):
    connector = goals_connector.GoalsConnector(auth=None)
    base_url = connector.build_url(resource='goals')
    params = {
        'person_id': person_review.person_id,
        # потому что в голс deadline почему-то совпадает с началом квартала
        'deadline__min': get_quarter_first_day(
            person_review.review_goals_from_date
        ).isoformat(),
        'deadline__max': get_quarter_first_day(
            person_review.review_goals_to_date
        ).isoformat(),
    }
    return build_cia_url(
        base_url=base_url,
        params=params,
        subject=subject,
    )


def get_st_goals_url(subject, person_review):
    base_url = settings.ST_GOALS_URL
    login = person_review.person_login
    deadline__min = get_quarter_first_day(
        person_review.review_goals_from_date
    ).isoformat()
    deadline__max = get_quarter_first_day(
        person_review.review_goals_to_date
    ).isoformat()
    query = (
        'queue:GOALZ '
        'and (assignee:{login} or customers:{login} or participants:{login}) '
        'and deadline:<={deadline__max} deadline:>={deadline__min}'
        .format(
            login=login,
            deadline__min=deadline__min,
            deadline__max=deadline__max,
        )
    )
    params = {
        'query': query,
    }
    return base_url + '?' + urllib.parse.urlencode(params)


def get_feedback_url(subject, person_review):
    connector = FeedbackPersonConnector(auth=None)
    base_url = connector.build_url(
        resource='subordinates',
        url_vars={'login': person_review.person_login},
    )
    params = {
        'from_date': person_review.review_feedback_from_date.isoformat(),
        'to_date': person_review.review_feedback_to_date.isoformat(),
    }
    return build_cia_url(
        base_url=base_url,
        params=params,
        subject=subject,
    )


def build_cia_url(base_url, params, subject):
    url = base_url + '?' + '&'.join([
        '{key}={val}'.format(key=key, val=val) for
        key, val in list(params.items())
    ])
    if yenv.type not in ('production', 'prestable'):
        url += '&debug-login={}'.format(subject.login)
    return url


def get_quarter_first_day(dt):
    month_to_first_quarter_month = {
        1: 1,
        2: 1,
        3: 1,
        4: 4,
        5: 4,
        6: 4,
        7: 7,
        8: 7,
        9: 7,
        10: 10,
        11: 10,
        12: 10,
    }
    return date(
        year=dt.year,
        month=month_to_first_quarter_month[dt.month],
        day=1,
    )
