from typing import List

import attr

from review.core import const


class PersonReview(object):

    def __init__(self, **kwargs):
        for field in const.FIELDS.ALL:
            setattr(self, field, kwargs.pop(field, const.NOT_SET))

    def __hash__(self):
        return self.id

    def __eq__(self, other):
        return self.id == other.id

    def __repr__(self):
        return '<PRE: %s @ %s>' % (
            self.person_login if self.person_login is not const.NOT_SET
            else self.person_id,
            self.review_id
        )


@attr.s(
    slots=True,
    hash=False,
    repr=False
)
class PersonReviewComment(object):

    id = attr.ib(default=const.NOT_SET)
    person_review_id = attr.ib(default=const.NOT_SET)
    subject_id = attr.ib(default=const.NOT_SET)
    subject_login = attr.ib(default=const.NOT_SET)
    subject_first_name = attr.ib(default=const.NOT_SET)
    subject_last_name = attr.ib(default=const.NOT_SET)
    subject_gender = attr.ib(default=const.NOT_SET)
    subject_type = attr.ib(default=const.NOT_SET)
    text_wiki = attr.ib(default=const.NOT_SET)
    created_at = attr.ib(default=const.NOT_SET)
    updated_at = attr.ib(default=const.NOT_SET)
    actions = attr.ib(default=attr.Factory(dict))


@attr.s(
    slots=True,
    hash=False,
    repr=False
)
class PersonReviewChange(object):

    id = attr.ib(default=const.NOT_SET)
    person_review_id = attr.ib(default=const.NOT_SET)
    subject_id = attr.ib(default=const.NOT_SET)
    subject_login = attr.ib(default=const.NOT_SET)
    subject_first_name = attr.ib(default=const.NOT_SET)
    subject_last_name = attr.ib(default=const.NOT_SET)
    subject_gender = attr.ib(default=const.NOT_SET)
    subject_type = attr.ib(default=const.NOT_SET)
    diff = attr.ib(default=const.NOT_SET)
    created_at = attr.ib(default=const.NOT_SET)


@attr.s(
    slots=True,
    hash=False,
    repr=False
)
class Review(object):

    id = attr.ib(default=const.NOT_SET)
    name = attr.ib(default=const.NOT_SET)
    type = attr.ib(default=const.NOT_SET)
    bonus_type = attr.ib(default=const.NOT_SET)
    bonus_reason = attr.ib(default=const.NOT_SET)
    status = attr.ib(default=const.NOT_SET)
    start_date = attr.ib(default=const.NOT_SET)
    finish_date = attr.ib(default=const.NOT_SET)

    notify_reminder_days = attr.ib(default=const.NOT_SET)
    notify_reminder_date_from = attr.ib(default=const.NOT_SET)
    notify_reminder_date_to = attr.ib(default=const.NOT_SET)
    notify_events_other = attr.ib(default=const.NOT_SET)
    notify_events_superreviewer = attr.ib(default=const.NOT_SET)

    evaluation_from_date = attr.ib(default=const.NOT_SET)
    evaluation_to_date = attr.ib(default=const.NOT_SET)
    feedback_from_date = attr.ib(default=const.NOT_SET)
    feedback_to_date = attr.ib(default=const.NOT_SET)

    finish_feedback_date = attr.ib(default=const.NOT_SET)
    finish_submission_date = attr.ib(default=const.NOT_SET)
    finish_calibration_date = attr.ib(default=const.NOT_SET)
    finish_approval_date = attr.ib(default=const.NOT_SET)
    include_dismissed_after_date = attr.ib(default=const.NOT_SET)
    salary_date = attr.ib(default=const.NOT_SET)

    scale_id = attr.ib(default=const.NOT_SET)
    mark_mode = attr.ib(default=const.NOT_SET)
    goldstar_mode = attr.ib(default=const.NOT_SET)
    level_change_mode = attr.ib(default=const.NOT_SET)
    salary_change_mode = attr.ib(default=const.NOT_SET)
    bonus_mode = attr.ib(default=const.NOT_SET)
    options_rsu_mode = attr.ib(default=const.NOT_SET)
    options_rsu_unit = attr.ib(default=const.NOT_SET)
    deferred_payment_mode = attr.ib(default=const.NOT_SET)

    kpi_loaded = attr.ib(default=const.NOT_SET)

    product_schema_loaded = attr.ib(default=const.NOT_SET)

    marks_scale = attr.ib(default=const.NOT_SET)

    goodies = attr.ib(default=())

    admins = attr.ib(default=const.NOT_SET)
    super_reviewers = attr.ib(default=const.NOT_SET)
    accompanying_hrs = attr.ib(default=const.NOT_SET)

    roles = attr.ib(default=attr.Factory(list))
    permissions = attr.ib(default=attr.Factory(dict))
    actions = attr.ib(default=attr.Factory(dict))

    def __hash__(self):
        return self.id

    def __repr__(self):
        return '<RE: %s>' % self.id


@attr.s(
    slots=True,
    hash=False,
    repr=False
)
class Goodie(object):

    id = attr.ib(default=const.NOT_SET)
    level = attr.ib(default=const.NOT_SET)
    mark = attr.ib(default=const.NOT_SET)
    goldstar = attr.ib(default=const.NOT_SET)
    level_change = attr.ib(default=const.NOT_SET)
    salary_change = attr.ib(default=const.NOT_SET)
    bonus = attr.ib(default=const.NOT_SET)
    options_rsu = attr.ib(default=const.NOT_SET)


def get_most_important_role(review):
    return min(
        review.roles,
        key=lambda role: const.ROLE.ORDER_OF_IMPORTANCE.get(role, float('inf')),
    )


def sort_reviews_by_importance(reviews):
    """
    Review importance order:
    by status: in progress > finished > archived
    by date: descending
    by important roles: reviewer > super reviewer > calibrator > calibration admin > reader > admin of review
    :param reviews: List[Review]
    :return: List[Review]
    """
    return sorted(
        reviews,
        key=lambda review: (
            const.REVIEW_STATUS.ORDERING.index(review.status),
            -review.start_date.toordinal(),
            const.ROLE.ORDER_OF_IMPORTANCE.get(
                get_most_important_role(review),
                float('inf'),
            ),
        ),
    )


def get_most_relevant_review(reviews):
    """
    There is most relevant id only if reviews list is not empty and
    there are not several reviews in progress.
    Reviews has to be sorted.
    :param reviews: List[Review], sorted by importance
    :return: Review, one of reviews or None
    """
    self_reviews_in_progress = [
        r for r in reviews
        if (
            r.status == const.REVIEW_STATUS.IN_PROGRESS and
            # global roles have access to non-self reviews
            any(role not in const.ROLE.GLOBAL.ALL for role in r.roles)
        )
    ]
    has_several_active = len(self_reviews_in_progress) > 1
    if reviews and not has_several_active:
        return reviews[0]


@attr.s(
    slots=True,
    hash=False,
    repr=False
)
class Calibration(object):
    id = attr.ib(default=const.NOT_SET)
    author = attr.ib(default=const.NOT_SET)
    name = attr.ib(default=const.NOT_SET)
    status = attr.ib(default=const.NOT_SET)
    start_date = attr.ib(default=const.NOT_SET)
    finish_date = attr.ib(default=const.NOT_SET)

    roles = attr.ib(default=attr.Factory(list))
    admins = attr.ib(default=attr.Factory(list))
    permissions = attr.ib(default=attr.Factory(dict))
    actions = attr.ib(default=attr.Factory(dict))
    review_ids = attr.ib(default=attr.Factory(list))

    def __hash__(self):
        return self.id

    def __repr__(self):
        return '<CE: %s>' % (
            self.name if self.name is not const.NOT_SET
            else self.id,
        )


@attr.s(
    slots=True,
    hash=False,
    repr=False
)
class CalibrationPersonReview(object):
    id = attr.ib(default=const.NOT_SET)
    discuss = attr.ib(default=const.NOT_SET)
    person_review = attr.ib(default=const.NOT_SET)
    calibration_id = attr.ib(default=const.NOT_SET)
    updated_at = attr.ib(default=const.NOT_SET)

    roles = attr.ib(default=attr.Factory(list))
    permissions = attr.ib(default=attr.Factory(dict))
    actions = attr.ib(default=attr.Factory(dict))

    def __hash__(self):
        return self.id

    def __repr__(self):
        return '<CPRE: %s>' % self.id
