# coding: utf-8

from collections import defaultdict

from review.core import models
from review.core import const

from review.core.models import PersonReview, PersonReviewRole
from review.staff.models import Person, Subordination
from review.staff import const as staff_const


def ensure_review_model(review):
    if isinstance(review, int):
        return models.Review.objects.get(id=review)
    return review


def select_filters(chosen, stage_subset):
    return {
        key: value for key, value in chosen.items()
        if key in stage_subset and
        value is not const.NOT_SET
    }


def select_review_filters(chosen):
    selected_filters = select_filters(
        chosen,
        stage_subset=const.FILTERS.REVIEW_FILTERS)
    if 'statuses' in selected_filters:
        selected_filters['statuses'] = set(
            selected_filters.get('statuses') or const.REVIEW_STATUS.ALL
        )
    return selected_filters


def select_calibration_filters(chosen):
    selected_filters = select_filters(
        chosen,
        stage_subset=const.FILTERS.CALIBRATION_FILTERS,
    )
    return selected_filters


def get_first_reviewer_ids(person_review_ids):
    """по id person_review находит id первых согласующих из цепочки (их может быть > 1)"""
    reviewers = defaultdict(list)  # {pr_id: [reviewer_id, reviewer_id, ..]}
    reviewers_data = (
        PersonReviewRole.objects
        .filter(person_review_id__in=person_review_ids)
        .filter(type__in=(const.ROLE.PERSON_REVIEW.REVIEWER, const.ROLE.PERSON_REVIEW.TOP_REVIEWER))
        .filter(position=0)
        .values_list('person_review_id', 'person_id')
    )
    for pr_id, person_id in reviewers_data:
        reviewers[pr_id].append(person_id)
    return dict(reviewers)


def get_chief_ids(person_ids):
    """Возвращает id непосредственного руководителя по id сотрудика"""
    return dict(
        Subordination.objects
        .filter(object_id__in=person_ids)
        .filter(type=staff_const.SUBORDINATION.DIRECT, position=0)
        .values_list('object_id', 'subject_id')
    )


def get_persons_details(person_ids):
    """Информация по сотруднику по его id"""
    persons_data = (
        Person.objects
        .filter(id__in=person_ids)
        .values(
            'id',
            'login',
            'first_name_ru',
            'last_name_ru',
            'first_name_en',
            'last_name_en',
            'language',
            'work_email',
        )
    )
    return {p['id']: p for p in persons_data}


def get_reviews_details(person_review_ids):
    """по person_review_id возвращает информации о ревью, к которому оно относится"""
    reviews_data = (
        PersonReview.objects
        .filter(id__in=person_review_ids)
        .values('id', 'review_id', 'review__name')
    )
    return {r['id']: r for r in reviews_data}


def localized(dict_data, lang):
    """убирает поля с _en и _ru, оставляя само навание поля по переданному языку"""
    new_dict = {}
    correct_ending = '_' + lang
    for key, value in dict_data.items():
        ending = key[-3:]
        if ending not in ('_ru', '_en'):
            new_dict[key] = value
        elif key.endswith(correct_ending):
            new_dict[key[:-3]] = value
    return new_dict
