# coding=utf-8
import yenv
import datetime
from collections import defaultdict
from typing import DefaultDict, Dict, List, Optional, Iterable

import arrow
import attr

from django.conf import settings
from django.db import transaction
from django.db.models import OuterRef, Q, Subquery, QuerySet
from django.utils import timezone

from review.core import (
    const,
    models,
)
from review.core.logic import roles, bulk, domain_objs
from review.lib import errors, helpers
from review.lib.std import remove_adjacent_doubles
from review.staff import logic
from review.staff import models as staff_models


def create_review(author, params):
    # type: (staff_models.Person, Dict) -> models.Review
    required_fields = {
        'name',
        'start_date',
        'finish_date',
        'author',
    }

    review_fields = models.Review.get_real_fields() | {'scale'}
    actual_params = [__value1 for __value1 in params.items() if __value1[1] is not None and __value1[1] != '']
    review_params = [key__ for key__ in actual_params if key__[0] in review_fields]
    review_params = dict(review_params)
    actual_params = dict(actual_params)

    if not (required_fields & actual_params.keys()):
        raise Exception('Missing fields in parameters {}'.format(actual_params))

    with transaction.atomic():
        review = models.Review.objects.create(**review_params)

        if 'goodies' in actual_params:
            _create_review_goodies(review, actual_params['goodies'])
        models.ReviewRole.objects.create(
            person=author,
            review_id=review.id,
            type=const.ROLE.REVIEW.ACCOMPANYING_HR,
        )

    return review


@transaction.atomic
def edit_review(review, params):
    # type: (domain_objs.Review, Dict) -> domain_objs.Review
    actual_params = dict([__value for __value in params.items() if __value[1] is not None])

    current_review_roles = models.ReviewRole.objects.filter(
        review_id=review.id,
    ).values(
        'id',
        'person__login',
        'type',
    )

    type_login_to_role_id = {
        (it['type'], it['person__login']): it['id']
        for it in current_review_roles
    }
    role_to_new_persons_state = {
        const.ROLE.REVIEW.ADMIN: actual_params.get('admins', []),
        const.ROLE.REVIEW.SUPERREVIEWER: actual_params.get('super_reviewers', []),
        const.ROLE.REVIEW.ACCOMPANYING_HR: actual_params.get('accompanying_hrs', []),
    }
    role_to_add_list = defaultdict(list)
    for role_type, new_persons_state in role_to_new_persons_state.items():
        add_list = role_to_add_list[role_type]
        for person in new_persons_state:
            role_id = type_login_to_role_id.pop((role_type, person.login), None)
            if not role_id:
                add_list.append(person)
    to_del_role_ids = list(type_login_to_role_id.values())

    review_fields = models.Review.get_real_fields()
    review_update = {
        key: value for key, value in actual_params.items()
        if key in review_fields and getattr(review, key) != value
    }
    new_scale_id = actual_params.get('scale')
    if new_scale_id != review.scale_id:
        review_update['scale'] = new_scale_id

    models.Review.objects.filter(id=review.id).update(**review_update)

    unavailable_roles = [
        role for act, role in const.REVIEW_ACTIONS.EDIT_TO_ROLE.items()
        if review.actions[act] != const.OK
    ]
    (
        models.ReviewRole.objects
        .filter(id__in=to_del_role_ids)
        .exclude(type__in=unavailable_roles)
        .delete()
    )
    _create_review_roles(
        review=review,
        admins=role_to_add_list[const.ROLE.REVIEW.ADMIN],
        super_reviewers=role_to_add_list[const.ROLE.REVIEW.SUPERREVIEWER],
        accompying_hrs=role_to_add_list[const.ROLE.REVIEW.ACCOMPANYING_HR],
    )

    if 'goodies' in actual_params:
        models.Goodie.objects.filter(review_id=review.id).delete()
        _create_review_goodies(review, actual_params['goodies'])

    return review


def _create_review_roles(review, admins, super_reviewers, accompying_hrs):
    # type: (domain_objs.Review, Iterable[staff_models.Person], Iterable[staff_models.Person], Iterable[staff_models.Person]) -> None
    role_kwargs = []
    if review.actions[const.REVIEW_ACTIONS.EDIT_ADMINS] == const.OK:
        role_kwargs += [{'person': p, 'type': const.ROLE.REVIEW.ADMIN} for p in admins]
    if review.actions[const.REVIEW_ACTIONS.EDIT_SUPERREVIEWERS] == const.OK:
        role_kwargs += [{'person': p, 'type': const.ROLE.REVIEW.SUPERREVIEWER} for p in super_reviewers]
    if review.actions[const.REVIEW_ACTIONS.EDIT_ACCOMPANYING_HRS] == const.OK:
        role_kwargs += [{'person': p, 'type': const.ROLE.REVIEW.ACCOMPANYING_HR} for p in accompying_hrs]
    models.ReviewRole.objects.bulk_create([models.ReviewRole(review_id=review.id, **kw) for kw in role_kwargs])


def _create_review_goodies(review, goodies):
    goodies = [
        models.Goodie(
            review_id=review.id,
            **goodie
        )
        for goodie in goodies
    ]

    models.Goodie.objects.bulk_create(goodies)


def _structure_change_to_working(person_query, department_subjects, person_subjects_to_exclude):
    # type: (staff_models.Person.objects, List[ReviewSubject], List[ReviewSubject]) -> DefaultDict[int, List[int]]
    # Сортируем по глубине подразделений (более мелкие в конце)
    # чтобы при ситуации, когда для большого подразделения
    # указана одна дата, а для его части другая,
    # мы для этой части поставим дату, которая для неё указана,
    # а не для всего подразделения.
    # Исключаем сотрудников из person_subjects_to_exclude, т.к. они обрабатываются отдельно.
    path_to_structure_change = ((d.department.path, d.structure_change.id) for d in department_subjects)
    path_to_structure_change = sorted(path_to_structure_change, key=lambda path__: len(path__[0]))
    result = defaultdict(list)
    person_id_to_path = list(
        person_query
        .filter(is_dismissed=False)
        .exclude(id__in=[subj.person.id for subj in person_subjects_to_exclude])
        .values_list('id', 'department__path')
    )
    for dep_path, structure_change in path_to_structure_change:
        for person_id, person_path in person_id_to_path:
            if person_path.startswith(dep_path):
                result[structure_change].append(person_id)
    return result


def _dismissed_from_departments_person_ids(person_query, dep_filter, include_date):
    # type: (staff_models.Person.objects, Q, datetime.date) -> List[int]
    if not include_date:
        return {}
    filter_q = dep_filter & Q(is_dismissed=True, quit_at__gte=include_date)
    person_ids_qs = person_query.filter(filter_q).values_list('id', flat=True)
    return list(person_ids_qs)


@attr.s
class ReviewSubject(object):
    type = attr.ib(type=str)
    structure_change = attr.ib(type=staff_models.StaffStructureChange)
    person = attr.ib(type=Optional[staff_models.Person], default=None)
    department = attr.ib(type=Optional[staff_models.Department], default=None)


@transaction.atomic
def add_review_subjects(person, review, subjects):
    # type: (staff_models.Person, domain_objs.Review, List[ReviewSubject]) -> object
    dep_subjs, pers_subjs = [], []
    for subj in subjects:
        if subj.type == 'department':
            dep_subjs.append(subj)
        else:
            pers_subjs.append(subj)

    person_q = staff_models.Person.objects.filter(is_robot=False)
    dep_filter = logic.get_persons_from_department_forest_q(
        roots=[subj.department for subj in dep_subjs],
        include_root=True,
    )

    # Сотрудники заданные подразделениями
    structure_change_id_to_person_ids = _structure_change_to_working(
        person_query=person_q.filter(dep_filter),
        department_subjects=dep_subjs,
        person_subjects_to_exclude=pers_subjs,
    )

    # Сотрудники уволенные из подразделений после review.include_dismissed_after_date
    _dismissed_person_ids = _dismissed_from_departments_person_ids(
        person_query=person_q,
        dep_filter=dep_filter,
        include_date=review.include_dismissed_after_date,
    )
    if _dismissed_person_ids:
        last_structure_change_id = (
            staff_models.StaffStructureChange.objects
            .order_by('id')
            .values_list('id', flat=True)
            .last()
        )
        structure_change_id_to_person_ids[last_structure_change_id].extend(_dismissed_person_ids)

    # Сотрудники заданные лично
    for subject in pers_subjs:
        ssc_id, p_id = subject.structure_change.id, subject.person.id
        structure_change_id_to_person_ids[ssc_id].append(p_id)

    if not structure_change_id_to_person_ids:
        return

    to_update = _create_person_reviews(person, review, structure_change_id_to_person_ids)
    if to_update:
        bulk.bulk_different_action_set(
            subject=person,
            data=to_update,
        )


def _get_person_id_to_heads_queryset(structure_change_id_to_person_ids: Dict[int, List[int]]) -> QuerySet:
    last_change_id_in_that_day = dict(
        staff_models.StaffStructureChange.objects
        .filter(id__in=structure_change_id_to_person_ids)
        .annotate(
            maxid=Subquery(
                staff_models.StaffStructureChange.objects
                .filter(date=OuterRef('date'))
                .order_by('-id')
                .values_list('id', flat=True)[:1]
            )
        )
        .values_list('id', 'maxid')
    )

    def get_q_for_structure_change(change_id: int, prson_ids: List[int]) -> Q:
        return Q(
            structure_change_id__lte=last_change_id_in_that_day[change_id],
            person_id__in=prson_ids,
        )

    query_chain_by_structure_changes = Q()
    for change_id, person_ids in structure_change_id_to_person_ids.items():
        query_chain_by_structure_changes |= get_q_for_structure_change(change_id, person_ids)

    personheads_ids_subquery = (
        staff_models.PersonHeads.objects
        .filter(query_chain_by_structure_changes)
        .order_by('person_id', '-structure_change_id')
        .distinct('person_id')
        .values_list('id')
    )
    return (
        staff_models.PersonHeads.objects
        .filter(id__in=personheads_ids_subquery)
        .values_list('person_id', 'heads')
    )


def _create_person_reviews(person, review, structure_change_id_to_person_ids):
    # type: (staff_models.Person, domain_objs.Review, DefaultDict[int, List[int]]) -> Dict[int, Dict[str, List[int]]]

    person_id_to_heads_qs = _get_person_id_to_heads_queryset(structure_change_id_to_person_ids)

    if review.actions[const.REVIEW_ACTIONS.ADD_ANY_PERSONS] == const.NO_ACCESS:
        cared_persons = staff_models.HR.objects.filter(hr_person=person).values_list('cared_person')
        person_id_to_heads_qs = person_id_to_heads_qs.filter(person_id__in=cared_persons)
    person_id_to_heads = dict(person_id_to_heads_qs)
    existing_person_reviews = models.PersonReview.objects.filter(
        review_id=review.id,
        person_id__in=person_id_to_heads
    ).values_list('id', 'person_id')

    existing_person_ids = {person_id for _, person_id in existing_person_reviews}
    person_to_person_reviews = {
        person_id: models.PersonReview(
            person_id=person_id,
            review_id=review.id,
            updated_at=timezone.now(),
        )
        for person_id in set(person_id_to_heads) - existing_person_ids
    }

    chains_to_update = {
        id: {
            const.FIELDS.REVIEWERS: _parse_reviewers_chain(
                person_id_to_heads[person_id]
            )
        }
        for id, person_id in existing_person_reviews
    }

    models.PersonReview.objects.bulk_create(list(person_to_person_reviews.values()))
    reviewers = _build_person_review_roles(person_to_person_reviews, review, person_id_to_heads)
    models.PersonReviewRole.objects.bulk_create(reviewers)
    roles.async_denormalize_person_review_roles(
        review_id=review.id,
        person_ids=list(person_to_person_reviews.keys()),
    )

    if review.status == const.REVIEW_STATUS.IN_PROGRESS:
        from review.core.tasks import freeze_gradient_data_task
        person_review_ids = [v.id for v in person_to_person_reviews.values()]
        freezing_datetime = review.product_schema_loaded or review.start_date
        freeze_gradient_data_task.delay(review.id, freezing_datetime, person_review_ids)

    return chains_to_update


def _build_person_review_roles(person_ids, review, person_id_to_heads):
    created_reviews = models.PersonReview.objects.filter(
        review_id=review.id,
        person_id__in=person_ids
    ).values('person_id', 'id')
    person_reviews = {r['person_id']: r['id'] for r in created_reviews}
    reviewers = [
        rev
        for person_id, person_review in person_reviews.items()
        for rev in create_reviewers_chain(
            reviewers=_parse_reviewers_chain(person_id_to_heads[person_id]),
            person_review=person_review,
            person_id=person_id,
        )
    ]
    return reviewers


def _parse_reviewers_chain(head):
    if isinstance(head, list):
        reviewers = head
    else:
        if head:
            reviewers = [int(it) for it in head.split(',')]
        else:
            # if volozh case
            reviewers = []
    remove_adjacent_doubles(reviewers)
    return reviewers


def create_reviewers_chain(reviewers, person_review, person_id):
    reviewers = [r for r in reviewers if r != person_id]
    if not reviewers:
        return []
    reviewers_obj = [
        models.PersonReviewRole(
            person_id=reviewer,
            position=i,
            person_review_id=person_review,
            type=const.ROLE.PERSON_REVIEW.REVIEWER
        )
        for i, reviewer in enumerate(reviewers[:-1])
    ]
    reviewers_obj.append(
        models.PersonReviewRole(
            person_id=reviewers[-1],
            position=len(reviewers) - 1,
            person_review_id=person_review,
            type=const.ROLE.PERSON_REVIEW.TOP_REVIEWER
        )
    )
    return reviewers_obj


def delete_review_subject(person, review, person_reviews):
    # type: (staff_models.Person, domain_objs.Review, models.PersonReview.objects) -> None
    if review.actions[const.REVIEW_ACTIONS.DELETE_ANY_PERSONS] == const.NO_ACCESS:
        person_reviews = person_reviews.filter(person__hr__hr_person=person)
    person_reviews.filter(review_id=review.id).delete()


def validate_transition(review, transition):
    if review.status == const.REVIEW_STATUS.DRAFT:
        person_reviews_empty_chains = _get_person_reviews_with_empty_chains(
            review=review,
        )
        if person_reviews_empty_chains:
            raise errors.Error(
                code=const.ERROR_CODES.REVIEW_WORKFLOW_EMPTY_CHAINS,
                msg_tpl='Review has person_reviews with empty chains',
                logins=[
                    pr['person__login']
                    for pr in person_reviews_empty_chains
                ]
            )
    return


def _get_person_reviews_with_empty_chains(review):
    pr_ids_with_chains = models.PersonReviewRole.objects.filter(
        person_review__review_id=review.id,
        type__in=[
            const.ROLE.PERSON_REVIEW.REVIEWER,
            const.ROLE.PERSON_REVIEW.TOP_REVIEWER,
        ],
    ).values_list('person_review_id', flat=True)
    person_reviews = models.PersonReview.objects.filter(
        review_id=review.id
    ).exclude(id__in=pr_ids_with_chains).values(
        'person__login',
    )
    return person_reviews


@transaction.atomic
def follow_workflow(review, new_workflow):
    workflow_to_status = {
        const.REVIEW_ACTIONS.STATUS_ARCHIVE: const.REVIEW_STATUS.ARCHIVE,
        const.REVIEW_ACTIONS.STATUS_PUBLISH: const.REVIEW_STATUS.IN_PROGRESS,
        const.REVIEW_ACTIONS.STATUS_IN_DRAFT: const.REVIEW_STATUS.DRAFT,
        const.REVIEW_ACTIONS.STATUS_UNARCHIVE: const.REVIEW_STATUS.IN_PROGRESS,
    }
    review = models.Review.objects.get(id=review.id)
    old_status = review.status
    new_status = workflow_to_status[new_workflow]
    if old_status == new_status:
        return
    if old_status == const.REVIEW_STATUS.ARCHIVE:
        if review.finish_date < arrow.now().date():
            new_status = const.REVIEW_STATUS.FINISHED
        else:
            new_status = const.REVIEW_STATUS.IN_PROGRESS

    review.status = new_status
    review.save()
    if const.REVIEW_STATUS.DRAFT in (new_status, old_status):
        roles.async_denormalize_person_review_roles(
            review_id=review.id,
        )
    if new_status == const.REVIEW_STATUS.ARCHIVE:
        _publish_review_results(review.id)
    if new_workflow == const.REVIEW_ACTIONS.STATUS_PUBLISH:
        from review.core import tasks
        freezing_datetime = timezone.now()
        tasks.freeze_gradient_data_task.delay(review.id, freezing_datetime)


def _publish_review_results(review_id):
    from review.core import tasks
    func = tasks.publish_review_results
    if yenv.type != 'development':
        func = func.delay
    func(review_id=review_id)


def publish_review_results(review_id):
    robot = staff_models.Person.objects.get(login='robot-review')
    return bulk.publish_person_reviews(robot, review_id)


def finish_reviews():
    return models.Review.objects.filter(
        finish_date__lt=arrow.now().date(),
        status=const.REVIEW_STATUS.IN_PROGRESS,
    ).update(status=const.REVIEW_STATUS.FINISHED)


def add_global_admins(review_id=None):
    admins = (
        models.GlobalRole.objects
        .filter(type=const.ROLE.GLOBAL.ADMIN)
        .values_list('person_id', flat=True)
    )
    to_create_for = []
    for pid in admins:
        # quantity of admins is low, so we can use cycle
        reviews_with_admin = (
            models.ReviewRole.objects
            .filter(type=const.ROLE.REVIEW.ADMIN, person_id=pid)
            .values_list('review_id')
        )
        review_ids = (
            models.Review.objects
            .exclude(id__in=reviews_with_admin)
            .values_list('id', flat=True)
        )
        if review_id:
            review_ids = review_ids.filter(id=review_id)
        to_create_for += [(pid, rid) for rid in review_ids]
    return models.ReviewRole.objects.bulk_create([
        models.ReviewRole(type=const.ROLE.REVIEW.ADMIN, person_id=pid, review_id=rid)
        for pid, rid in to_create_for
    ])


def delete_from_global_admins(global_admin_role):
    if global_admin_role.type == const.ROLE.GLOBAL.ADMIN:
        models.ReviewRole.objects.filter(person=global_admin_role.person).delete()


def load_product_scheme(review_id, date_from, date_to):
    from review.core import tasks
    func = tasks.load_product_scheme
    if yenv.type != 'development':
        func = func.delay
    func(review_id=review_id, date_from=date_from, date_to=date_to)
